package com.ysscale.redis.config;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.Data;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.core.env.MapPropertySource;
import org.springframework.data.redis.connection.RedisClusterConfiguration;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.connection.RedisNode;
import org.springframework.data.redis.connection.RedisSentinelConfiguration;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.Jackson2JsonRedisSerializer;
import org.springframework.data.redis.serializer.StringRedisSerializer;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.Map;

/**
 * @ClassName: YsscaleRedisConfig
 * @Desprition: TODO
 * @Auth: lie_w
 * @Date: 2018/11/23 8:59
 * @Version: V1.0.0
 */
@Data
@Component
@EnableConfigurationProperties
@ConfigurationProperties(prefix = "spring.redis.sentinel")
public class RedisConfig {

    private static final Logger logger = LoggerFactory.getLogger(RedisConfig.class);

    @Value("${spring.redis.pool.max-active}")
    private long maxActive;
    @Value("${spring.redis.pool.max-wait}")
    private long maxWaitMillis;
    @Value("${spring.redis.pool.max-idle}")
    private int maxIdle;
    @Value("${spring.redis.pool.min-idle}")
    private int minIdle;
    @Value("${spring.redis.timeout}")
    private int timeout;
    @Value("${spring.redis.commandTimeout}")
    private int commandTimeout;
    @Value("${spring.redis.max-redirects}")
    private int maxRedirects;


    @Value("${spring.redis.host}")
    private String host;
    @Value("${spring.redis.password}")
    private String password;

    @Value("${spring.redis.database}")
    private int database;

    private String master;
    private String nodes;


    /**
     * @description: redis 集群配置
     **/
    @Bean
    public RedisClusterConfiguration getClusterConfiguration() {
        if (host.split(",").length > 1) {
            //如果是host是集群模式的才进行以下操作
            Map<String, Object> source = new HashMap<String, Object>();
            source.put("spring.redis.cluster.nodes", host);
            source.put("spring.redis.cluster.password", password);
            source.put("spring.redis.cluster.timeout", timeout);
            source.put("spring.redis.cluster.max-redirects", maxRedirects);
            return new RedisClusterConfiguration(new MapPropertySource("RedisClusterConfiguration", source));
        }
        return null;
    }

    /**
     * @description: redis 哨兵配置
     **/
    @Bean
    public RedisSentinelConfiguration redisSentinelConfiguration() {
        if (StringUtils.isNotBlank(nodes)) {
            RedisSentinelConfiguration configuration = new RedisSentinelConfiguration();
            String[] hosts = nodes.split(",");
            for (String redisHost : hosts) {
                String[] item = redisHost.split(":");
                configuration.addSentinel(new RedisNode(item[0], Integer.parseInt(item[1])));
            }
            configuration.setMaster(master);
            return configuration;
        }
        return null;
    }

    @Bean
    public JedisConnectionFactory jedisConnectionFactory() {
        if (StringUtils.isNotBlank(master)) {
            logger.info("redis 哨兵配置。。。。。。");
            JedisConnectionFactory jcf = new JedisConnectionFactory(redisSentinelConfiguration());
            String[] hostIp = host.split(":");
            jcf.setHostName(hostIp[0]);
            jcf.setPort(Integer.parseInt(hostIp[1]));
            jcf.setTimeout(timeout);
            jcf.setPassword(password);
            jcf.setDatabase(database);
            return jcf;
        } else {
            if (!host.contains(",")) {
                logger.info("redis 单点配置。。。。。。");
                JedisConnectionFactory jcf = new JedisConnectionFactory();
                String[] hostIp = host.split(":");
                jcf.setHostName(hostIp[0]);
                jcf.setPort(Integer.valueOf(hostIp[1]));
                jcf.setTimeout(timeout);
                jcf.setPassword(password);
                return jcf;
            } else {
                logger.info("redis 集群配置。。。。。。");
                JedisConnectionFactory jcf = new JedisConnectionFactory(getClusterConfiguration());
                return jcf;
            }
        }
    }

    /**
     * 设置数据存入redis 的序列化方式
     * </br>redisTemplate序列化默认使用的jdkSerializeable,存储二进制字节码,导致key会出现乱码，所以自定义
     * 序列化类
     *
     * @paramredisConnectionFactory
     */
    @Bean
    public RedisTemplate<Object, Object> redisTemplate(RedisConnectionFactory redisConnectionFactory) {
        RedisTemplate<Object, Object> redisTemplate = new RedisTemplate<>();
        redisTemplate.setConnectionFactory(redisConnectionFactory);
        Jackson2JsonRedisSerializer jackson2JsonRedisSerializer = new Jackson2JsonRedisSerializer(Object.class);
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.setVisibility(PropertyAccessor.ALL, JsonAutoDetect.Visibility.ANY);
        objectMapper.enableDefaultTyping(ObjectMapper.DefaultTyping.NON_FINAL);
        jackson2JsonRedisSerializer.setObjectMapper(objectMapper);

        redisTemplate.setValueSerializer(jackson2JsonRedisSerializer);
        redisTemplate.setKeySerializer(new StringRedisSerializer());
        redisTemplate.afterPropertiesSet();
        return redisTemplate;
    }

}
