package com.ysscale.redis.service.impl;

import com.ysscale.redis.service.IRedisApplicationService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * @ClassName: IRedisApplicationServiceImpl
 * @Desprition: Redis操作
 * @Auth: lie_w
 * @Date: 2018/11/23 9:28
 * @Version: V1.0.0
 */
@Service
public class RedisApplicationServiceImpl implements IRedisApplicationService {

    @Autowired
    private RedisTemplate redisTemplate;

    private static final String SYMBOL_COLON = ":";

    @Override
    public void addKeyValue(String keyName, String key, Object value) {
        String saveKey = checkKey(keyName, key);
        redisTemplate.opsForValue().set(saveKey, value);
    }

    @Override
    public void addKeyValue(String keyName, String key, Object value, long timeOut) {
        String saveKey = checkKey(keyName, key);
        redisTemplate.opsForValue().set(saveKey, value, timeOut, TimeUnit.SECONDS);
    }

    @Override
    public void addKeyValue(String keyName, String key, Object value, long timeOut, TimeUnit timeUnit) {
        String saveKey = checkKey(keyName, key);
        redisTemplate.opsForValue().set(saveKey, value, timeOut, timeUnit);
    }

    @Override
    public Object getKeyValue(String keyName, String key) {
        String getKey = checkKey(keyName, key);
        return redisTemplate.opsForValue().get(getKey);
    }

    @Override
    public Object updateAndGetValue(String keyName, String key, Object newVal) {
        String updateKey = checkKey(keyName, key);
        return redisTemplate.opsForValue().getAndSet(updateKey, newVal);
    }

    @Override
    public void removeKeyValue(String keyName, String key) {
        String removeKey = checkKey(keyName, key);
        redisTemplate.delete(removeKey);
    }

    @Override
    public void removeKeyValue(String keyName, List<String> keys) {
        if (StringUtils.isBlank(keyName)) {
            redisTemplate.delete(keys);
        } else {
            List<String> deleteKeys = new ArrayList<>();
            for (String key : keys) {
                String deleteKey = checkKey(keyName, key);
                deleteKeys.add(deleteKey);
            }
            redisTemplate.delete(deleteKeys);
        }
    }

    @Override
    public Long stringIncrement(String keyName, String key, long val) {
        String saveKey = checkKey(keyName, key);
        return redisTemplate.opsForValue().increment(saveKey, val);
    }

    @Override
    public Double stringIncrement(String keyName, String key, double val) {
        String saveKey = checkKey(keyName, key);
        return redisTemplate.opsForValue().increment(saveKey, val);
    }

    @Override
    public String addHashKey(String keyName, String key1, String key2, Object value) {
        String saveKey = checkKey(keyName, key1);
        redisTemplate.opsForHash().put(saveKey, key2, value);
        return saveKey;
    }

    @Override
    public void addHashKey(String keyName, String key1, String key2, Object value, long timeOut) {
        String saveKey = addHashKey(keyName, key1, key2, value);
        redisTemplate.expire(saveKey, timeOut, TimeUnit.SECONDS);
    }

    @Override
    public void addHashKey(String keyName, String key1, String key2, Object value, long timeOut, TimeUnit timeUnit) {
        String saveKey = addHashKey(keyName, key1, key2, value);
        redisTemplate.expire(saveKey, timeOut, timeUnit);
    }

    @Override
    public String addHashKey(String keyName, String key1, Map value) {
        String saveKey = checkKey(keyName, key1);
        redisTemplate.opsForHash().putAll(saveKey, value);
        return saveKey;
    }

    @Override
    public void addHashKey(String keyName, String key1, Map value, long timeOut) {
        String saveKey = addHashKey(keyName, key1, value);
        redisTemplate.expire(saveKey, timeOut, TimeUnit.SECONDS);
    }

    @Override
    public void addHashKey(String keyName, String key1, Map value, long timeOut, TimeUnit timeUnit) {
        String saveKey = addHashKey(keyName, key1, value);
        redisTemplate.expire(saveKey, timeOut, timeUnit);
    }

    @Override
    public boolean hasKey(String keyName, String key1, String key2) {
        String saveKey = checkKey(keyName, key1);
        return redisTemplate.opsForHash().hasKey(saveKey, key2);
    }

    @Override
    public Long hashIncrement(String keyName, String key1, String key2, long val) {
        String saveKey = checkKey(keyName, key1);
        return redisTemplate.opsForHash().increment(saveKey, key2, val);
    }

    @Override
    public Double hashIncrement(String keyName, String key1, String key2, double val) {
        String saveKey = checkKey(keyName, key1);
        return redisTemplate.opsForHash().increment(saveKey, key2, val);
    }

    @Override
    public Object getHashVal(String keyName, String key1, String key2) {
        String saveKey = checkKey(keyName, key1);
        return redisTemplate.opsForHash().get(saveKey, key2);
    }

    @Override
    public Map getHashValMap(String keyName, String key1) {
        String saveKey = checkKey(keyName, key1);
        return redisTemplate.opsForHash().entries(saveKey);
    }

    @Override
    public List getHashValList(String keyName, String key1) {
        String saveKey = checkKey(keyName, key1);
        return redisTemplate.opsForHash().values(saveKey);
    }

    @Override
    public void removeHashKeyValue(String keyName, String key1) {
        String saveKey = checkKey(keyName, key1);
        Map map = getHashValMap(keyName, key1);
        Set set = map.keySet();
        List list = new ArrayList();
        set.forEach(o -> list.add(o));
        redisTemplate.opsForHash().delete(saveKey, list.toArray());
    }

    @Override
    public void removeHashKeyValue(String keyName, String key1, String key2) {
        String saveKey = checkKey(keyName, key1);
        redisTemplate.opsForHash().delete(saveKey, key2);
    }

    @Override
    public void removeHashKeyValue(String keyName, String key1, List<String> key2s) {
        String saveKey = checkKey(keyName, key1);
        redisTemplate.opsForHash().delete(saveKey, key2s.toArray());
    }

    @Override
    public void removeHashKeyValue(String keyName, List<String> key1s, String key2) {
        key1s.forEach(key1 -> redisTemplate.opsForHash().delete(checkKey(keyName, key1), key2));
    }

    @Override
    public void removeHashKeyValue(String keyName, List<String> key1s, List<String> key2s) {
        key1s.forEach(key1 -> redisTemplate.opsForHash().delete(checkKey(keyName, key1), key2s.toArray()));
    }

    @Override
    public synchronized String addHashKeyKeyVal(String keyName, String key1, String key2, String key, Object value) {
        Map<String, Object> hashVal = (Map<String, Object>) getHashVal(keyName, key1, key2);
        if (hashVal == null) hashVal = new HashMap<>();
        hashVal.put(key, value);
        return addHashKey(keyName, key1, key2, hashVal);
    }

    @Override
    public void addHashKeyKeyVal(String keyName, String key1, String key2, String key, Object value, long timeOut) {
        String saveKey = addHashKeyKeyVal(keyName, key1, key2, key, value);
        redisTemplate.expire(saveKey, timeOut, TimeUnit.SECONDS);
    }

    @Override
    public void addHashKeyKeyVal(String keyName, String key1, String key2, String key, Object value, long timeOut, TimeUnit timeUnit) {
        String saveKey = addHashKeyKeyVal(keyName, key1, key2, key, value);
        redisTemplate.expire(saveKey, timeOut, timeUnit);
    }

    private String checkKey(String keyName, String key) {
        if (!StringUtils.isBlank(keyName)) {
            return keyName + SYMBOL_COLON + key;
        } else {
            return key;
        }
    }
}
