package com.ysscale.redis.config;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.ysscale.framework.exception.SystemException;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.env.MapPropertySource;
import org.springframework.data.redis.connection.RedisClusterConfiguration;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.connection.RedisNode;
import org.springframework.data.redis.connection.RedisSentinelConfiguration;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.listener.KeyExpirationEventMessageListener;
import org.springframework.data.redis.listener.PatternTopic;
import org.springframework.data.redis.listener.RedisMessageListenerContainer;
import org.springframework.data.redis.serializer.Jackson2JsonRedisSerializer;
import org.springframework.data.redis.serializer.StringRedisSerializer;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.Map;

/**
 * @ClassName: YsscaleRedisConfig
 * @Desprition: TODO
 * @Auth: lie_w
 * @Date: 2018/11/23 8:59
 * @Version: V1.0.0
 */
@Slf4j
@Data
@Configuration
@Component
@EnableConfigurationProperties
@ConfigurationProperties(prefix = "com.jhscale.redis")
public class RedisConfig {

    // 连接池最大连接数（使用负值表示没有限制）
    private long maxActive = 50;
    // 连接池最大阻塞等待时间（使用负值表示没有限制）
    private long maxWait = -1;
    // 连接池中的最大空闲连接
    private int maxIdle = 10;
    // 连接池中的最小空闲连接
    private int minIdle = 2;
    // 连接超时时间（毫秒）
    private int timeout = 0;
    private int commandTimeout = 5000;
    private int maxRedirects = 6;

    // 单点redis 默认本地 ，间隔为集群模式
    private String host = "127.0.0.1:6379";
    // Redis服务器连接密码（默认为空）
    private String password;
    // Redis数据库索引（默认为0）,如果设置为1，那么存入的key-value都存放在select 1中
    private int database = 0;
    // Redis 失效监听，默认关闭
    private boolean listenInvalid = false;

    // 哨兵模式下配置使用
    private String master;
    private String nodes;

    /**
     * @description: redis 集群配置
     **/
    @Bean
    public RedisClusterConfiguration getClusterConfiguration() {
        if (host.split(",").length > 1) {
            //如果是host是集群模式的才进行以下操作
            Map<String, Object> source = new HashMap<String, Object>();
            source.put("spring.redis.cluster.nodes", host);
            source.put("spring.redis.cluster.password", password);
            source.put("spring.redis.cluster.timeout", timeout);
            source.put("spring.redis.cluster.max-redirects", maxRedirects);
            return new RedisClusterConfiguration(new MapPropertySource("RedisClusterConfiguration", source));
        }
        return null;
    }

    /**
     * @description: redis 哨兵配置
     **/
    @Bean
    public RedisSentinelConfiguration redisSentinelConfiguration() {
        if (StringUtils.isNotBlank(nodes)) {
            RedisSentinelConfiguration configuration = new RedisSentinelConfiguration();
            String[] hosts = nodes.split(",");
            for (String redisHost : hosts) {
                String[] item = redisHost.split(":");
                configuration.addSentinel(new RedisNode(item[0], Integer.parseInt(item[1])));
            }
            configuration.setMaster(master);
            return configuration;
        }
        return null;
    }

    @Bean
    public JedisConnectionFactory jedisConnectionFactory() throws SystemException {
        if (StringUtils.isBlank(host)) {
            throw new SystemException("Redis init fail...");
        }

        if (host.contains(",")) {
            log.info("redis 集群配置......");
            return new JedisConnectionFactory(getClusterConfiguration());
        }

        JedisConnectionFactory jcf = null;
        if (StringUtils.isNotBlank(master)) {
            log.info("redis 哨兵配置......");
            jcf = new JedisConnectionFactory(redisSentinelConfiguration());
        } else {
            log.info("redis 单点配置......");
            jcf = new JedisConnectionFactory();
        }

        String[] hostIp = host.split(":");
        jcf.setHostName(hostIp[0]);
        jcf.setPort(Integer.parseInt(hostIp[1]));
        jcf.setTimeout(timeout);
        jcf.setPassword(password);
        jcf.setDatabase(database);
        return jcf;
    }

    /**
     * 设置数据存入redis 的序列化方式
     * </br>redisTemplate序列化默认使用的jdkSerializeable,存储二进制字节码,导致key会出现乱码，所以自定义
     * 序列化类
     *
     * @paramredisConnectionFactory
     */
    @Bean
    public RedisTemplate<Object, Object> redisTemplate(RedisConnectionFactory redisConnectionFactory) {
        RedisTemplate<Object, Object> redisTemplate = new RedisTemplate<>();
        redisTemplate.setConnectionFactory(redisConnectionFactory);
        Jackson2JsonRedisSerializer jackson2JsonRedisSerializer = new Jackson2JsonRedisSerializer(Object.class);
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.setVisibility(PropertyAccessor.ALL, JsonAutoDetect.Visibility.ANY);
        objectMapper.enableDefaultTyping(ObjectMapper.DefaultTyping.NON_FINAL);
        jackson2JsonRedisSerializer.setObjectMapper(objectMapper);

        redisTemplate.setValueSerializer(jackson2JsonRedisSerializer);
        redisTemplate.setKeySerializer(new StringRedisSerializer());
        redisTemplate.afterPropertiesSet();
        return redisTemplate;
    }

    /**
     * 定义配置RedisListenerConfig
     *
     * @param connectionFactory
     * @return
     */
    @Bean
    RedisMessageListenerContainer container(RedisConnectionFactory connectionFactory) {
        RedisMessageListenerContainer container = new RedisMessageListenerContainer();
        container.setConnectionFactory(connectionFactory);
        container.addMessageListener(new KeyExpirationEventMessageListener(container), new PatternTopic("__keyevent@0__:expired"));
        return container;
    }

}
