package com.ysscale.redis.service.impl;

import com.ysscale.redis.service.RedisHandleService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * @author lie_w
 * @title: RedisHandleServiceImpl
 * @projectName Rely-On-DB
 * @description: TODO
 * @date 2019/12/1214:05
 */
@Service
public class RedisHandleServiceImpl implements RedisHandleService {

    @Autowired
    private RedisTemplate redisTemplate;

    /**
     * @param keyName
     * @param key
     * @param value
     * @description: Redis String类型操作
     */
    @Override
    public void addKeyValue(String keyName, String key, Object value) {
        redisTemplate.opsForValue().set(checkKey(keyName, key), value);
    }

    @Override
    public void addKeyValue(String keyName, String key, Object value, long timeOut) {
        addKeyValue(keyName, key, value, timeOut, TimeUnit.SECONDS);
    }

    @Override
    public void addKeyValue(String keyName, String key, Object value, long timeOut, TimeUnit timeUnit) {
        redisTemplate.opsForValue().set(checkKey(keyName, key), value, timeOut, timeUnit);
    }

    @Override
    public Object getKeyValue(String keyName, String key) {
        return redisTemplate.opsForValue().get(checkKey(keyName, key));
    }

    @Override
    public List getKeyValues(String keyName, List<String> keys) {
        if (keys != null && !keys.isEmpty()) {
            List<String> redisKeys = new ArrayList<>();
            keys.forEach(key -> {
                redisKeys.add(checkKey(keyName, key));
            });
            return redisTemplate.opsForValue().multiGet(redisKeys);
        }
        return null;
    }

    @Override
    public Object getAndSet(String keyName, String key, Object value) {
        return redisTemplate.opsForValue().getAndSet(checkKey(keyName, key), value);
    }

    @Override
    public void removeKeyValue(String keyName, String key) {
        redisTemplate.delete(checkKey(keyName, key));
    }

    /**
     * @param keyName 目录名称
     * @param key1    HASH Key
     * @param key2    VAL Key
     * @param value   VAL Val
     * @return
     */
    @Override
    public String addHashKey(String keyName, String key1, String key2, Object value) {
        String saveKey = checkKey(keyName, key1);
        redisTemplate.opsForHash().put(saveKey, key2, value);
        return saveKey;
    }

    @Override
    public void addHashKey(String keyName, String key1, String key2, Object value, long timeOut) {
        addHashKey(keyName, key1, key2, value, timeOut, TimeUnit.SECONDS);
    }

    /**
     * @param keyName
     * @param key1
     * @param key2
     * @param value
     * @param timeOut
     * @description: HASH保存 并且 没有监听逐条删除
     */
    @Override
    public void addHashKeyWithoutDel(String keyName, String key1, String key2, Object value, long timeOut) {
        addHashKeyWithoutDel(keyName, key1, key2, value, timeOut, TimeUnit.SECONDS);
    }

    @Override
    public void addHashKey(String keyName, String key1, String key2, Object value, long timeOut, TimeUnit timeUnit) {
        String saveKey = addHashKey(keyName, key1, key2, value);
        redisTemplate.expire(saveKey, timeOut, timeUnit);
        addKeyValue(HASH_INIVALID, saveKey + SYMBOL_COLON + key2, System.currentTimeMillis(), timeOut, timeUnit);
    }

    /**
     * @param keyName
     * @param key1
     * @param key2
     * @param value
     * @param timeOut
     * @param timeUnit
     * @description: HASH保存 并且 没有监听逐条删除
     */
    @Override
    public void addHashKeyWithoutDel(String keyName, String key1, String key2, Object value, long timeOut, TimeUnit timeUnit) {
        String saveKey = addHashKey(keyName, key1, key2, value);
        redisTemplate.expire(saveKey, timeOut, timeUnit);
    }

    @Override
    public void addHashKey(String keyName, Map<String, Object> key1Data, String key2, long timeOut, TimeUnit timeUnit) {
        if (key1Data != null && !key1Data.isEmpty()) {
            key1Data.forEach((key1, vale) -> {
                addHashKey(keyName, key1, key2, vale, timeOut, timeUnit);
            });
        }
    }

    @Override
    public void addHashKey(String keyName, String key1, Map<String, Object> key2Data, long timeOut, TimeUnit timeUnit) {
        String saveKey = checkKey(keyName, key1);
        redisTemplate.opsForHash().putAll(saveKey, key2Data);
        redisTemplate.expire(saveKey, timeOut, timeUnit);
    }

    @Override
    public void removeHashKey(String keyName, String key1, String key2) {
        String saveKey = checkKey(keyName, key1);
        redisTemplate.opsForHash().delete(saveKey, key2);
        removeKeyValue(HASH_INIVALID, saveKey + SYMBOL_COLON + key2);
    }

    @Override
    public Long removeHashKey(String keyName, String key1, String... key2s) {
        String saveKey = checkKey(keyName, key1);
        Long delete = redisTemplate.opsForHash().delete(saveKey, key2s);
        for (String key2 : key2s) {
            removeKeyValue(HASH_INIVALID, saveKey + SYMBOL_COLON + key2);
        }
        return delete;
    }

    @Override
    public Object getHashVal(String keyName, String key1, String key2) {
        return redisTemplate.opsForHash().get(checkKey(keyName, key1), key2);
    }

    @Override
    public List getHashValList(String keyName, String key1) {
        return redisTemplate.opsForHash().values(checkKey(keyName, key1));
    }

    @Override
    public Map getHashValMap(String keyName, String key1) {
        return redisTemplate.opsForHash().entries(key1);
    }

    @Override
    public List getHashValList(String keyName, String key1, List<String> key2s) {
        return redisTemplate.opsForHash().multiGet(checkKey(keyName, key1), key2s);
    }

    /**
     * @description: 目录层级
     **/
    private String checkKey(String keyName, String key) {
        if (!StringUtils.isBlank(keyName)) {
            return keyName + SYMBOL_COLON + key;
        } else {
            return key;
        }
    }
}
