package com.ysscale.socket.service.impl;

import com.ysscale.redis.service.IRedisApplicationService;
import com.ysscale.socket.config.WebSocketConfig;
import com.ysscale.socket.service.AsyncService;
import com.ysscale.socket.service.SocketService;
import com.ysscale.socket.utils.SocketManager;
import com.ysscale.socket.vo.ServerWebSocket;
import com.ysscale.socket.vo.WebSocketNotice;
import io.netty.channel.ChannelHandlerContext;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.client.serviceregistry.Registration;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import java.net.InetSocketAddress;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * @ClassName: SocketServiceImpl
 * @Desprition: TODO
 * @Auth: lie_w
 * @Date: 2019/3/20 22:41
 * @Version: V1.0.0
 */
@Service
public class SocketServiceImpl implements SocketService {

    private static final Logger LOGGER = LoggerFactory.getLogger(SocketServiceImpl.class);

    @Autowired
    private AsyncService asyncService;

    @Autowired
    private RestTemplate restTemplate;

    @Autowired
    private IRedisApplicationService redisApplicationService;

    @Autowired
    private Registration registration;

    @Autowired
    private WebSocketConfig webSocketConfig;

    private String ipPort = null;

    private String ipPortKey = null;

    private String webSocketKey = null;

    /**
     * @description: ipport key
     **/
    public String getIpPortKey() {
        if (StringUtils.isBlank(ipPortKey)) {
            ipPortKey = webSocketConfig.getSocketKey() + "-IPPORT";
        }
        return ipPortKey;
    }

    /**
     * @description: websocket key
     **/
    public String getWebSocketKey() {
        if (StringUtils.isBlank(webSocketKey)) {
            webSocketKey = webSocketConfig.getSocketKey() + "-WEBSOCKET";
        }
        return webSocketKey;
    }

    /**
     * @Description 获取服务IP port
     **/
    private String getIpPort() {
        if (ipPort == null) {
            String ipAddress = registration.getHost();
            int port = registration.getPort();
            ipPort = String.format("%s-%d", ipAddress, port);
        }
        return ipPort;
    }

    @Override
    public void notice(WebSocketNotice notice) {
        asyncService.notice(notice, getIpPortKey(), getIpPort());
    }

    @Override
    public boolean noticeWithCheck(WebSocketNotice notice) {
        if (check(notice.getKey())) {
            asyncService.notice(notice, getIpPortKey(), getIpPort());
            return true;
        } else {
            return false;
        }
    }

    @Override
    public boolean noticeWithCheck(String key, String msg) {
        if (check(key)) {
            asyncService.notice(new WebSocketNotice(key, msg), getIpPortKey(), getIpPort());
            return true;
        } else {
            return false;
        }
    }

    private String getWebKey(String uniqueKey) {
        return getIpPort() + "#" + uniqueKey.replace(":", "-");
    }

    @Override
    public void create(String uniqueKey) {
        String key = getWebKey(uniqueKey);
        if (webSocketConfig.getSocketKeySaveTime() == 0) {
            redisApplicationService.addKeyValue(getWebSocketKey(), key, "");
        } else {
            redisApplicationService.addKeyValue(getWebSocketKey(), key, "", webSocketConfig.getSocketKeySaveTime(), TimeUnit.HOURS);
        }

    }

    @Override
    public void remove(String uniqueKey) {
        String wb_key = getWebKey(uniqueKey);
        String ip_key = (String) redisApplicationService.getKeyValue(getWebSocketKey(), wb_key);
        if (StringUtils.isNotBlank(ip_key)) {
            String now_wb_key = (String) redisApplicationService.getKeyValue(getIpPortKey(), ip_key);
            if (wb_key.equals(now_wb_key)) {
                redisApplicationService.removeKeyValue(getIpPortKey(), ip_key);
            }
        }
        redisApplicationService.removeKeyValue(getWebSocketKey(), wb_key);
    }

    @Override
    public void putKey(String uniqueKey, String key) {
        String wb_key = getWebKey(uniqueKey);
        if (webSocketConfig.getSocketKeySaveTime() == 0) {
            redisApplicationService.addKeyValue(getWebSocketKey(), wb_key, key);
            redisApplicationService.addKeyValue(getIpPortKey(), key, wb_key);
        } else {
            redisApplicationService.addKeyValue(getWebSocketKey(), wb_key, key, webSocketConfig.getSocketKeySaveTime(), TimeUnit.HOURS);
            redisApplicationService.addKeyValue(getIpPortKey(), key, wb_key, webSocketConfig.getSocketKeySaveTime(), TimeUnit.HOURS);
        }
    }

    @Override
    public void putKey(String uniqueKey, String key, int num) {
        String wb_key = getWebKey(uniqueKey);
        redisApplicationService.addKeyValue(getWebSocketKey(), wb_key, key, num, TimeUnit.MINUTES);
        redisApplicationService.addKeyValue(getIpPortKey(), key, wb_key, num, TimeUnit.MINUTES);
    }

    @Override
    public String getModelName() {
        return getIpPort();
    }

    @Override
    public boolean check(String key) {
        if (StringUtils.isNotBlank(key)) {
            Object keyValue = redisApplicationService.getKeyValue(getIpPortKey(), key);
            if (Objects.nonNull(keyValue)) {
                return true;
            }
        }
        return false;
    }

    @Override
    public ServerWebSocket getWebServer() {
        ServerWebSocket serverWebSocket = new ServerWebSocket();
        serverWebSocket.setIp(webSocketConfig.getNettyIp());
        serverWebSocket.setPort(webSocketConfig.getNettyPort());
        serverWebSocket.setMaxCount(webSocketConfig.getMaxConnection());
        serverWebSocket.setNowCount(SocketManager.getInstance().getNowConnection());
        serverWebSocket.setHeatTime(webSocketConfig.getNoWriteHeartTime());
        return serverWebSocket;
    }

    @Override
    public String getTerminalIp(ChannelHandlerContext ctx) {
        InetSocketAddress insocket = (InetSocketAddress) ctx.channel().remoteAddress();
        return insocket.getAddress().getHostAddress();
    }
}
