package com.jhscale.network.client;

import com.jhscale.network.entity.food.BaseMarketRequest;
import com.jhscale.network.entity.food.BaseMarketResponse;
import com.jhscale.network.entity.telecom.res.TelecomTokenInfo;
import com.ysscale.framework.utils.JSONUtils;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Hex;
import org.apache.http.client.HttpClient;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.TrustSelfSignedStrategy;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.ssl.SSLContextBuilder;
import org.springframework.http.*;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.client.RestTemplate;

import javax.net.ssl.SSLContext;
import java.security.MessageDigest;
import java.util.HashMap;
import java.util.Objects;

/**
 * @param <T>  token
 * @param <LQ> 登录请求对象
 * @param <LR> 登录响应对象
 * @author lj
 * @title:
 * @projectName rely-on-utils
 * @description:
 * @date 2023/7/20 19:34
 */

@Slf4j
@NoArgsConstructor
public abstract class BaseMarketClient<T, LR extends BaseMarketResponse, LQ extends BaseMarketRequest<LR>> implements MarketClient {

    // 发送请求对象
    private RestTemplate restTemplate;
    private RestTemplate httpsRestTemplate;

    /**
     * @description: 获取日志前缀
     **/
    public abstract String getLogPrefix();

    /**
     * @description: 设置请求请求头token
     **/
    protected abstract <R extends BaseMarketResponse> void setToken(HttpHeaders httpHeaders, BaseMarketRequest<R> request) throws Exception;

    /**
     * @description: token是否过期
     **/
    protected abstract boolean tokenExpire(T tokenBean);


    /**
     * @description: 获取市场token对象
     **/
    public abstract T getTokenBean(String marketId) throws Exception;

    /**
     * @description: 获取市场token
     **/
    public abstract String getToken(String marketId) throws Exception;

    /**
     * @description: 保存token缓存
     **/
    protected abstract void saveToken(String mkId, TelecomTokenInfo data);

    /**
     * @description: 登录
     **/
    public abstract LR login(LQ request) throws Exception;

    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    /**
     * @param marketId
     * @description: token是否过期
     */
    public boolean tokenExpire(String marketId) throws Exception {
        T tokenBean = getTokenBean(marketId);
        return tokenExpire(tokenBean);
    }

    /**
     * @description: 获取完整url
     **/
    public String getUrl(BaseMarketRequest request) {
        return request.getUrlPrefix() + request.url();
    }

    /**
     * @description: 是否登录
     **/
    public boolean isLogin(String marketId) throws Exception {
        T tokenBean = this.getTokenBean(marketId);
        if (Objects.isNull(tokenBean)) return false;
        return this.tokenExpire(tokenBean);
    }


    /**
     * @description: 获取普通连接
     **/
    private RestTemplate getRestTemplate() {
        if (Objects.isNull(restTemplate)) {
            restTemplate = new RestTemplate();
        }
        return restTemplate;
    }

    /**
     * @description: 获取https连接
     **/
    private RestTemplate getHttpsRestTemplate() {
        if (Objects.isNull(httpsRestTemplate)) {
            httpsRestTemplate = new RestTemplate();
            // 设置SSL上下文
            SSLContext sslContext = null;
            try {
                sslContext = SSLContextBuilder.create()
                        .loadTrustMaterial(new TrustSelfSignedStrategy())
                        .build();
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }
            SSLConnectionSocketFactory socketFactory = new SSLConnectionSocketFactory(sslContext,
                    NoopHostnameVerifier.INSTANCE);
            HttpClient httpClient = HttpClients.custom()
                    .setSSLSocketFactory(socketFactory)
                    .build();
            HttpComponentsClientHttpRequestFactory requestFactory =
                    new HttpComponentsClientHttpRequestFactory(httpClient);
            httpsRestTemplate.setRequestFactory(requestFactory);
        }
        return httpsRestTemplate;
    }

    /**
     * @description: postHttpsForm
     **/
    public <R extends BaseMarketResponse> R postHttpsForm(BaseMarketRequest<R> request, HttpHeaders headers) {
        if (Objects.isNull(headers)) {
            headers = new HttpHeaders();
        }
        headers.setContentType(MediaType.APPLICATION_FORM_URLENCODED);
        Object realRequest = request.parseRequest();
        if (Objects.isNull(realRequest)) {
            realRequest = request;
        }
        MultiValueMap formData = new LinkedMultiValueMap();
        formData.setAll(JSONUtils.beanToBean(realRequest, HashMap.class));
        HttpEntity<MultiValueMap> requestBody = new HttpEntity<>(formData, headers);
        String url = this.getUrl(request);
        ResponseEntity<R> response = getHttpsRestTemplate().postForEntity(url, requestBody, request.resClass());
        log.debug("{}---请求  marketId:{}-------url:{}-------请求参数：{}-------响应：{}", this.getLogPrefix(), request.marketId(), url, JSONUtils.objectToJson(request), JSONUtils.objectToJson(response));

        return response.getBody();
    }

    /**
     * @description: postHttps
     **/
    public <R extends BaseMarketResponse> R postHttps(BaseMarketRequest<R> request, HttpHeaders headers) {
        if (Objects.isNull(headers)) {
            headers = new HttpHeaders();
        }
        headers.setContentType(MediaType.APPLICATION_JSON);
        Object realRequest = request.parseRequest();
        if (Objects.isNull(realRequest)) {
            realRequest = request;
        }
        HttpEntity requestBody = new HttpEntity<>(realRequest, headers);
        String url = this.getUrl(request);
        ResponseEntity<R> response = getHttpsRestTemplate().postForEntity(url, requestBody, request.resClass());
        log.debug("{}---请求  marketId:{}-------url:{}-------请求参数：{}-------响应：{}", this.getLogPrefix(), request.marketId(), url, JSONUtils.objectToJson(request), JSONUtils.objectToJson(response));
        return response.getBody();
    }

    /**
     * @description: post
     **/
    public <R extends BaseMarketResponse> R basePost(BaseMarketRequest<R> request, HttpHeaders headers) {
        if (Objects.isNull(headers)) {
            headers = new HttpHeaders();
        }
        headers.setContentType(MediaType.APPLICATION_JSON);
        Object realRequest = request.parseRequest();
        if (Objects.isNull(realRequest)) {
            realRequest = request;
        }
        HttpEntity requestBody = new HttpEntity<>(realRequest, headers);
        String url = this.getUrl(request);
        ResponseEntity<R> response = getRestTemplate().postForEntity(url, requestBody, request.resClass());

        log.debug("{}---请求  marketId:{}-------url:{}-------请求参数：{}-------响应：{}", this.getLogPrefix(), request.marketId(), url, JSONUtils.objectToJson(request), JSONUtils.objectToJson(response));
        return response.getBody();
    }

    /**
     * @description: 基础get请求
     **/
    public <R extends BaseMarketResponse> R baseGet(BaseMarketRequest<R> request, HttpHeaders httpHeaders) {
        httpHeaders.setContentType(MediaType.APPLICATION_JSON);
        HttpEntity requestBody = new HttpEntity<>(httpHeaders);
        String url = this.getUrl(request);
        ResponseEntity<R> response = getRestTemplate().exchange(url, HttpMethod.GET, requestBody, request.resClass());
        log.debug("{}---请求  marketId:{}-------url:{}-------请求参数：{}-------响应：{}", this.getLogPrefix(), request.marketId(), url, JSONUtils.objectToJson(request), JSONUtils.objectToJson(response));
        return response.getBody();
    }

    /**
     * @param request 请求对象
     * @return 响应对象
     * @description: 发起请求
     **/
    @Override
    public <R extends BaseMarketResponse> R execute(BaseMarketRequest<R> request) throws Exception {
        HttpHeaders httpHeaders = new HttpHeaders();
        //设置token
        this.setToken(httpHeaders, request);
//        httpHeaders.add("Authorization", getToken(request.marketId()));
        if (request.formDataPost()) {
            return this.postHttpsForm(request, httpHeaders);
        } else if (RequestMethod.GET.equals(request.method())) {
            return this.baseGet(request, httpHeaders);
        } else {
            return this.basePost(request, httpHeaders);
        }
    }

    /**
     * @description: sha256加密
     **/
    protected String SHA256(String str) {
        MessageDigest messageDigest;
        String encdeStr = "";
        try {
            messageDigest = MessageDigest.getInstance("SHA-256");
            byte[] hash = messageDigest.digest(str.getBytes("UTF-8"));
            encdeStr = Hex.encodeHexString(hash);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return encdeStr;
    }
}
