package com.jhscale.network.client;

import com.jhscale.common.exception.ProfessionalException;
import com.jhscale.network.config.NetWorkConfig;
import com.jhscale.network.entity.food.BaseMarketRequest;
import com.jhscale.network.entity.food.BaseMarketResponse;
import com.jhscale.network.entity.food.req.UploadFoodPriceLoginRequest;
import com.jhscale.network.entity.food.res.UploadFoodPriceLoginResponse;
import com.jhscale.network.entity.food.res.UploadFoodTokenInfo;
import com.ysscale.framework.utils.JSONUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.HttpClient;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.TrustSelfSignedStrategy;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.ssl.SSLContextBuilder;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.client.RestTemplate;

import javax.net.ssl.SSLContext;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * @author lj
 * @title:
 * @projectName rely-on-utils
 * @description:
 * @date 2023/7/20 19:34
 */
@Slf4j
public class UploadFoodPriceClient implements MarketClient {
    //请求成功状态
    public static final Integer SUCCESS_CODE = 200;

    // 发送请求对象
    private RestTemplate restTemplate;
    private RestTemplate httpsRestTemplate;
    //token信息
    private Map<String, UploadFoodTokenInfo> marketIdAndTokenMap = new HashMap<>();
    private Map<String, String> marketIdUsernameMap = new HashMap<>();
    private NetWorkConfig workConfig;

    public UploadFoodPriceClient(NetWorkConfig workConfig) {
        this.workConfig = workConfig;
    }


    /**
     * @param request 请求对象
     * @return 响应对象
     * @description: 发起请求
     **/
    @Override
    public <R extends BaseMarketResponse> R execute(BaseMarketRequest<R> request) throws ProfessionalException {
        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.add("Authorization", getToken(request.marketId()));
        if (request.formDataPost()) {
            return postHttpsForm(request, httpHeaders);
        } else if (RequestMethod.POST.equals(request.method())) {
            return postHttps(request, httpHeaders);
        } else {
            throw new ProfessionalException(request.method() + "暂未实现");
        }
    }

    /**
     * @description: 是否登录
     **/
    public boolean isLogin(String marketId) {
        String username = marketIdUsernameMap.get(marketId);
        if (StringUtils.isBlank(username)) return false;
        UploadFoodTokenInfo uploadFoodTokenInfo = marketIdAndTokenMap.get(username);
        return Objects.nonNull(uploadFoodTokenInfo) && !uploadFoodTokenInfo.expire();
    }

    /**
     * @description: 获取市场token
     **/
    private String getToken(String marketId) throws ProfessionalException {
        String username = marketIdUsernameMap.get(marketId);
        if (StringUtils.isBlank(username))
            throw new ProfessionalException("市场 " + marketId + " 未登录或登录过期");
        UploadFoodTokenInfo uploadFoodTokenInfo = marketIdAndTokenMap.get(username);
        if (Objects.isNull(uploadFoodTokenInfo) || uploadFoodTokenInfo.expire())
            throw new ProfessionalException("市场 " + marketId + " 未登录或登录过期");
        return uploadFoodTokenInfo.getOriginal();
    }

    /**
     * @description: 登录
     **/
    public UploadFoodPriceLoginResponse login(UploadFoodPriceLoginRequest request) throws ProfessionalException {
        if (StringUtils.isBlank(request.getUsername()) || StringUtils.isBlank(request.getPassword())) {
            request.setUsername(this.workConfig.getFoodUploadAccount());
            request.setPassword(this.workConfig.getFoodUploadPwd());
        }

        UploadFoodPriceLoginResponse resultBody = null;
        if (isLogin(request.getMarketId())) {
            //已登录
            resultBody = new UploadFoodPriceLoginResponse();
            resultBody.setCode(SUCCESS_CODE);
            return resultBody;
        }

        // 请求
        resultBody = postHttpsForm(request, null);
        // UploadFoodPriceLoginResponse resultBody = postHttps(request, null);

        if (Objects.isNull(resultBody) || !SUCCESS_CODE.equals(resultBody.getCode())) {
            log.error("上海主副食品运行调控系统零售数据对接---请求失败  marketId:{}-------url:{}-------请求参数：{}-------响应：{}", request.marketId(), request.getUrlPrefix() + request.url(), JSONUtils.objectToJson(request), JSONUtils.objectToJson(resultBody));
            throw new ProfessionalException("登录失败" + (Objects.isNull(resultBody) ? "" : resultBody.getMsg()));
        }
        UploadFoodTokenInfo uploadFoodTokenInfo = resultBody.parseToken();
        marketIdUsernameMap.put(request.getMarketId(), request.getUsername());
        marketIdAndTokenMap.put(request.getUsername(), uploadFoodTokenInfo);
        return resultBody;
    }

    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    /**
     * @description: 获取普通连接
     **/
    private RestTemplate getRestTemplate() {
        if (Objects.isNull(restTemplate)) {
            restTemplate = new RestTemplate();
        }
        return restTemplate;
    }

    /**
     * @description: 获取https连接
     **/
    private RestTemplate getHttpsRestTemplate() {
        if (Objects.isNull(httpsRestTemplate)) {
            httpsRestTemplate = new RestTemplate();
            // 设置SSL上下文
            SSLContext sslContext = null;
            try {
                sslContext = SSLContextBuilder.create()
                        .loadTrustMaterial(new TrustSelfSignedStrategy())
                        .build();
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }
            SSLConnectionSocketFactory socketFactory = new SSLConnectionSocketFactory(sslContext,
                    NoopHostnameVerifier.INSTANCE);
            HttpClient httpClient = HttpClients.custom()
                    .setSSLSocketFactory(socketFactory)
                    .build();
            HttpComponentsClientHttpRequestFactory requestFactory =
                    new HttpComponentsClientHttpRequestFactory(httpClient);
            httpsRestTemplate.setRequestFactory(requestFactory);
        }
        return httpsRestTemplate;
    }

    /**
     * @description: postHttpsForm
     **/
    private <R extends BaseMarketResponse> R postHttpsForm(BaseMarketRequest<R> request, HttpHeaders headers) {
        if (Objects.isNull(headers)) {
            headers = new HttpHeaders();
        }
        headers.setContentType(MediaType.APPLICATION_FORM_URLENCODED);
        Object realRequest = request.parseRequest();
        if (Objects.isNull(realRequest)) {
            realRequest = request;
        }
        MultiValueMap formData = new LinkedMultiValueMap();
        formData.setAll(JSONUtils.beanToBean(realRequest, HashMap.class));
        HttpEntity<MultiValueMap> requestBody = new HttpEntity<>(formData, headers);
        String url = request.getUrlPrefix() + request.url();
        ResponseEntity<R> response = getHttpsRestTemplate().postForEntity(url, requestBody, request.resClass());
        log.debug("上海主副食品运行调控系统零售数据对接---请求  marketId:{}-------url:{}-------请求参数：{}-------响应：{}", request.marketId(), url, JSONUtils.objectToJson(request), JSONUtils.objectToJson(response));

        return response.getBody();
    }

    /**
     * @description: postHttps
     **/
    private <R extends BaseMarketResponse> R postHttps(BaseMarketRequest<R> request, HttpHeaders headers) {
        if (Objects.isNull(headers)) {
            headers = new HttpHeaders();
        }
        headers.setContentType(MediaType.APPLICATION_JSON);
        Object realRequest = request.parseRequest();
        if (Objects.isNull(realRequest)) {
            realRequest = request;
        }
        HttpEntity requestBody = new HttpEntity<>(realRequest, headers);
        String url = request.getUrlPrefix() + request.url();
        ResponseEntity<R> response = getHttpsRestTemplate().postForEntity(url, requestBody, request.resClass());
        log.debug("上海主副食品运行调控系统零售数据对接---请求  marketId:{}-------url:{}-------请求参数：{}-------响应：{}", request.marketId(), url, JSONUtils.objectToJson(request), JSONUtils.objectToJson(response));
        return response.getBody();
    }

    /**
     * @description: post
     **/
    private <R extends BaseMarketResponse> R basePost(BaseMarketRequest<R> request, HttpHeaders headers) {
        if (Objects.isNull(headers)) {
            headers = new HttpHeaders();
        }
        headers.setContentType(MediaType.APPLICATION_JSON);
        Object realRequest = request.parseRequest();
        if (Objects.isNull(realRequest)) {
            realRequest = request;
        }
        HttpEntity requestBody = new HttpEntity<>(realRequest, headers);
        String url = request.getUrlPrefix() + request.url();
        ResponseEntity<R> response = getRestTemplate().postForEntity(url, requestBody, request.resClass());

        log.debug("上海主副食品运行调控系统零售数据对接---请求  marketId:{}-------url:{}-------请求参数：{}-------响应：{}", request.marketId(), url, JSONUtils.objectToJson(request), JSONUtils.objectToJson(response));
        return response.getBody();
    }

}
