package com.jhscale.network.client.impl;

import com.jhscale.common.exception.ProfessionalException;
import com.jhscale.network.client.BaseMarketClient;
import com.jhscale.network.entity.food.BaseMarketRequest;
import com.jhscale.network.entity.food.BaseMarketResponse;
import com.jhscale.network.entity.telecom.req.TelecomLoginRequest;
import com.jhscale.network.entity.telecom.res.TelecomTokenInfo;
import com.jhscale.network.entity.telecom.res.TelecomLoginResponse;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpHeaders;

import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;
import java.nio.charset.StandardCharsets;
import java.util.*;

/**
 * @author lj
 * @title:
 * @projectName rely-on-utils
 * @description:
 * @date 2023/9/14 18:46
 */
@Slf4j
@NoArgsConstructor
public class TelecomServerClient extends BaseMarketClient<TelecomTokenInfo, TelecomLoginResponse, TelecomLoginRequest> {
    //请求成功状态
    public static final Integer SUCCESS_CODE = 200;
    //token提前过期时间,避免刚检查token后token失效
    private Long EARLY_EXPIRE_MILLIS = 1000 * 60 * 5L;
    //token缓存
    private Map<String, TelecomTokenInfo> idAndTokenMap = new HashMap();


    /**
     * @description: 获取日志前缀
     **/
    @Override
    public String getLogPrefix() {
        return "电信监控服务---";
    }

    /**
     * @param httpHeaders
     * @param request
     * @description: 设置请求请求头token
     */
    @Override
    protected <R extends BaseMarketResponse> void setToken(HttpHeaders httpHeaders, BaseMarketRequest<R> request) throws Exception {
        httpHeaders.set("token", this.getToken(request.marketId()));
    }

    /**
     * @param tokenBean
     * @description: token是否过期
     */
    @Override
    protected boolean tokenExpire(TelecomTokenInfo tokenBean) {
        if (Objects.nonNull(tokenBean) && Objects.nonNull(tokenBean.getExpireTime())) {
            Long expireTime = Long.parseLong(tokenBean.getExpireTime());
            long millis = System.currentTimeMillis();
            return expireTime.compareTo(millis - EARLY_EXPIRE_MILLIS) < 0;
        }
        return true;
    }


    /**
     * @param marketId
     * @description: 获取市场token对象
     */
    @Override
    public TelecomTokenInfo getTokenBean(String marketId) throws Exception {
        return idAndTokenMap.get(marketId);
    }

    /**
     * @param marketId
     * @description: 获取市场token
     */
    @Override
    public String getToken(String marketId) throws Exception {
        TelecomTokenInfo tokenBean = getTokenBean(marketId);
        if (Objects.nonNull(tokenBean)) return tokenBean.getAccessToken();
        return null;
    }

    /**
     * @param mkId
     * @param data
     * @description: 保存token缓存
     */
    @Override
    protected void saveToken(String mkId, TelecomTokenInfo data) {
        idAndTokenMap.put(mkId, data);
    }

    /**
     * @param request
     * @description: 登录
     */
    @Override
    public TelecomLoginResponse login(TelecomLoginRequest request) throws Exception {
        String mkId = request.getMarketId();
        String appId = request.getAppId();
        String appKey = request.getAppKey();

        long securityStart = System.currentTimeMillis();
        String transactionId = UUID.randomUUID().toString();
        String sign = null;
        try {
            String key = appId.substring(0, 8) + appKey.substring(appKey.length() - 8);
            String plainText = appId + securityStart;
            byte[] raw = key.getBytes(StandardCharsets.UTF_8);
            SecretKeySpec keySpec = new SecretKeySpec(raw, "AES");
            Cipher cipher = Cipher.getInstance("AES/ECB/PKCS5Padding");
            cipher.init(1, keySpec);
            byte[] p = plainText.getBytes(StandardCharsets.UTF_8);
            byte[] result = cipher.doFinal(p);
            sign = Base64.getEncoder().encodeToString(result);
        } catch (Exception e) {
            log.error("获取电信监控服务登录token---获取签名失败 mkId: {}  appId:{}   appKey:{}   {}", mkId, appId, appKey, e.getMessage(), e);
            throw new ProfessionalException("获取签名失败  mkId: {}" + mkId);
        }
        long securityEnd = System.currentTimeMillis();
        log.debug("   获取电信监控服务登录token---mkId: {} 生成签名---耗时：{}   appId: {}   appKey: {}   transactionId: {}   sign: {}   ", mkId, securityEnd - securityStart, appId, appKey, transactionId, sign);


        long httpStart = System.currentTimeMillis();
        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.set("transactionId", transactionId);
        httpHeaders.set("appKey", this.SHA256(appKey));
        httpHeaders.set("sign", sign);
        TelecomLoginResponse response = this.baseGet(request, httpHeaders);
        long httpEnd = System.currentTimeMillis();
        log.debug("获取电信监控服务登录token---mkId: {} ---请求第三方服务---耗时：{}", mkId, httpEnd - httpStart);

        //保存token
        if (Objects.nonNull(response) && SUCCESS_CODE.equals(response.getCode()) && Objects.nonNull(response.getData())) {
            TelecomTokenInfo data = response.getData();
            this.saveToken(mkId, data);
        }
        return response;
    }

}
