package com.ysscale.scan.utils;

import com.ysscale.framework.exception.BusinessException;
import com.ysscale.framework.exception.CommonException;
import com.ysscale.framework.exception.SystemException;
import com.ysscale.framework.utils.AESUtils;
import com.ysscale.framework.utils.ConfigHelper;
import com.ysscale.framework.utils.MapXml;
import com.ysscale.scan.em.QrCodeLoseType;
import com.ysscale.scan.entity.ServerInfo;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;
import java.util.Objects;

/**
 * @author lie_w
 * @title: QrCodeBulid
 * @projectName assist-core
 * @description: 二维码生成工具
 * @date 2019/6/410:42
 */
public class QrCodeBulid {

    private static final Logger LOGGER = LoggerFactory.getLogger(QrCodeBulid.class);

    private static String REALMS = "https://scan.s.jhscale.com/?c=";

    private static String REALM = "http://scan.jhscale.com/?c=";

    public static final String MIN = "min";

    public static final String HOUR = "hour";

    public static final String DAY = "day";

    public static final String TIME = "time";

    static {
        try {
            ConfigHelper helper = new ConfigHelper("scan.properties");
            String https = helper.getStringValue("scan.https");
            String http = helper.getStringValue("scan.http");
            if (StringUtils.isNotBlank(https)) {
                REALMS = https;
            }

            if (StringUtils.isNotBlank(http)) {
                REALM = http;
            }
        } catch (Exception e) {
            LOGGER.error("读取扫码配置文件失败，使用默认域名配置信息");
        }
    }

    /**
     * @description: HTTPS 操作后失效
     **/
    public static String httpsHandleLose(String server, String content, String type, int num) throws SystemException {
        return getQrCodeUrl(server, content, type, num, REALMS, QrCodeLoseType.AFTER);
    }

    /**
     * @description: HTTP 操作后失效
     **/
    public static String httpHandleLose(String server, String content, String type, int num) throws SystemException {
        return getQrCodeUrl(server, content, type, num, REALMS, QrCodeLoseType.AFTER);
    }

    /**
     * @description: HTTPS 扫描一次失效
     **/
    public static String httpsScanLose(String server, String content, String type, int num) throws SystemException {
        return getQrCodeUrl(server, content, type, num, REALMS, QrCodeLoseType.BEFORE);
    }

    /**
     * @description: HTTP 扫描一次失效
     **/
    public static String httpScanLose(String server, String content, String type, int num) throws SystemException {
        return getQrCodeUrl(server, content, type, num, REALM, QrCodeLoseType.BEFORE);
    }

    /**
     * @description: HTTPS 操作永久不失效
     **/
    public static String https(String server, String content, String type, int num) throws SystemException {
        return getQrCodeUrl(server, content, type, num, REALMS, QrCodeLoseType.FOREVER);
    }

    /**
     * @description: HTTP 操作永久不失效
     **/
    public static String http(String server, String content, String type, int num) throws SystemException {
        return getQrCodeUrl(server, content, type, num, REALM, QrCodeLoseType.FOREVER);
    }

    /**
     * @description: 拼装url
     **/
    private static String getQrCodeUrl(String server, String content, String type, int num, String realm, QrCodeLoseType loseType) throws SystemException {
        switch (type) {
            case TIME:
                return realm + encrype(server, content, loseType, num);
            case MIN:
                return realm + encrypeWithValidMin(server, content, loseType, num);
            case HOUR:
                return realm + encrypeWithValidHour(server, content, loseType, num);
            case DAY:
                return realm + encrypeWithValidDay(server, content, loseType, num);
            default:
                return realm + encrype(server, content, loseType, -1);
        }
    }

    /**
     * @description: 内容有效分钟
     **/
    private static String encrypeWithValidMin(String server, String content, QrCodeLoseType loseType, int num) throws SystemException {
        return encrype(server, content, loseType, (System.currentTimeMillis() + (num * 60000)));
    }

    /**
     * @description: 内容有效小时
     **/
    private static String encrypeWithValidHour(String server, String content, QrCodeLoseType loseType, int num) throws SystemException {
        return encrype(server, content, loseType, (System.currentTimeMillis() + (num * 3600000)));
    }

    /**
     * @description: 内容有效天数
     **/
    private static String encrypeWithValidDay(String server, String content, QrCodeLoseType loseType, int num) throws SystemException {
        return encrype(server, content, loseType, (System.currentTimeMillis() + (num * 86400000)));
    }


    /**
     * @description: 二维码扫码限制和失效限制
     **/
    private static String encrype(String server, String content, QrCodeLoseType loseType, long validTime) throws SystemException {
        String lose = loseType.getType() + "-";
        if (loseType != QrCodeLoseType.FOREVER) {
            String key = String.valueOf(System.currentTimeMillis()).substring(3);
            lose += key;
        }
        String valid = "";
        if (validTime != 0) {
            valid = String.valueOf(validTime);
        }
        return AESUtils.aesEncrypt(server + "&" + content + "&" + lose + "&" + valid);
    }

    /**
     * @description: 解码
     **/
    public static ServerInfo decrypt(String content) throws CommonException {
        content = AESUtils.aesDecrypt(content);
        String[] contentArr = content.split("&");
        if (contentArr != null && contentArr.length == 4) {
            return new ServerInfo(contentArr);
        } else {
            throw new BusinessException("二维码无效");
        }
    }

    /**
     * @description: 创建Content
     **/
    public static String buildContent(String sgin, Object data) {
        StringBuffer sb = new StringBuffer();
        if (StringUtils.isNotBlank(sgin)) {
            sb.append(ScanOperateConstants.QRCODE_SGIN + "!" + sgin + "#");
        }
        if (Objects.nonNull(data)) {
            Map<String, Object> map = MapXml.toMap(data);
            if (map != null && !map.isEmpty()) {
                for (String key : map.keySet()) {
                    Object valObj = map.get(key);
                    if (Objects.nonNull(valObj))
                        sb.append(key + "!" + valObj.toString() + "#");
                }
            }
        }
        return sb.toString().substring(0, sb.length() - 1);
    }

    /**
     * @description: 创建Content
     **/
    public static String buildContent(Object data) {
        return buildContent("", data);
    }
}
