package com.codingapi.sds.delivery.service.impl;

import com.codingapi.sds.delivery.config.DeliveryConfig;
import com.codingapi.sds.delivery.model.SocketDeliveryParam;
import com.codingapi.sds.delivery.service.DeliveryServerService;
import com.codingapi.sds.delivery.service.OnlineService;
import com.ysscale.framework.utils.JSONUtils;
import com.ysscale.socket.entity.Server;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.client.ServiceInstance;
import org.springframework.cloud.client.discovery.DiscoveryClient;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * create by lorne on 2017/10/14
 */
@Slf4j
@Service
public class DeliveryServerServiceImpl implements DeliveryServerService {


    @Autowired
    private DiscoveryClient discoveryClient;

    @Autowired
    private RestTemplate restTemplate;

    // 服务类型 true 本地配置服务；false consul服务配置
    private Boolean serverType;

    @Autowired
    private DeliveryConfig deliveryConfig;

    @Autowired
    private OnlineService onlineService;


    @Override
    public Server getOkServer() {

        List<ServiceInstance> serviceInstances = discoveryClient.getInstances(SOCKET_SERVER_KEY);

        //选取socket服务
        if (serviceInstances != null && serviceInstances.size() > 0) {
            for (ServiceInstance instance : serviceInstances) {
                URI uri = instance.getUri();
                if (uri != null) {
                    Server server = restTemplate.getForObject(uri + "/socket/getServer", Server.class);
                    if (server.getNowCount() <= server.getMaxCount()) {
                        return server;
                    }
                }
            }
        }

        return null;
    }


    @Override
    public List<Server> serverList() {
        List<ServiceInstance> serviceInstances = discoveryClient.getInstances(SOCKET_SERVER_KEY);

        List<Server> servers = new ArrayList<>();

        //选取socket服务
        if (serviceInstances != null && serviceInstances.size() > 0) {
            for (ServiceInstance instance : serviceInstances) {
                URI uri = instance.getUri();
                if (uri != null) {
                    Server server = restTemplate.getForObject(uri + "/socket/getServer", Server.class);
                    servers.add(server);
                }
            }
        }

        return servers;
    }

    @Override
    public List<Server> serverList(String param) {
        List<ServiceInstance> serviceInstances = discoveryClient.getInstances(SOCKET_SERVER_KEY);
        List<Server> servers = new ArrayList<>();
        //选取socket服务
        if (serviceInstances != null && serviceInstances.size() > 0) {
            for (ServiceInstance instance : serviceInstances) {
                URI uri = instance.getUri();
                if (uri != null) {
                    Server server = restTemplate.getForObject(uri + "/socket/getServerParam?param={1}", Server.class, param);
                    if (server != null) {
                        servers.add(server);
                    }
                }
            }
        }
        return servers;
    }

    @Override
    public List<Server> serverListWithType(String param, String sType, String dType) {
        List<ServiceInstance> serviceInstances = discoveryClient.getInstances(SOCKET_SERVER_KEY);
        List<Server> servers = new ArrayList<>();
        //选取socket服务
        if (serviceInstances != null && serviceInstances.size() > 0) {
            for (ServiceInstance instance : serviceInstances) {
                URI uri = instance.getUri();
                try {
                    if (uri != null) {
                        Server server = restTemplate.getForObject(uri + "/socket/getServerParam?param={1}", Server.class, param);

                        log.info("serverListWithType 查询到的Socket信息：{}", JSONUtils.objectJsonParse(server));

                        if (server != null) {
                            if (fiflterSocket(dType, server)) continue;

                            if (StringUtils.isNotBlank(sType) && !sType.equalsIgnoreCase(server.getType())) {
                                continue;
                            } else {
                                servers.add(server);
                            }
                        }
                    }
                } catch (Exception e) {
                    log.error("delivery serverListWithType socket link err", e.getMessage(), e);
                }
            }
        }
        return servers;
    }

    @Override
    public List<Server> serverListWithOutType(String param, String sType, String dType) {
        List<ServiceInstance> serviceInstances = discoveryClient.getInstances(SOCKET_SERVER_KEY);
        List<Server> servers = new ArrayList<>();
        //选取socket服务
        if (serviceInstances != null && serviceInstances.size() > 0) {
            for (ServiceInstance instance : serviceInstances) {
                URI uri = instance.getUri();
                try {
                    if (uri != null) {
                        Server server = restTemplate.getForObject(uri + "/socket/getServerParam?param={1}", Server.class, param);

                        log.info("serverListWithOutType 查询到的Socket信息：{}", JSONUtils.objectJsonParse(server));

                        if (server != null) {
                            if (fiflterSocket(dType, server)) continue;

                            if (StringUtils.isNotBlank(sType) && sType.equalsIgnoreCase(server.getType())) {
                                continue;
                            } else {
                                servers.add(server);
                            }
                        }
                    }
                } catch (Exception e) {
                    log.error("delivery serverListWithOutType socket link err", e.getMessage(), e);
                }
            }
        }
        return servers;
    }

    private boolean fiflterSocket(String dType, Server server) {
        if (StringUtils.isNotBlank(server.getServerId())) {
            SocketDeliveryParam socketUpper = onlineService.getSocketUpper(server.getServerId());
            if (Objects.nonNull(socketUpper)) {
                if ("VC".equals(dType) && socketUpper.getVcCount() != 0 && server.getVNowCount() > socketUpper.getVcCount())
                    return true;

                if (socketUpper.getTotalCount() != 0 && server.getNowCount() > socketUpper.getTotalCount())
                    return true;
            }
        }
        return false;
    }

    @Override
    public boolean clearAllSocketCache(String params) {
        List<ServiceInstance> serviceInstances = discoveryClient.getInstances(SOCKET_SERVER_KEY);
        if (serviceInstances != null && !serviceInstances.isEmpty()) {
            for (ServiceInstance instance : serviceInstances) {
                URI uri = instance.getUri();
                if (uri != null) {
                    restTemplate.getForObject(uri + "/socket/clearDevicesCache?params={1}", Boolean.class, params);
                }
            }
        }
        return true;
    }

    @Override
    public List<Server> changeServerList(String param) {
        // 确定服务类型
        if (Objects.isNull(serverType)) {
            serverType = deliveryConfig.isServerType();
        }

        if (serverType) {
            List<Server> servers = new ArrayList<>();
            // 本地 delivery 服务
            String[] sockets = deliveryConfig.getSockets().split(",");
            if (sockets != null && sockets.length != 0) {
                for (String socket : sockets) {
                    Server server = restTemplate.getForObject(socket + "/socket/getServerParam?param={1}", Server.class, param);
                    if (server != null) {
                        servers.add(server);
                    }
                }
            }
            return servers;
        } else {
            // consul delivery 服务
            return serverList(param);
        }
    }

    @Override
    public boolean changeServerType(String key) {
        if ("consul".equalsIgnoreCase(key)) {
            serverType = false;
        } else if ("local".equalsIgnoreCase(key)) {
            serverType = true;
        } else {
            return false;
        }
        return true;
    }

    @Override
    public String getServerType() {
        if (serverType) {
            return "local";
        } else {
            return "consul";
        }
    }
}
