package com.jhscale.common.exception;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.jhscale.common.em.ExpParamType;
import com.jhscale.common.em.ServerExp;
import com.jhscale.common.internatonal.ExpInternationalEntity;
import com.jhscale.common.model.exp.ExpKVModel;
import com.jhscale.common.utils.JSONUtils;
import org.apache.commons.lang3.StringUtils;

import java.security.PrivilegedActionException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

/**
 * @author lie_w
 * @title: CommonException
 * @projectName common
 * @description: 通用异常
 * @date 2020-11-0515:29
 */
public abstract class GeneralException extends Exception {

    // 0-JAVA 1-安卓
    private int type;

    // 错误码
    private String code;

    // 异常类型
    private ExpParamType paramType;

    // 普通异常信息值
    private String msg;

    // String %s 替换异常信息值
    private List<String> msgs;

    // KV 异常替换信息值
    private List<ExpKVModel> kvModels;

    /**
     * @description: 异常等级
     **/
    public abstract String getExpLevel();

    /**
     * @description: 获取应用名
     **/
    public String getApplication() {
        return this.getClass().getSimpleName();
    }

    @Override
    public String getMessage() {
        return type == 1 ? new ExpInternationalEntity(this).toString() : JSONObject.toJSONString(new ExpInternationalEntity(this));
    }

    public GeneralException(GeneralException exception) {
        super(exception.getCause());
        this.type = exception.getType();
        this.code = exception.getCode();
        this.paramType = exception.getParamType();
        this.msg = exception.getMsg();
        this.msgs = exception.getMsgs();
        this.kvModels = exception.getKvModels();
    }

    /**
     * Constructs a new exception with the specified cause and a detail
     * message of <tt>(cause==null ? null : cause.toString())</tt> (which
     * typically contains the class and detail message of <tt>cause</tt>).
     * This constructor is useful for exceptions that are little more than
     * wrappers for other throwables (for example, {@link
     * PrivilegedActionException}).
     *
     * @param cause the cause (which is saved for later retrieval by the
     *              {@link #getCause()} method).  (A <tt>null</tt> value is
     *              permitted, and indicates that the cause is nonexistent or
     *              unknown.)
     * @since 1.4
     */
    public GeneralException(Throwable cause, String code, ExpParamType paramType, String... messages) {
        super(cause);
        this.code = code;
        this.paramType = paramType;
        if (ExpParamType.SIMPLE.equals(paramType)) {
            this.msg = messages[0];
        } else if (ExpParamType.STRING.equals(paramType)) {
            this.msgs = Arrays.asList(messages);
        }
    }

    public GeneralException(Throwable cause, String code, ExpKVModel... models) {
        super(cause);
        this.code = code;
        this.paramType = ExpParamType.MAP;
        this.kvModels = Arrays.asList(models);
    }

    public GeneralException(Throwable cause, String code, String message) {
        this(cause, code, ExpParamType.SIMPLE, message);
    }

    public GeneralException(Throwable cause, String code, String... messages) {
        this(cause, code, ExpParamType.STRING, messages);
    }

    public GeneralException(Throwable cause, String code, Object... objects) {
        super(cause);
        this.code = code;
        if (objects != null && objects.length > 0) {
            if (objects.length == 1) {
                Object obj = objects[0];
                if (Objects.nonNull(obj)) {
                    if (obj instanceof ExpKVModel) {
                        this.paramType = ExpParamType.MAP;
                        if (this.kvModels == null) this.kvModels = new ArrayList<>();
                        this.kvModels.add((ExpKVModel) obj);
                    } else {
                        this.paramType = ExpParamType.SIMPLE;
                        this.msg = obj.toString();
                    }
                }
            } else {
                for (Object obj : objects) {
                    if (Objects.nonNull(obj)) {
                        if (obj instanceof ExpKVModel) {
                            if (Objects.isNull(this.kvModels)) this.kvModels = new ArrayList<>();
                            this.kvModels.add((ExpKVModel) obj);
                        } else {
                            if (Objects.isNull(this.msgs)) this.msgs = new ArrayList<>();
                            this.msgs.add(obj.toString());
                        }
                    }
                }
                if (this.msgs != null && this.msgs.size() > 1) this.paramType = ExpParamType.STRING;
                if (this.kvModels != null && !this.kvModels.isEmpty()) this.paramType = ExpParamType.MAP;
                if (this.msgs != null && this.msgs.size() > 1 && this.kvModels != null && !this.kvModels.isEmpty())
                    this.paramType = ExpParamType.ALL;
            }
        }
        if (Objects.isNull(this.paramType)) {
            this.paramType = ExpParamType.SIMPLE;
            this.msg = "通用异常";
        }
    }

    public GeneralException(Throwable cause, ServerExp serverExp) {
        this(cause, serverExp.getCode(), serverExp.getMessage());
    }

    public GeneralException(Throwable cause, String message) {
        this(cause, ServerExp.系统异常.getCode(), message);
    }

    public GeneralException(Throwable cause, Integer code, String message) {
        this(cause, code.toString(), message);
    }

    /**
     * Constructs a new exception with {@code null} as its detail message.
     * The cause is not initialized, and may subsequently be initialized by a
     * call to {@link #initCause}.
     */
    public GeneralException(String code, ExpParamType paramType, String... messages) {
        this.code = code;
        this.paramType = paramType;
        if (ExpParamType.SIMPLE.equals(paramType)) {
            this.msg = messages[0];
        } else {
            this.msgs = Arrays.asList(messages);
        }
    }

    public GeneralException(String code, ExpKVModel... models) {
        this.code = code;
        this.paramType = ExpParamType.MAP;
        this.kvModels = Arrays.asList(models);
    }

    public GeneralException(String code, String message) {
        this(code, ExpParamType.SIMPLE, message);
    }

    public GeneralException(String code, String... messages) {
        this(code, ExpParamType.STRING, messages);
    }

    public GeneralException(String code, Object... objects) {
        this.code = code;
        if (objects != null && objects.length > 0) {
            if (objects.length == 1) {
                Object obj = objects[0];
                if (Objects.nonNull(obj)) {
                    if (obj instanceof ExpKVModel) {
                        this.paramType = ExpParamType.MAP;
                        if (this.kvModels == null) this.kvModels = new ArrayList<>();
                        this.kvModels.add((ExpKVModel) obj);
                    } else {
                        this.paramType = ExpParamType.SIMPLE;
                        this.msg = obj.toString();
                    }
                }
            } else {
                for (Object obj : objects) {
                    if (Objects.nonNull(obj)) {
                        if (obj instanceof ExpKVModel) {
                            if (Objects.isNull(this.kvModels)) this.kvModels = new ArrayList<>();
                            this.kvModels.add((ExpKVModel) obj);
                        } else {
                            if (Objects.isNull(this.msgs)) this.msgs = new ArrayList<>();
                            this.msgs.add(obj.toString());
                        }
                    }
                }
                if (this.msgs != null && this.msgs.size() > 1) this.paramType = ExpParamType.STRING;
                if (this.kvModels != null && !this.kvModels.isEmpty()) this.paramType = ExpParamType.MAP;
                if (this.msgs != null && this.msgs.size() > 1 && this.kvModels != null && !this.kvModels.isEmpty())
                    this.paramType = ExpParamType.ALL;
            }
        }
        if (Objects.isNull(this.paramType)) {
            this.paramType = ExpParamType.SIMPLE;
            this.msg = "通用异常";
        }
    }

    public GeneralException(ServerExp serverExp) {
        this(serverExp.getCode(), serverExp.getMessage());
    }

    public GeneralException(String message) {
        this(ServerExp.系统异常.getCode(), message);
    }

    public GeneralException(Integer code, String messages) {
        this(code.toString(), messages);
    }

    /**
     * @description: 字符串异常信息隐射
     **/
    protected void mapping(String message) {
        try {
            if (JSONUtils.isJSON(message)) {
                JSONObject jsonObject = JSONObject.parseObject(message);
                String jsonMessage = jsonObject.getString("message");
                if (StringUtils.isNotBlank(jsonMessage)) {
                    JSONObject jsonObjectObject = JSONObject.parseObject(jsonMessage);
                    this.setCode(jsonObjectObject.getString("jsl"));
                    this.setMsg(jsonObjectObject.getString("msg"));
                    this.setParamType(ExpParamType.valueOf(jsonObjectObject.getString("type")));
                    mapping(jsonObjectObject);
                } else {
                    this.setCode(jsonObject.getString("jsl"));
                    this.setMsg(jsonObject.getString("msg"));
                    this.setParamType(ExpParamType.valueOf(jsonObject.getString("type")));
                    mapping(jsonObject);
                }
            }
        } catch (Exception e) {
        }
    }

    /**
     * @description: 映射异常内容信息
     **/
    private void mapping(JSONObject jsonObject) {
        String expVal = jsonObject.getString("expVal");
        if (StringUtils.isNotBlank(expVal)) {
            JSONObject expValObject = JSONObject.parseObject(expVal);
            JSONArray msgs = expValObject.getJSONArray("msgs");
            if (msgs != null && msgs.size() > 0)
                this.setMsgs(JSONObject.parseArray(msgs.toJSONString(), String.class));

            JSONArray kvModels = expValObject.getJSONArray("kvModels");
            if (kvModels != null && kvModels.size() > 0)
                this.setKvModels(JSONObject.parseArray(kvModels.toJSONString(), ExpKVModel.class));
        }
    }

    public int getType() {
        return type;
    }

    public void setType(int type) {
        this.type = type;
    }

    public String getCode() {
        return code;
    }

    public void setCode(String code) {
        this.code = code;
    }

    public ExpParamType getParamType() {
        return paramType;
    }

    public void setParamType(ExpParamType paramType) {
        this.paramType = paramType;
    }

    public String getMsg() {
        return msg;
    }

    public void setMsg(String msg) {
        this.msg = msg;
    }

    public List<String> getMsgs() {
        return msgs;
    }

    public void setMsgs(List<String> msgs) {
        this.msgs = msgs;
    }

    public List<ExpKVModel> getKvModels() {
        return kvModels;
    }

    public void setKvModels(List<ExpKVModel> kvModels) {
        this.kvModels = kvModels;
    }
}
