package com.jhscale.common.model.device.polymerization.inner;

import com.jhscale.common.exception.GeneralInternational;
import com.jhscale.common.exception.ProfessionalException;
import com.jhscale.common.utils.AesHexUtils;
import com.jhscale.common.utils.ByteUtils;
import com.jhscale.common.utils.CrcUtils;
import com.jhscale.common.utils.RandomUtils;

import static com.jhscale.common.content.JHContents.END;
import static com.jhscale.common.content.JHContents.TOP;

/**
 * @author wang lie
 * @version 1.0
 * @projectName common
 * @title CloudPayTX
 * @description 云支付发送通知
 * @create 2024/1/22 13:47
 */
public abstract class AggregatedEntity implements POSCombination {

    private AggregatedMark mark;

    // 通讯标志
    private int nid = RandomUtils.integer_random(255);

    // 发送内容
    private StringBuilder content;

    public AggregatedEntity() {
    }

    public AggregatedEntity(AggregatedMark mark) {
        this.mark = mark;
    }

    public AggregatedEntity(RXTXData rxtxData) {
        this.nid = rxtxData.getNid();
        this.content = new StringBuilder(rxtxData.getContent());
    }

    /**
     * @param aesKey
     * @description: 组装
     */
    @Override
    public void assemble(String aesKey) throws ProfessionalException {
        StringBuilder inner_content = this.inner_assembly();
        if (this.mark.isEncrypt()) {
            try {
                int innerLength = inner_content.length() / 2;
                int fillingLength = 16 - innerLength % 16;
                for (int i = 1; i <= fillingLength; i++) {
                    inner_content.append(ByteUtils.int2Hex(i));
                }
                inner_content = new StringBuilder(AesHexUtils.encrypt(inner_content.toString(), aesKey));
            } catch (Exception e) {
                throw new ProfessionalException(GeneralInternational.密钥加密失败);
            }
        }

        String inner_content_length_hex = Integer.toHexString(inner_content.length() / 2);
        inner_content_length_hex = inner_content_length_hex.length() % 2 == 0 ? inner_content_length_hex : ("0" + inner_content_length_hex);
        this.mark.setLen(inner_content_length_hex.length());
        this.mark.assemble(aesKey);

        this.content = new StringBuilder()
                .append(this.mark.getHex())
                .append(ByteUtils.int2HexWithPush(this.nid, 4))
                .append(inner_content_length_hex)
                .append(inner_content);
        String crc = CrcUtils.crc(this.content.toString());
        this.content.insert(0, TOP)
                .append(crc)
                .append(END);
    }

    /**
     * @param aesKey
     * @return
     * @description: 解析
     */
    @Override
    public boolean analyse(String aesKey) throws ProfessionalException {
        String inner_content = this.content.substring(2, this.content.length() - 4);
        String p_crc = this.content.substring(this.content.length() - 4, this.content.length() - 2);
        String crc = CrcUtils.crc(inner_content);
        if (!crc.equalsIgnoreCase(p_crc)) return false;

        this.mark = new AggregatedMark(inner_content.substring(0, 2));
        this.mark.analyse(aesKey);
        this.nid = ByteUtils.hex2Ten(inner_content.substring(2, 6));

        int inner_content_length = Integer.parseInt(inner_content.substring(6, 6 + this.mark.getLen()), 16) * 2;
        inner_content = inner_content.substring(6 + this.mark.getLen(), 6 + this.mark.getLen() + inner_content_length);

        if (this.mark.isEncrypt()) {
            try {
                inner_content = AesHexUtils.decrypt(inner_content, aesKey);
                int filling = Integer.parseInt(inner_content.substring(inner_content.length() - 2), 16);
                inner_content = inner_content.substring(0, inner_content.length() - filling * 2);
            } catch (Exception e) {
                throw new ProfessionalException(GeneralInternational.密钥解密失败);
            }
        }

        this.inner_analyse(inner_content);

        return true;
    }

    /**
     * @description: NID通讯内容
     **/
    public RXTXData rxtxData() {
        return new RXTXData(this.nid, this.content.toString());
    }

    public AggregatedMark getMark() {
        return mark;
    }

    public void setMark(AggregatedMark mark) {
        this.mark = mark;
    }

    public int getNid() {
        return nid;
    }

    public void setNid(int nid) {
        this.nid = nid;
    }

    public StringBuilder getContent() {
        return content;
    }

    public void setContent(StringBuilder content) {
        this.content = content;
    }
}
