package com.jhscale.common.model.http;

import com.alibaba.fastjson.JSONObject;
import com.jhscale.common.exception.GeneralInternational;
import com.jhscale.common.exception.ProfessionalException;
import com.jhscale.common.exception.SignatureException;
import com.jhscale.common.utils.ByteUtils;
import com.jhscale.common.utils.DateUtils;
import com.jhscale.common.utils.MD5Utils;
import com.jhscale.common.utils.RandomUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.util.*;

/**
 * @author lie_w
 * @title: Signature
 * @projectName common
 * @description: TODO
 * @date 2020-12-080:54
 */
public interface Signature extends Serializable {

    /**
     * @description: 默认初始化填充内容
     **/
    default void defaultInit() {
        this.setVersion(StringUtils.isNotBlank(this.getVersion()) ? this.getVersion() : "1.0");
        this.setNonce_str(StringUtils.isNotBlank(this.getNonce_str()) ? this.getNonce_str() : RandomUtils.getEncryptRandomStrByLength(32));
        this.setTimestamp(Objects.nonNull(this.getTimestamp()) ? this.getTimestamp() : Long.parseLong(String.valueOf(System.currentTimeMillis() / 1000L)));
        this.setSign_type(StringUtils.isNotBlank(this.getSign_type()) ? this.getSign_type() : "MD5");
    }

    /**
     * @description: 构建签名
     * <p>
     * 第一步，设所有发送或者接收到的数据为集合M，将集合M内非空参数值的参数按照参数名ASCII码从小到大排序（字典序），使用URL键值对的格式（即key1=value1&key2=value2…）拼接成字符串stringA。
     * <p>
     * 特别注意以下重要规则：
     * <p>
     * ◆ 参数名ASCII码从小到大排序（字典序）；
     * ◆ 如果参数的值为空不参与签名；
     * ◆ 参数名区分大小写；
     * <p>
     * stringA="appid=wxd930ea5d5a258f4f&body=test&device_info=1000&mch_id=10000100&nonce_str=ibuaiVcKdpRxkhJA";
     * stringSignTemp=stringA+"&key=192006250b4c09247ec02edce69f6a2d" //注：key为商户平台设置的密钥key
     **/
    default String bulidSign(String key, String... ignore) {
        return MD5Utils.bulidMD5Signature(this.toObjectMap(ignore), key);
    }

    /**
     * @description: 构建签名
     * <p>
     * 第一步，设所有发送或者接收到的数据为集合M，将集合M内非空参数值的参数按照参数名ASCII码从小到大排序（字典序），使用URL键值对的格式（即key1=value1&key2=value2…）拼接成字符串stringA。
     * <p>
     * 特别注意以下重要规则：
     * <p>
     * ◆ 参数名ASCII码从小到大排序（字典序）；
     * ◆ 如果参数的值为空不参与签名；
     * ◆ 参数名区分大小写；
     * <p>
     * stringA="appid=wxd930ea5d5a258f4f&body=test&device_info=1000&mch_id=10000100&nonce_str=ibuaiVcKdpRxkhJA";
     * stringSignTemp=stringA+"&key=192006250b4c09247ec02edce69f6a2d" //注：key为商户平台设置的密钥key
     **/
    default String bulidSign_ad(String key, String... ignore) {
        return MD5Utils.bulidMD5Signature_ad(this.toObjectMap(ignore), key);
    }

    /**
     * @description: 构建签名源码
     **/
    default String bulidSignSource(String key, String... ignore) {
        return MD5Utils.bulidSourceSignature(this.toObjectMap(ignore), key);
    }

    /**
     * @description: 签名源码MD5加密
     **/
    default String buliMD5Sign(String result) {
        return MD5Utils.bulidMD5Signature(result);
    }

    /**
     * @description: 检查全部参数
     **/
    default boolean check(String key) throws SignatureException {
        return this.checkPackage(key) && this.checkSign(key);
    }

    /**
     * @description: 对象转Map
     **/
    default Map<String, String> toMap(String... params) {
        boolean cont = false;
        List<String> paramsList = new ArrayList<>();
        if (params != null && params.length > 0) {
            cont = true;
            paramsList = Arrays.asList(params);
        }

        Map<String, String> map = new HashMap<>();
        List<Field> fields = new ArrayList<>();
        getAllFieldList(this.getClass(), fields);
        for (Field field : fields) {
            if ("sign".equals(field.getName()) || cont && paramsList.contains(field.getName())) continue;
            Object obj;
            try {
                field.setAccessible(true);
                obj = field.get(this);
                if (obj != null) {
                    if (obj instanceof String || obj instanceof BigDecimal) {
                        map.put(field.getName().replace("_$_", ""), obj.toString());
                    } else if (obj instanceof List) {
                        List list = ((List) obj);
                        if (!list.isEmpty()) {
                            Object o = list.get(0);
                            if (o != null) {
                                if (o instanceof String || o instanceof BigDecimal) {
                                    map.put(field.getName().replace("_$_", ""), o.toString());
                                } else {
                                    map.put(field.getName().replace("_$_", ""), JSONObject.toJSONString(obj));
                                }
                            }
                        }
                    } else {
                        map.put(field.getName().replace("_$_", ""), JSONObject.toJSONString(obj));
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return map;
    }

    /**
     * @description: 对象转Map
     **/
    default Map toObjectMap(String... params) {
        boolean cont = false;
        List<String> paramsList = new ArrayList<>();
        if (params != null && params.length > 0) {
            cont = true;
            paramsList = Arrays.asList(params);
        }

        Map map = new HashMap<>();
        List<Field> fields = new ArrayList<>();
        getAllFieldList(this.getClass(), fields);
        for (Field field : fields) {
            if ("sign".equals(field.getName()) || cont && paramsList.contains(field.getName())) continue;
            Object obj;
            try {
                field.setAccessible(true);
                obj = field.get(this);
                if (obj != null) {
                    map.put(field.getName().replace("_$_", ""), obj);
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return map;
    }

    /**
     * @description: 获取类全部属性
     **/
    default void getAllFieldList(Class clazz, List<Field> fields) {
        Field[] declaredFields = clazz.getDeclaredFields();
        if (declaredFields != null && declaredFields.length > 0) fields.addAll(Arrays.asList(declaredFields));
        Class superclass = clazz.getSuperclass();
        if (Objects.nonNull(superclass)) getAllFieldList(superclass, fields);

    }

    /**
     * @description: 参数有效性校验
     **/
    default boolean checkPackage(String key) throws SignatureException {
        return this.checkPackage(key, 300);
    }

    /**
     * @description: 参数有效性校验
     **/
    default boolean checkPackage(String key, int validMin) throws SignatureException {
        if (StringUtils.isBlank(this.getVersion()))
            throw new SignatureException(GeneralInternational.接口版本无效);

        if (StringUtils.isBlank(this.getNonce_str()))
            throw new SignatureException(GeneralInternational.随机数无效);

        if (Objects.isNull(this.getTimestamp()))
            throw new SignatureException(GeneralInternational.时间戳不存在);

        if (StringUtils.isBlank(this.getSign()))
            throw new SignatureException(GeneralInternational.签名不存在);

        if (StringUtils.isBlank(key))
            throw new SignatureException(GeneralInternational.密钥不存在);

        String timezone = StringUtils.isNotBlank(this.getTime_zone()) ? this.getTime_zone() : "8";
        Integer nowTime = Integer.parseInt(String.valueOf(DateUtils.getLongTime(new Date(), timezone) / 1000L));
        if (nowTime - this.getTimestamp() > validMin)
            throw new SignatureException(GeneralInternational.时间戳超时);
        return true;
    }

    /**
     * @description: 校验签名
     **/
    default boolean checkSign(String key) throws SignatureException {
        if (!this.bulidSign(key).equals(this.getSign()))
            throw new SignatureException(GeneralInternational.签名无效);
        return true;
    }

    String getVersion();

    void setVersion(String version);

    String getNonce_str();

    void setNonce_str(String nonce_str);

    Long getTimestamp();

    void setTimestamp(Long timestamp);

    String getSign();

    void setSign(String sign);

    String getSign_type();

    void setSign_type(String sign_type);

    String getTime_zone();

    void setTime_zone(String time_zone);

    /**
     * @description: 字符内容加密
     **/
    default String encrypt(String content) {
        if (StringUtils.isBlank(this.getNonce_str()) || Objects.isNull(this.getTimestamp()))
            this.defaultInit();
        return this.encrypt(content, this.getNonce_str(), this.getTimestamp());
    }

    /**
     * @description: 加密
     **/
    default String encrypt(String content, String nonce_str, Long timestamp) {
        if (StringUtils.isBlank(content) || StringUtils.isBlank(nonce_str) || Objects.isNull(timestamp)) return null;
        nonce_str = nonce_str.toUpperCase();
        StringBuffer buffer = new StringBuffer(ByteUtils.int2Hex(content.length())).append(timestamp);
        for (int i = 0; i < nonce_str.length(); i++) {
            try {
                buffer.append(content.charAt(i));
                int numChar = Integer.parseInt(String.valueOf(nonce_str.charAt(i)), 16) + i;
                buffer.append(nonce_str.charAt(numChar));
            } catch (Exception e) {
            }
        }
        return buffer.toString();
    }

    /**
     * @description: 解密
     **/
    default String decrypt(String content) throws ProfessionalException {
        return this.decrypt(content, this.getNonce_str(), this.getTimestamp());
    }

    /**
     * @description: 解密
     **/
    default String decrypt(String content, String nonce_str, Long timestamp) throws ProfessionalException {
        if (StringUtils.isBlank(content) || StringUtils.isBlank(nonce_str) || Objects.isNull(timestamp)) return null;
        try {
            int sign_index = content.indexOf(timestamp.toString());
            Integer length = Integer.parseInt(content.substring(0, sign_index), 16);
            content = content.substring(sign_index + timestamp.toString().length());
            StringBuffer buffer = new StringBuffer();
            int index = 0;
            for (int i = 0; i < nonce_str.length(); i++) {
                if (buffer.length() == length)
                    break;
                buffer.append(content.charAt(index));
                String part = null;
                try {
                    int numChar = Integer.parseInt(String.valueOf(nonce_str.charAt(i)), 16) + i;
                    part = String.valueOf(nonce_str.charAt(numChar));
                } catch (Exception e) {
                }
                index = StringUtils.isNotBlank(part) ? index + 2 : index + 1;
            }
            return buffer.toString();
        } catch (Exception e) {
            e.printStackTrace();
            throw new ProfessionalException(GeneralInternational.解密失败);
        }
    }
}
