package com.jhscale.common.system;

import com.jhscale.common.exception.GeneralInternational;
import com.jhscale.common.exception.ProfessionalException;
import com.jhscale.common.system.entity.TerminalInfo;
import org.apache.commons.lang3.StringUtils;

import java.io.Serializable;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;

/**
 * @author lie_w
 * @title: AbstractServerInFO
 * @projectName common
 * @description: TODO
 * @date 2022/2/1712:38
 */
public abstract class AbstractTerminal implements Serializable {

    /**
     * @description: 获取系统终端信息
     **/
    public static TerminalInfo terminalInfo(String osName, OtherTerminal otherTerminal) {
        if (StringUtils.isBlank(osName)) {
            osName = System.getProperty("os.name");
        }
        osName = osName.toLowerCase();
        AbstractTerminal terminal = null;
        if (osName.startsWith("windows")) {
            terminal = new WindowsTerminal();
        } else if (osName.startsWith("linux")) {
            terminal = new LinuxTerminal();
        } else {
            terminal = otherTerminal.otherTerminal(osName);
        }
        return terminal.terminalInfo();
    }

    /**
     * @description: 获取设备信息
     **/
    public TerminalInfo terminalInfo() {
        TerminalInfo terminalInfo = new TerminalInfo();
        try {
            terminalInfo.setIpAddresses(this.getIpAddresses());
            terminalInfo.setMacAddresses(this.getMacAddresses());
            terminalInfo.setCpuSerial(this.getCPUSerial());
            terminalInfo.setMainBoardSerial(this.getMainBoardSerial());
        } catch (Exception e) {
            e.printStackTrace();
        }
        return terminalInfo;
    }

    /**
     * @description: 获取IP地址信息
     **/
    public abstract List<String> getIpAddresses() throws ProfessionalException;

    /**
     * @description: 获取Mac地址
     **/
    protected abstract List<String> getMacAddresses() throws ProfessionalException;

    /**
     * @description: 获取CPU序列号
     **/
    protected abstract String getCPUSerial() throws ProfessionalException;

    /**
     * @description: 获取主板序列号
     **/
    protected abstract String getMainBoardSerial() throws ProfessionalException;


    /**
     * @description: 获取当前服务器所有符合条件的InetAddress
     **/
    protected List<InetAddress> getLocalAllInetAddresses() throws ProfessionalException {
        try {
            List<InetAddress> result = new ArrayList<>(4);
            // 遍历所有的网络接口
            for (Enumeration networkInterfaces = NetworkInterface.getNetworkInterfaces(); networkInterfaces.hasMoreElements(); ) {
                NetworkInterface iface = (NetworkInterface) networkInterfaces.nextElement();
                // 在所有的接口下再遍历IP
                for (Enumeration inetAddresses = iface.getInetAddresses(); inetAddresses.hasMoreElements(); ) {
                    InetAddress inetAddr = (InetAddress) inetAddresses.nextElement();

                    //排除LoopbackAddress、SiteLocalAddress、LinkLocalAddress、MulticastAddress类型的IP地址
                    if (!inetAddr.isLoopbackAddress() /*&& !inetAddr.isSiteLocalAddress()*/
                            && !inetAddr.isLinkLocalAddress() && !inetAddr.isMulticastAddress()) {
                        result.add(inetAddr);
                    }
                }
            }

            return result;
        } catch (SocketException e) {
            e.printStackTrace();
            throw new ProfessionalException(GeneralInternational.网络接口获取失败);
        }
    }

    /**
     * @description: 获取某个网络接口的Mac地址
     **/
    protected String getMacByInetAddress(InetAddress inetAddress) {
        try {
            byte[] mac = NetworkInterface.getByInetAddress(inetAddress).getHardwareAddress();
            if (mac != null) {
                StringBuffer buffer = new StringBuffer();
                for (int i = 0; i < mac.length; i++) {
                    if (i != 0) {
                        buffer.append("-");
                    }
                    //将十六进制byte转化为字符串
                    String temp = Integer.toHexString(mac[i] & 0xff);
                    buffer.append(temp.length() == 1 ? ("0" + temp) : temp);
                }
                return buffer.toString().toUpperCase();
            }
        } catch (SocketException e) {
            e.printStackTrace();
        }
        return null;
    }
}
