package com.jhscale.common.utils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.TimeZone;

/**
 * @program: ysscale_core
 * @description: 时间、时区、时间格式转换的工具类
 * @author: zhao
 * @create: 2018-11-08 09:30
 **/
public class DateConvertUtil {
    /**
     * 获取北京时区
     *
     * @return
     */
    public static TimeZone getBeijingTimeZone() {
        return TimeZone.getTimeZone("GMT+8:00");
    }

    /**
     * 获取当前手机对应的系统时区
     */
    public static TimeZone getPhoneTimeZone() {
        return TimeZone.getDefault();
    }

    /**
     * 以“GMT+8：00”形式返回当前系统对应的时区
     *
     * @return
     */
    public static String getCurrentTimeZoneStr() {
        TimeZone tz = TimeZone.getDefault();
        // tz.getRawOffset()获取时间偏移 单位ms
        return createGmtOffsetString(true, true, tz.getRawOffset());
    }

    /**
     * 通过当前系统时区的时间偏移量得到以“GMT+8：00”形式的时区
     *
     * @return
     */
    public static String createGmtOffsetString(boolean includeGmt,
                                               boolean includeMinuteSeparator, int offsetMillis) {
        int offsetMinutes = offsetMillis / 60000;
        char sign = '+';
        if (offsetMinutes < 0) {
            sign = '-';
            offsetMinutes = -offsetMinutes;
        }
        StringBuilder builder = new StringBuilder(9);
        if (includeGmt) {
            builder.append("GMT");
        }
        builder.append(sign);
        appendNumber(builder, 2, offsetMinutes / 60);
        if (includeMinuteSeparator) {
            builder.append(':');
        }
        appendNumber(builder, 2, offsetMinutes % 60);
        return builder.toString();
    }

    private static void appendNumber(StringBuilder builder, int count, int value) {
        String string = Integer.toString(value);
        for (int i = 0; i < count - string.length(); i++) {
            builder.append('0');
        }
        builder.append(string);
    }


    /**
     * 格式化日期的标准字符串
     */
    public final static String Detail_Format = "yyyy-MM-dd HH:mm:ss";
    //  private final static String Simple_Format="yyyy-MM-dd";


    /**
     * 将日期字符串转换为Date对象
     *
     * @param date   日期字符串，必须为"yyyy-MM-dd HH:mm:ss"
     * @param format 格式化字符串
     * @return 日期字符串的Date对象表达形式
     */
    public static Date parseToDate(String date, String format) {
        Date dt = null;
        SimpleDateFormat dateFormat = new SimpleDateFormat(format);
        try {
            dt = dateFormat.parse(date);
        } catch (ParseException e) {
            e.printStackTrace();
        }

        return dt;
    }


    /**
     * 将date----->String
     * 将Date对象转换为指定格式的字符串
     *
     * @param date     Date对象
     * @param //String format 格式化字符串
     * @return Date对象的字符串表达形式
     */
    public static String formatDateToStr(Date date, String format) {
        SimpleDateFormat dateFormat = new SimpleDateFormat(format);
        return dateFormat.format(date);
    }


    /**
     * 获取更改时区后的时间
     *
     * @param date    时间
     * @param oldZone 旧时区
     * @param newZone 新时区
     * @return 时间
     */
    public static Date changeTimeZone(Date date, TimeZone oldZone, TimeZone newZone) {
        Date dateTmp = null;
        if (date != null) {
            int timeOffset = oldZone.getRawOffset() - newZone.getRawOffset();
            dateTmp = new Date(date.getTime() - timeOffset);
        }
        return dateTmp;
    }


    /**
     * 将北京时区的时间转化为当前系统对应时区的时间
     *
     * @param beijingTime
     * @param format
     * @return
     */
    public static String beijingTime2PhoneTime(String beijingTime, String format) {
        Date beijingDate = parseToDate(beijingTime, format);
        Date phoneDate = changeTimeZone(beijingDate, getBeijingTimeZone(), getPhoneTimeZone());
        String phoneTime = formatDateToStr(phoneDate, format);
        return phoneTime;
    }

    /**
     * 将本地系统对应时区的时间转换为上北京时区对应的时间
     *
     * @param phoneTime
     * @return
     */
    public static String phoneTime2BeijingTime(String phoneTime) {
        Date phoneDate = parseToDate(phoneTime, Detail_Format);
        Date beijingDate = changeTimeZone(phoneDate, getPhoneTimeZone(), getBeijingTimeZone());
        String beijingTime = formatDateToStr(beijingDate, Detail_Format);
        return beijingTime;
    }

//    public static void main(String[] args) {
//        System.out.println("当前系统时区:" + getCurrentTimeZoneStr());
//        System.out.println("将本地系统对应时区的时间转换为上北京时区对应的时间:" + phoneTime2BeijingTime(formatDateToStr(new Date(), Detail_Format)));
//        System.out.println("将北京时区的时间转化为当前系统对应时区的时间:" + beijingTime2PhoneTime("2018-10-08 10:11:00", Detail_Format));
//        // 当前手机时区
//        TimeZone phoneTimeZone = getPhoneTimeZone();
//        // 北京时区
//        TimeZone beijingTimeZone = getBeijingTimeZone();
//        // 根据时区字符串获取时区
//        TimeZone timeZone = TimeZone.getTimeZone("GMT+8:00");
//        // 根据当前时区获取时区字符串
//        String timeZoneStr = getCurrentTimeZoneStr();
//        System.out.println("获取更改时区后的时间:" + changeTimeZone(new Date(), phoneTimeZone, beijingTimeZone));
//    }

}
