package com.jhscale.common.utils;

import com.jhscale.common.em.ServerExp;
import com.jhscale.common.exception.GeneralException;
import com.jhscale.common.exception.ProfessionalException;
import com.jhscale.common.exception.TechnologyException;
import org.apache.commons.lang3.StringUtils;

import java.text.ParseException;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.time.*;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAdjusters;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

/**
 * @author lie_w
 * @title: CommonDate
 * @projectName common
 * @description: 时间工具类
 * @date 2020-10-2216:27
 */
public class DateUtils {

    /**
     * 2000年1月1日0时0分0秒的毫秒数
     */
    public static final Long TIME_2000_1_1_0_0_0 = 946656000000L;

    public static final String YYYY_MM_DD_HH_MM_SS_Space = "yyyy MM dd HH mm ss";
    public static final String YY_MM_DD_HH_MM_SS_Space = "yy MM dd HH mm ss";
    public static final String YYYY_MM_DD_HH_MM_SS = "yyyy-MM-dd HH:mm:ss";
    public static final String YYYY_MM_DD_HH_MM_SS_S = "yyyy-MM-dd HH:mm:ss.S";
    public static final String YYYY_MM_DD_HH_MM_SS_C = "yyyy年MM月dd日 HH时mm分ss秒";
    public static final String YYYYMMDD = "yyyyMMdd";
    public static final String YYMMDD = "yyMMdd";
    public static final String YYYYMMDDHH = "yyyyMMddHH";
    public static final String YYYYMMDDHHMMSS = "yyyyMMddHHmmss";
    public static final String YYYYMMDDHHMMSSS = "yyyyMMddHHmmssS";
    public static final String HHMMSS = "HHmmss";
    public static final String HHMMSSS = "HHmmssS";
    public static final String HHMMSSSSS = "HHmmssSSS";
    public static final String YYYY = "yyyy";
    public static final String C_YYYY = "yyyy年";
    public static final String YYYYMM = "yyyyMM";
    public final static String YEAR_MONTH_FORMAT = "yyyy-MM";
    public static final String C_YYYYMM = "yyyy年MM月";
    public static final String YYYY_MM_DD = "yyyy-MM-dd";
    public static final String C_YYYYMMDD = "yyyy年MM月dd日";
    public static final String LOCATE_DATE_FORMAT = "yyyyMMddHHmmss";
    public static final String YYYYMMDD_HH_MM_SS = "yyyyMMdd HH:mm:ss";
    public static final String YYYYMMDD_T_HH_MM_SS = "yyyyMMddTHH:mm:ss";
    private static final int DAYS_OF_A_WEEK = 7;
    private static final SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");

    private DateUtils() {
    }

    /**
     * @Description: 获取秤的当前时间，秤默认开始时间2000-01-01 00:00:00
     */
    public static String getTime() throws TechnologyException {
        Date date = getDefaultTime();
        int seconds = (int) ((new Date().getTime() - date.getTime()) / 1000);
        String time = Integer.toHexString(seconds);
        for (int i = 0; i < 8 - time.length(); i++) {
            time = "0" + time;
        }
        return ByteUtils.toHexString(ByteUtils.reverse(ByteUtils.fromHexAscii(time)));
    }

    /**
     * @Description: 获取收银秤默认系统时间
     */
    public static Date getDefaultTime() {
        try {
            return getDateTime("2000-01-01 00:00:00", YYYY_MM_DD_HH_MM_SS);
        } catch (TechnologyException e) {
            return new Date(946656000000L);
        }
    }

    /**
     * @Description: 获取时间的时间戳
     */
    public synchronized static Date timeStrFormat(String strDate) {
        try {
            SimpleDateFormat sdf = new SimpleDateFormat(YYYY_MM_DD_HH_MM_SS);
            return sdf.parse(strDate);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * @Description: 获取时间的时间戳
     */
    public synchronized static String timeFormat(Date date) {
        try {
            return getString(date);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * @Description: YYYY_MM_DD_HH_MM_SS_Space 字符串
     */
    public synchronized static String getFormatDate_YYYY_MM_DD_HH_MM_SS_Space(Date date) {
        return getFormatDate(date, YYYY_MM_DD_HH_MM_SS_Space);
    }

    /**
     * @Description: YYYY_MM_DD_HH_MM_SS_Space 字符串
     */
    public synchronized static String getFormatDate_YY_MM_DD_HH_MM_SS_Space(Date date) {
        return getFormatDate(date, YY_MM_DD_HH_MM_SS_Space);
    }

    /**
     * @Description: YYYY_MM_DD_HH_MM_SS_Space 解析时间
     */
    public synchronized static Date getDateTime_YYYY_MM_DD_HH_MM_SS_Space(String dateStr) {
        try {
            return getDateTime(dateStr, YYYY_MM_DD_HH_MM_SS_Space);
        } catch (TechnologyException e) {
            return null;
        }
    }

    /**
     * @Description: YYYY_MM_DD_HH_MM_SS_Space 解析时间
     */
    public synchronized static Date getDateTime_YY_MM_DD_HH_MM_SS_Space(String dateStr) {
        try {
            return getDateTime(dateStr, YY_MM_DD_HH_MM_SS_Space);
        } catch (TechnologyException e) {
            return null;
        }
    }

    /**
     * @Description: 获取时间的时间戳
     */
    public synchronized static Date timeStrFormat1(String strDate) {
        try {
            SimpleDateFormat sdf = new SimpleDateFormat(YYYY_MM_DD_HH_MM_SS_S);
            return sdf.parse(strDate);
        } catch (ParseException e) {
            return null;
        }
    }

    /**
     * @Description: 时间转字符串
     */
    public synchronized static String getCString(Date date) {
        SimpleDateFormat sdf = new SimpleDateFormat(YYYY_MM_DD_HH_MM_SS_C);
        return sdf.format(date);
    }

    public synchronized static String getString(Date date) {
        SimpleDateFormat sdf = new SimpleDateFormat(YYYY_MM_DD_HH_MM_SS);
        return sdf.format(date);
    }

    /**
     * @Description: 字符串时间解析
     */
    public synchronized static Date getFormtDateTime(String dateStr, String format) throws TechnologyException {
        return getDateTime(dateStr, format);

    }

    /**
     * @Description 时间转换成整点
     */
    public static Date formatDateTime(Date date, String format) throws TechnologyException {
        return formatDate(date, format);
    }

    /**
     * @description: 指定格式 YYYY_MM_DD_HH_MM_SS
     **/
    public synchronized static String getDate_YYYY_MM_DD_HH_MM_SS(Date date) {
        return getFormatDate(date, YYYY_MM_DD_HH_MM_SS);
    }

    /**
     * @description: 指定格式 YYYYMMDD
     **/
    public synchronized static String getDate_YYYYMMDD(Date date) {
        return getFormatDate(date, YYYYMMDD);
    }

    /**
     * @description: 指定格式 YYYYMMDD
     **/
    public synchronized static String getDate_YYMMDD(Date date) {
        return getFormatDate(date, YYMMDD);
    }

    public synchronized static Date getDate__YYMMDD(String dateStr) {
        return getFormatDate(dateStr, YYMMDD);
    }

    /**
     * @description: 指定格式 YYYYMM
     **/
    public synchronized static String getDate_YYYYMM(Date date) {
        return getFormatDate(date, YYYYMM);
    }

    /**
     * @description: 指定格式 YYYYMMDDHH
     **/
    public synchronized static String getDate_YYYYMMDDHH(Date date) {
        return getFormatDate(date, YYYYMMDDHH);
    }

    /**
     * @description: 指定格式 YYYYMMDDHH
     **/
    public synchronized static Date getDate_YYYYMMDDHH_start(String yyyymmddhh) {
        try {
            return getDateTime(yyyymmddhh + "0000", YYYYMMDDHHMMSS);
        } catch (TechnologyException e) {
            e.printStackTrace();
            return new Date();
        }
    }

    public synchronized static Date getDate_yyyyMMdd_start(String yyyyMMdd) {
        try {
            return getDateTime(yyyyMMdd + "000000", YYYYMMDDHHMMSS);
        } catch (TechnologyException e) {
            e.printStackTrace();
            return new Date();
        }
    }

    /**
     * @description: 指定格式 YYYYMMDDHH
     **/
    public synchronized static Date getDate_YYYYMMDDHH_end(String yyyymmddhh) {
        try {
            return getDateTime(yyyymmddhh + "5959", YYYYMMDDHHMMSS);
        } catch (TechnologyException e) {
            e.printStackTrace();
            return new Date();
        }
    }

    public synchronized static Date getDate_yyyyMMdd_end(String yyyyMMdd) {
        try {
            return getDateTime(yyyyMMdd + "235959", YYYYMMDDHHMMSS);
        } catch (TechnologyException e) {
            e.printStackTrace();
            return new Date();
        }
    }

    /**
     * @description: 指定格式 YYYY_MM_DD_HH_MM_SS
     **/
    public synchronized static String getDate_YYYY_MM_DD_HH_MM_SS(Date date, String timeZone) {
        return getFormatDate(sysTimeToTimeZone(date, timeZone), YYYY_MM_DD_HH_MM_SS);
    }

    /**
     * @description: 指定格式 YYYYMMDD
     **/
    public synchronized static String getDate_YYYYMMDD(Date date, String timeZone) {
        return getFormatDate(sysTimeToTimeZone(date, timeZone), YYYYMMDD);
    }

    /**
     * @description: 指定格式 YYYYMM
     **/
    public synchronized static String getDate_YYYYMM(Date date, String timeZone) {
        return getFormatDate(sysTimeToTimeZone(date, timeZone), YYYYMM);
    }

    /**
     * @description: 时间字符串转换默认格式
     **/
    public static Date getDefaultDateTime(String dateStr, String timeZone) throws TechnologyException {
        return sysTimeToTimeZone(getDateTime(dateStr, YYYY_MM_DD_HH_MM_SS), timeZone);
    }

    /**
     * @description: 时间字符串转换默认时区默认格式
     **/
    public static Date getDefaultDateTime(String dateStr) throws TechnologyException {
        return sysTimeToTimeZone(getDateTime(dateStr, YYYY_MM_DD_HH_MM_SS), null);
    }

    /**
     * @Description: 字符串时间解析
     */
    public synchronized static Date getDateTime(String dateStr, String format) throws TechnologyException {
        try {
            SimpleDateFormat sdf = new SimpleDateFormat(format);
            return sdf.parse(dateStr);
        } catch (Exception e) {
            throw new TechnologyException(e, ServerExp.时间解析异常);
        }
    }

    /**
     * @description: 获取当前时区时间字符串 yyyyMMdd
     **/
    public static String getTimeZoneStr(String timeZone) {
        return getFormatDate(new Date(getLongTime(timeZone)), YYYYMMDD);
    }


    /**
     * @description: 获取对应时区时间字符串
     **/
    public static String getTimeZoneStr(Date date, String timeZone) {
        return getFormatDate(new Date(getLongTime(date, timeZone)), YYYY_MM_DD_HH_MM_SS);
    }

    /**
     * @description: 获取对应时区时间字符串 默认8 时区
     **/
    public static String getTimeZoneStr(Date date) {
        return getTimeZoneStr(date, null);
    }

    /**
     * @Description: 将时间转换成指定格式
     */
    public synchronized static String getFormatDate(Date date, String format) {
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        return sdf.format(date);
    }

    /**
     * @Description: 将时间转换成指定格式
     */
    public synchronized static String getRFC3339FormatDate(Date date) {
        ZonedDateTime zonedDateTime = date.toInstant()
                .atZone(ZoneId.of("Asia/Shanghai"));
        // 定义 RFC 3339 格式
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ssXXX");
        // 格式化失效时间为 RFC 3339 格式字符串
        return zonedDateTime.format(formatter);
    }

    /**
     * @description: 时间格式化不报异常
     **/
    public synchronized static Date getFormatDate(String dateStr, String format) {
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        try {
            return sdf.parse(dateStr);
        } catch (ParseException e) {
        }
        return new Date();
    }

    /**
     * @description: 时区转当前时间毫秒数
     **/
    public static long getLongTime(String timeZone) {
        return getLongTime(new Date(), timeZone);
    }

    /**
     * @description: 时区转指定时间毫秒数
     **/
    public static long getLongTime(Date date, String timeZone) {
        if (StringUtils.isBlank(timeZone)) {
            timeZone = "8";
        }
        return (long) (date.getTime() + ((Double.parseDouble(timeZone) - 8) * 3600000));
    }

    /**
     * @description: 时区时间转系统时间
     **/
    public static Date sysTimeToTimeZone(Date date, String timeZone) {
        if (StringUtils.isBlank(timeZone)) {
            timeZone = "8";
        }
        return new Date((long) (date.getTime() + ((Double.parseDouble(timeZone) - 8) * 3600000)));
    }

    /**
     * @description: 时区时间转系统时间
     **/
    public static Date timeZoneToSysTime(Date date, String timeZone) {
        if (StringUtils.isBlank(timeZone)) {
            timeZone = "8";
        }
        return new Date((long) (date.getTime() + ((8 - Double.parseDouble(timeZone)) * 3600000)));
    }

    /**
     * @Description 时间转换成整点
     */
    public static Date formatDate(Date date, String format) throws TechnologyException {
        return getDateTime(getFormatDate(date, format), format);
    }

    /**
     * @Description: 获取当前年最后一天字符串 ——2018
     */
    public static String getNowYearVal() {
        return getFormatDate(getLastDayOfYear(new Date()), YYYY);
    }

    public static String getNowYearVal(Date date) {
        return getFormatDate(getLastDayOfYear(date), YYYY);
    }

    public static String getCNowYearVal() {
        return getFormatDate(getCurrertDate(new Date()), C_YYYY);
    }

    /**
     * @Description: 获取当前月最后一天字符串_201806
     */
    public static String getNowMonthVal() {
        return getFormatDate(getLastDayOfMonth(new Date()), YYYYMM);
    }

    public static String getNowMonthVal(Date date) {
        return getFormatDate(getLastDayOfMonth(date), YYYYMM);
    }

    public static String getCNowMonthVal() {
        return getFormatDate(getLastDayOfMonth(new Date()), C_YYYYMM);
    }

    /**
     * @Description: 获取今年当前第几周——2018-W23
     */
    public static String getNowWeekVal() {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date());
        return getFormatDate(getLastDayOfMonth(new Date()), YYYY) + "-W" + calendar.get(Calendar.WEEK_OF_YEAR);
    }

    public static String getNowWeekVal(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        return getFormatDate(getLastDayOfMonth(date), YYYY) + "-W" + calendar.get(Calendar.WEEK_OF_YEAR);
    }

    public static String getCNowWeekVal() {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date());
        return getFormatDate(getLastDayOfMonth(new Date()), C_YYYY) + "-第[" + calendar.get(Calendar.WEEK_OF_YEAR) + "]周";
    }

    /**
     * @Description: 获取当前时间字符串 ——20180609
     */
    public static String getNowDayVal() {
        return getFormatDate(getCurrertDate(new Date()), YYYYMMDD);
    }

    public static String getNowDayVal(Date date) {
        return getFormatDate(getCurrertDate(date), YYYYMMDD);
    }

    public static String getCNowDayVal() {
        return getFormatDate(getCurrertDate(new Date()), C_YYYYMMDD);
    }

    /**
     * @Description: 获取两个时间差，（分钟）
     */
    public static Integer getDateMinutes(Date date1, Date date2) {
        long time1 = date1.getTime();
        long time2 = date2.getTime();
        long diff = 0;
        if (time1 < time2) {
            diff = time2 - time1;
        } else {
            diff = time1 - time2;
        }
        return (int) (diff / 60000);
    }

    /**
     * @Description: 获取两时间差的天数
     */
    public static int getDateNum(Date fromDate, Date endDate) {
        long days = (endDate.getTime() - fromDate.getTime())
                / (1000 * 60 * 60 * 24);
        return (int) days;
    }

    /**
     * @Description: 获取时间年的第一天
     */
    public static Date getFirstDayOfYear(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(Calendar.DAY_OF_YEAR, 1);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        Date firstDayOfYear = calendar.getTime();
        return firstDayOfYear;
    }

    /**
     * @Description: 获取时间年的最后一天
     */
    public static Date getLastDayOfYear(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        int year = calendar.get(Calendar.YEAR);
        calendar.clear();
        calendar.set(Calendar.YEAR, year);
        calendar.roll(Calendar.DAY_OF_YEAR, -1);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        return calendar.getTime();
    }

    /**
     * 获取时间月份的第一天
     */
    public static Date getFirstDayOfMonth(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(Calendar.DAY_OF_MONTH, 1);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        Date firstDayOfMonth = calendar.getTime();
        return firstDayOfMonth;
    }

    /**
     * @Author 获取yyyyMM第一天
     * @Description //TODO
     * @Date 15:06 2018/11/15
     **/
    public static Date getFirstDayOfMonth(String dateStr) {
        if (StringUtils.isBlank(dateStr)) {
            return null;
        }
        Date date = strToDateNotDD(dateStr);
        LocalDateTime localDateTime = LocalDateTime.ofInstant(Instant.ofEpochMilli(date.getTime()), ZoneId.systemDefault());
        LocalDateTime endOfDay = localDateTime.with(TemporalAdjusters.firstDayOfMonth()).with(LocalTime.MIN);
        return Date.from(endOfDay.atZone(ZoneId.systemDefault()).toInstant());
    }

    /**
     * @Description 获取yyyyMM最后一天
     * @Author wanglie
     * @Date 15:07 2018/11/15
     **/
    public static Date getLastDayOfMonth(String dateStr) {
        if (StringUtils.isBlank(dateStr)) {
            return null;
        }
        Date date = strToDateNotDD(dateStr);
        LocalDateTime localDateTime = LocalDateTime.ofInstant(Instant.ofEpochMilli(date.getTime()), ZoneId.systemDefault());
        LocalDateTime endOfDay = localDateTime.with(TemporalAdjusters.lastDayOfMonth()).with(LocalTime.MAX);
        return Date.from(endOfDay.atZone(ZoneId.systemDefault()).toInstant());
    }

    private static Date strToDateNotDD(String strDate) {
        if (StringUtils.isBlank(strDate)) {
            return null;
        }
        SimpleDateFormat formatter = new SimpleDateFormat("yyyyMM");
        ParsePosition pos = new ParsePosition(0);
        Date strtodate = formatter.parse(strDate, pos);
        return strtodate;
    }

    /**
     * 获取时间月份的最后一天
     */
    public static Date getLastDayOfMonth(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(Calendar.DAY_OF_MONTH, 1);
        calendar.add(Calendar.MONTH, 1);
        calendar.add(Calendar.DAY_OF_MONTH, -1);
        Date lastDayOfMonth = calendar.getTime();
        return lastDayOfMonth;
    }

    /**
     * @Description: 获取时间周的第一天
     */
    public static Date getFirstDayOfWeek(Date date) {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        if (c.get(Calendar.DAY_OF_WEEK) == 1) {
            c.add(Calendar.DAY_OF_MONTH, -1);
        }
        c.add(Calendar.DATE, c.getFirstDayOfWeek() - c.get(Calendar.DAY_OF_WEEK) + 1);
        return c.getTime();
    }

    /**
     * @Description: 获取时间周的最后一天
     */
    public static Date getLastDayOfWeek(Date date) {
        Date currertDate = getCurrertDate(date);
        int dayOfWeek = getDayOfWeek(currertDate);
        if (dayOfWeek == 0) {
            return currertDate;
        }
        return addDate(currertDate, 7 - dayOfWeek);
    }

    /**
     * @Description: 获取时间天的开始
     */
    public static Date getCurrertDate(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setFirstDayOfWeek(Calendar.MONDAY);
        calendar.setTime(date);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        return calendar.getTime();
    }

    /**
     * @Description: 获取时间天的结束
     */
    public static Date getEndDate(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setFirstDayOfWeek(Calendar.MONDAY);
        calendar.setTime(date);
        calendar.set(Calendar.HOUR_OF_DAY, 23);
        calendar.set(Calendar.MINUTE, 59);
        calendar.set(Calendar.SECOND, 59);
        return calendar.getTime();
    }

    // 获得当前月--开始日期
    public static String getMinMonthDate(String date) {
        try {
            return dateFormat.format(getMinMonthDate(dateFormat.parse(date)));
        } catch (java.text.ParseException e) {
            e.printStackTrace();
            return null;
        }
    }

    // 获得当前月--开始日期
    public static Date getMinMonthDate(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(Calendar.DAY_OF_MONTH, calendar.getActualMinimum(Calendar.DAY_OF_MONTH));
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        return calendar.getTime();
    }

    // 获得当前月--结束日期
    public static String getMaxMonthDate(String date) {
        try {
            return dateFormat.format(getMaxMonthDate(dateFormat.parse(date)));
        } catch (ParseException e) {
            e.printStackTrace();
            return null;
        }
    }


    // 获得当前月--结束日期
    public static Date getMaxMonthDate(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(Calendar.DAY_OF_MONTH, calendar.getActualMaximum(Calendar.DAY_OF_MONTH));
        calendar.set(Calendar.HOUR_OF_DAY, 23);
        calendar.set(Calendar.MINUTE, 59);
        calendar.set(Calendar.SECOND, 59);
        calendar.set(Calendar.MILLISECOND, 999);
        return calendar.getTime();
    }

    /**
     * 获取指定日期是一周的第几天,星期日是第一天
     */
    public static int getDayOfWeek(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setFirstDayOfWeek(Calendar.MONDAY);
        calendar.setTime(date);
        calendar.setMinimalDaysInFirstWeek(DAYS_OF_A_WEEK);
        return calendar.get(Calendar.DAY_OF_WEEK) - 1;
    }

    /**
     * @description: 加减时间 type 类型 参考 java.util.Calendar
     **/
    public static Date addTime(Date date, int type, int num) {
        Calendar calendar = Calendar.getInstance();
        calendar.setFirstDayOfWeek(Calendar.MONDAY);
        calendar.setTime(date);
        calendar.add(type, num);
        return calendar.getTime();
    }

    /**
     * @description: 秒加减
     **/
    public static Date addMilliSecond(Date date, int num) {
        return addTime(date, Calendar.MILLISECOND, num);
    }

    /**
     * @description: 秒加减
     **/
    public static Date addSecond(Date date, int num) {
        return addTime(date, Calendar.SECOND, num);
    }

    /**
     * @description: 分钟加减
     **/
    public static Date addMin(Date date, int num) {
        return addTime(date, Calendar.MINUTE, num);
    }

    /**
     * @description: 小时加减
     **/
    public static Date addHour(Date date, int num) {
        return addTime(date, Calendar.HOUR, num);
    }

    /**
     * @description: 半天加减
     **/
    public static Date addHalfDay(Date date, int num) {
        return addTime(date, Calendar.AM_PM, num);
    }

    /**
     * @Description: 天加减
     */
    public static Date addDate(Date date, int num) {
        return addTime(date, Calendar.DATE, num);
    }

    /**
     * @Description: 周加减
     */
    public static Date addWeek(Date date, int num) {
        return addTime(date, Calendar.WEEK_OF_MONTH, num);
    }

    // public static void main(String[] args) {
    //     Date date = new Date();
    //     System.out.println(date.getTime());
    //     System.out.println(getFormatDate(date,YYYYMMDDHHMMSSS));

    //     System.out.println(addSecond(date,10).getTime());
    //     System.out.println(addMilliSecond(date,1000).getTime());

    //     System.out.println(getFormatDate(addMin(new Date(), 1), YYYY_MM_DD_HH_MM_SS_C));
    //     System.out.println(getFormatDate(addHour(new Date(), 1), YYYY_MM_DD_HH_MM_SS_C));
    //     System.out.println(getFormatDate(addHalfDay(new Date(), 1), YYYY_MM_DD_HH_MM_SS_C));
    //     System.out.println(getFormatDate(addDate(new Date(), 1), YYYY_MM_DD_HH_MM_SS_C));
    //     System.out.println(getFormatDate(addWeek(new Date(), 1), YYYY_MM_DD_HH_MM_SS_C));
    //     System.out.println(getFormatDate(addMonth(new Date(), 1), YYYY_MM_DD_HH_MM_SS_C));
    //     System.out.println(getFormatDate(addYear(new Date(), 1), YYYY_MM_DD_HH_MM_SS_C));

    //     String yyyymmddhh = getDate_YYYYMMDDHH(new Date());
    //     System.out.println(yyyymmddhh);
    //     System.out.println(getDate_YYYYMMDDHH_start(yyyymmddhh));
    //     System.out.println(getDate_YYYYMMDDHH_end(yyyymmddhh));
    // }

    /**
     * @Description: 月加减
     */
    public static Date addMonth(Date date, int num) {
        return addTime(date, Calendar.MONTH, num);
    }

    /**
     * @description: 年加减
     **/
    public static Date addYear(Date date, int num) {
        return addTime(date, Calendar.YEAR, num);
    }

    /**
     * @Description 获取上个月这个时间点
     */
    public static Date getLastMonth(Date date) {
        return addMonth(date, -1);
    }

    /**
     * @Description: 获取当前时间格式（yyyyMMddHHmmss）
     */
    public static String getCurrentDateTime() {
        return getFormatDate(new Date(), LOCATE_DATE_FORMAT);
    }

    /**
     * @Description: long时间转换成指令时间格式 （yyyyMMddHHmmss）
     */
    public static String getDateTime(long time) {
        return getFormatDate(new Date(time), LOCATE_DATE_FORMAT);
    }

    /**
     * @Description: 获取微信订单结束时间
     */
    public static String getWxOrderCloseTime(Date data, int second) {
        long time = data.getTime() + second;
        return getDateTime(time);
    }

    /**
     * @Description: 获取微信订单创建时间
     */
    public static String getWxOrderStartTime(Date date) {
        return getDateTime(date.getTime());
    }

    /**
     * @Description: 获取指定支付订单的失效时间
     */
    public static String getAliOrderCloseTime(int second) {
        int min = second / 60000;
        return min + "m";
    }

    /**
     * @Description: 获取指定时间的前一天
     */
    public static Date getDateOfYesterday(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.DAY_OF_MONTH, -1);
        return calendar.getTime();
    }

    /**
     * @Description 获取年月日
     */
    public static String getYearyMounthDay(Date date) {
        return getFormatDate(date, YYYY_MM_DD);
    }

    /**
     * @Description 获取一天的最后时间
     */
    public static Date getDayLastTime(Date date) throws TechnologyException {
        return getDateTime(getYearyMounthDay(date) + " 23:59:59", YYYY_MM_DD_HH_MM_SS);
    }

    /**
     * 获取明天的字符串
     */
    public static String getTomorrowVal() {
        Calendar calendar = Calendar.getInstance();
        SimpleDateFormat sdf = new SimpleDateFormat(YYYYMMDD);
        calendar.add(Calendar.DAY_OF_YEAR, 1);
        Date date = calendar.getTime();
        return sdf.format(date);
    }


    /**
     * @Description 不包含起止日期
     */
    public static List<String> getMonthBetweenDateStr(String minDate, String maxDate) throws TechnologyException {
        List<String> listDate = new ArrayList<String>();
        try {
            Calendar startCalendar = Calendar.getInstance();
            Calendar endCalendar = Calendar.getInstance();
            SimpleDateFormat df = new SimpleDateFormat(YYYYMMDD);
            Date startDate = df.parse(minDate);
            startCalendar.setTime(startDate);
            Date endDate = df.parse(maxDate);
            endCalendar.setTime(endDate);
            while (true) {
                startCalendar.add(Calendar.DAY_OF_MONTH, 1);
                if (startCalendar.getTimeInMillis() < endCalendar.getTimeInMillis()) {
                    listDate.add(df.format(startCalendar.getTime()));
                } else {
                    break;
                }
            }
        } catch (ParseException e) {
            throw new TechnologyException(e, ServerExp.时间解析异常);
        }
        return listDate;
    }


    /**
     * @Description 包含起止日期
     */
    public static List<Date> getMonthBetweenDate(Date beginDate, Date endDate) {
        if (beginDate.getTime() == endDate.getTime()) {
            return null;
        }
        List lDate = new ArrayList();
        lDate.add(beginDate);//把开始时间加入集合
        Calendar cal = Calendar.getInstance();
        //使用给定的 Date 设置此 Calendar 的时间
        cal.setTime(beginDate);
        boolean bContinue = true;
        while (bContinue) {
            //根据日历的规则，为给定的日历字段添加或减去指定的时间量
            cal.add(Calendar.DAY_OF_MONTH, 1);
            // 测试此日期是否在指定日期之后
            if (endDate.after(cal.getTime())) {
                lDate.add(cal.getTime());
            } else {
                break;
            }
        }
        lDate.add(endDate);//把结束时间加入集合
        return lDate;
    }

    /**
     * @Description 获取月的天数
     */
    public static List<Date> getMonthDay(String dateStr) throws GeneralException {
        if (StringUtils.isBlank(dateStr)) {
            throw new ProfessionalException("时间字符串不能为空");
        }

        Date startTime = getDateTime(dateStr, YYYYMM);
        Date endTime = getLastDayOfMonth(startTime);
        return getMonthBetweenDate(startTime, endTime);
    }

    public static List<String> getMonthDayStr(String dateStr) throws GeneralException {
        if (StringUtils.isBlank(dateStr)) {
            throw new ProfessionalException("时间字符串不能为空");
        }

        Date dateTime = getDateTime(dateStr, YYYYMM);
        return getMonthBetweenDateStr(getFormatDate(addDate(dateTime, -1), YYYYMMDD), getFormatDate(addDate(getLastDayOfMonth(dateTime), 1), YYYYMMDD));
    }

    /**
     * @Description: 获取时间天的结束
     */
    public static Date getDayEndDate(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.setFirstDayOfWeek(Calendar.MONDAY);
        calendar.set(Calendar.HOUR_OF_DAY, 23);
        calendar.set(Calendar.MINUTE, 59);
        calendar.set(Calendar.SECOND, 59);
        calendar.set(Calendar.MILLISECOND, 999);
        return calendar.getTime();
    }

    public static List<String> getMonthBetween(String minDate, String maxDate) throws GeneralException {
        try {
            int mi = Integer.parseInt(minDate);
            int mx = Integer.parseInt(maxDate);
            if (mi > mx) {
                throw new ProfessionalException(ServerExp.时间解析异常);
            }
        } catch (NumberFormatException e) {
            throw new TechnologyException(e, ServerExp.时间解析异常);
        }

        ArrayList<String> result = new ArrayList<String>();
        SimpleDateFormat sdf = new SimpleDateFormat(YYYYMM);//格式化为年月
        Calendar min = Calendar.getInstance();
        Calendar max = Calendar.getInstance();
        try {
            min.setTime(sdf.parse(minDate));
            min.set(min.get(Calendar.YEAR), min.get(Calendar.MONTH), 1);
            max.setTime(sdf.parse(maxDate));
            max.set(max.get(Calendar.YEAR), max.get(Calendar.MONTH), 2);
        } catch (ParseException e) {
            throw new TechnologyException(e, ServerExp.时间解析异常);
        }
        Calendar curr = min;
        while (curr.before(max)) {
            result.add(sdf.format(curr.getTime()));
            curr.add(Calendar.MONTH, 1);
        }
        return result;
    }

    /**
     * 指定时间前后推几个月
     */
    public static String getPassMonthDate(int flag, String format, Date date) {
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        Calendar cl = Calendar.getInstance();
        cl.setTime(date);
        cl.add(Calendar.MONTH, flag);
        //从现在算，之前一个月,如果是2个月，那么-1-----》改为-2
        Date dateFrom = cl.getTime();
        return sdf.format(dateFrom);
    }

    /**
     * 当前时间前后推几个月
     */
    public static String getPassMonthNowDate(int flag, String format) {
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        Calendar cl = Calendar.getInstance();
        cl.add(Calendar.MONTH, flag);
        //从现在算，之前一个月,如果是2个月，那么-1-----》改为-2
        Date dateFrom = cl.getTime();
        return sdf.format(dateFrom);
    }

    /**
     * @description: LocalDate, LocalDateTime <==> Date
     **/
    public static Date asDate(LocalDate localDate) {
        return Date.from(localDate.atStartOfDay().atZone(ZoneId.systemDefault()).toInstant());
    }

    public static Date asDate(LocalDateTime localDateTime) {
        return Date.from(localDateTime.atZone(ZoneId.systemDefault()).toInstant());
    }

    public static LocalDate asLocalDate(Date date) {
        return Instant.ofEpochMilli(date.getTime()).atZone(ZoneId.systemDefault()).toLocalDate();
    }

    public static LocalDateTime asLocalDateTime(Date date) {
        return Instant.ofEpochMilli(date.getTime()).atZone(ZoneId.systemDefault()).toLocalDateTime();
    }
}
