package com.jhscale.common.utils;

import com.jhscale.common.em.Platform;
import com.jhscale.common.em.ServerExp;
import com.jhscale.common.em.SyncDataType;
import com.jhscale.common.exception.TechnologyException;
import com.jhscale.common.model.file.ImageSimilar;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.*;
import java.math.BigDecimal;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import static com.jhscale.common.model.device.DConstant.*;
import static com.jhscale.common.utils.DateUtils.HHMMSSSSS;

/**
 * @author lie_w
 * @title: FileUtil
 * @projectName common
 * @description: TODO
 * @date 2020-11-151:06
 */
public class FileUtils {

    // private static final Logger LOGGER = LoggerFactory.getLogger(FileUtils.class);

    private FileUtils() {
    }

    /**
     * @description: 需要自行实现流处理事件
     **/
    public interface FileInputStreamExecute {

        /**
         * @description: 处理流文件
         **/
        void inputStream(InputStream inputStream) throws TechnologyException;

        /**
         * @description: 获取自定义的结果信息
         **/
        Object result();
    }

    /**
     * @description: 地址获取文件名
     **/
    public static String getFileName(String url) {
        return url.substring(url.lastIndexOf('/') + 1);
    }

    /**
     * @description: 文件转字节数组
     **/
    public static byte[] getFileBytes(File file) throws TechnologyException {
        try {
            return inputStreamToBytes(new FileInputStream(file));
        } catch (FileNotFoundException e) {
            e.printStackTrace();
            // LOGGER.error("文件转字节数组异常：{}", e.getMessage(), e);
            throw new TechnologyException(e, ServerExp.文件未发现);
        }
    }

    /**
     * @param url 文件地址
     * @Despription: 地址下载输入流 需要手动关闭流 可能存在内存移除问题
     */
    public static void getInputStream(String url, FileInputStreamExecute execute) throws TechnologyException {
        InputStream inputStream = null;
        try {
            URLConnection conn = new URL(url).openConnection();
            inputStream = conn.getInputStream();
            if (Objects.nonNull(execute)) execute.inputStream(inputStream);
        } catch (MalformedURLException e) {
            e.printStackTrace();
            // LOGGER.error("MalformedURLException 异常：{}", e.getMessage(), e);
            throw new TechnologyException(e, ServerExp.URL识别异常);
        } catch (IOException e) {
            e.printStackTrace();
            //LOGGER.error("地址下载输入流 异常：{}", e.getMessage(), e);
            throw new TechnologyException(e, ServerExp.IO异常);
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException ignore) {
                }
            }
        }
    }

    /**
     * @description: 流转字节数组
     **/
    public static byte[] inputStreamToBytes(InputStream inputStream) {
        byte[] bytes = null;
        ByteArrayOutputStream baos = null;
        try {
            baos = new ByteArrayOutputStream();
            byte[] buffer = new byte[4096];
            int len = 0;
            while (-1 != (len = inputStream.read(buffer))) {
                baos.write(buffer, 0, len);
            }
            bytes = baos.toByteArray();
        } catch (IOException e) {
            e.printStackTrace();
            // LOGGER.error("流转字节数组 异常：{}", e.getMessage(), e);
        } finally {
            if (baos != null) {
                try {
                    baos.close();
                } catch (IOException e) {
                }
            }
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                }
            }
        }
        return bytes;
    }

    /**
     * @param url 文件地址
     * @Despription: 地址下载二进制字节流
     */
    public static byte[] getBytes(String url) throws TechnologyException {
        FileInputStreamExecute execute = new FileInputStreamExecute() {
            byte[] bytes = null;

            /**
             * @description: 处理流文件
             */
            @Override
            public void inputStream(InputStream inputStream) throws TechnologyException {
                this.bytes = inputStreamToBytes(inputStream);
            }

            /**
             * @description: 获取自定义的结果信息
             **/
            @Override
            public Object result() {
                return bytes;
            }
        };
        getInputStream(url, execute);
        return (byte[]) execute.result();
    }

    /**
     * @param url 文件地址
     * @Despription: 地址下载文件
     */
    public static void getFile(String url, File file) throws TechnologyException {
        BufferedOutputStream bos = null;
        FileOutputStream fos = null;
        byte[] bytes = getBytes(url);
        try {
            //输出流
            fos = new FileOutputStream(file);
            //缓冲流
            bos = new BufferedOutputStream(fos);
            //将字节数组写出
            bos.write(bytes);
        } catch (FileNotFoundException e) {
            throw new TechnologyException(e, ServerExp.文件未发现);
        } catch (IOException e) {
            throw new TechnologyException(e, ServerExp.IO异常);
        } finally {
            if (bos != null) {
                try {
                    bos.close();
                } catch (IOException e) {
                }
            }
            if (fos != null) {
                try {
                    fos.close();
                } catch (IOException ignore) {
                }
            }
        }
    }

    /**
     * @param url 文件地址
     * @Despription: 地址下载文件
     */
    public static File getFile(String url, String filePath, String fileName) throws TechnologyException {
        try {
            File pathFile = new File(filePath);
            if (!pathFile.exists()) pathFile.mkdirs();
            File file = new File(pathFile, fileName);
            if (!file.exists()) file.createNewFile();
            getFile(url, file);
            return file;
        } catch (IOException e) {
            throw new TechnologyException(e, ServerExp.IO异常);
        }
    }

    /**
     * @param url 文件地址
     * @Despription: 地址下载文件 保存全路径
     */
    public static File getFileByFullPath(String url, String fullPath) throws TechnologyException {
        try {
            File file = new File(fullPath);
            if (!file.exists()) {
                File parentFile = file.getParentFile();
                if (!parentFile.exists()) parentFile.mkdirs();
                file.createNewFile();
            }
            getFile(url, file);
            return file;
        } catch (IOException e) {
            throw new TechnologyException(e, ServerExp.IO异常);
        }
    }

    /**
     * @description: 地址下载文件 保存父路径
     **/
    public static File getFileByParentPath(String url, String parentPath) throws TechnologyException {
        try {
            File parentFile = new File(parentPath);
            if (!parentFile.exists()) parentFile.mkdirs();
            File file = new File(parentFile, getFileName(url));
            file.createNewFile();
            getFile(url, file);
            return file;
        } catch (IOException e) {
            throw new TechnologyException(e, ServerExp.IO异常);
        }
    }

    /**
     * @description: 写文件
     **/
    public static boolean writer(String content, File file) {
        return writer(content, file, false);
    }

    /**
     * @description: 写文件
     **/
    public static boolean writer(String content, File file, boolean append) {
        return writer(content.getBytes(), file, append);
    }

    /**
     * @description: 写文件
     **/
    public static boolean writer(byte[] bytes, File file) {
        return writer(bytes, file, false);
    }

    /**
     * @description: 写文件
     **/
    public static boolean writer(byte[] bytes, File file, boolean append) {
        if (file != null) {
            FileOutputStream fos = null;

            try {
                if (!file.exists()) {
                    File parentFile = file.getParentFile();
                    if (!parentFile.exists()) parentFile.mkdirs();
                    file.createNewFile();
                }

                fos = new FileOutputStream(file, append);
                fos.write(bytes, 0, bytes.length);
                return true;
            } catch (IOException e) {
                e.printStackTrace();
            } finally {
                if (fos != null) {
                    try {
                        fos.close();
                    } catch (IOException e) {
                    }
                }
            }
        }
        return false;
    }

    /**
     * @description: 读文件
     **/
    public static String reader(File file) {
        if (file == null || !file.exists() || file.isDirectory()) {
            return null;
        }

        FileReader fr = null;
        BufferedReader br = null;

        try {
            fr = new FileReader(file);
            br = new BufferedReader(fr);
            StringBuffer buffer = new StringBuffer();
            while (br.ready()) {
                buffer = buffer.append(br.readLine());
            }
            return buffer.toString();
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (br != null) {
                try {
                    br.close();
                } catch (IOException e) {
                }
            }
            if (fr != null) {
                try {
                    fr.close();
                } catch (IOException e) {
                }
            }
        }
        return null;
    }

    /**
     * @description: 读行文件
     **/
    public static String reader(InputStream inputStream) {
        InputStreamReader reader = null;
        BufferedReader br = null;
        StringBuilder buffer = new StringBuilder();
        try {
            reader = new InputStreamReader(inputStream, FILE_Charset);
            br = new BufferedReader(reader);
            String line = "";
            while ((line = br.readLine()) != null) {
                buffer.append(line).append(PUBLIC_LINE_SPLIT);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (br != null) {
                try {
                    br.close();
                } catch (IOException e) {
                }
            }
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException e) {
                }
            }
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                }
            }
        }
        return buffer.toString();
    }

    /**
     * @description: 写文件
     **/
    public static boolean write(String content, OutputStream stream) {
        try {
            byte[] bytes = content.getBytes(FILE_Charset);
            stream.write(bytes, 0, bytes.length);
            stream.flush();
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        } finally {
            if (stream != null) {
                try {
                    stream.close();
                } catch (IOException e) {
                }
            }
        }
    }


    /**
     * @description: 读行文件
     **/
    public static String readerLine(File file) {
        if (file == null || !file.exists() || file.isDirectory()) {
            return null;
        }

        FileReader fr = null;
        BufferedReader br = null;

        try {
            fr = new FileReader(file);
            br = new BufferedReader(fr);
            StringBuffer buffer = new StringBuffer();
            while (br.ready()) {
                buffer = buffer.append(br.readLine() + "\r\n");
            }
            return buffer.toString();
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (br != null) {
                try {
                    br.close();
                } catch (IOException e) {
                }
            }
            if (fr != null) {
                try {
                    fr.close();
                } catch (IOException e) {
                }
            }
        }
        return null;
    }

    /**
     * @description: 图片文件比较测试
     **/
    /**
     * @description: 改变成二进制码
     **/
    public static int[][][] getPX(String path) {
        File file = new File(path);
        BufferedImage bi = null;
        try {
            bi = ImageIO.read(file);
        } catch (Exception e) {
            e.printStackTrace();
        }

        int width = bi.getWidth();
        int height = bi.getHeight();
        int minx = bi.getMinX();
        int miny = bi.getMinY();
        int[][][] list = new int[width][height][3];
        for (int i = minx; i < width; i++) {
            for (int j = miny; j < height; j++) {
                int[] rgb = new int[3];
                int pixel = bi.getRGB(i, j);
                rgb[0] = (pixel & 0xff0000) >> 16;
                rgb[1] = (pixel & 0xff00) >> 8;
                rgb[2] = (pixel & 0xff);
                list[i][j] = rgb;

            }
        }
        return list;
    }

    /**
     * @description: 图片比较
     **/
    @SuppressWarnings("all")
    public static ImageSimilar compareImage(String imgPath1, String imgPath2) {
        String[] images = {imgPath1, imgPath2};
        if (images.length == 0) {
            System.out.println("Usage >java BMPLoader ImageFile.bmp");
            System.exit(0);
        }

        // 分析图片相似度 begin
        int[][][] list1 = getPX(images[0]);
        int[][][] list2 = getPX(images[1]);
        int xiangsi = 0;
        int busi = 0;
        int i = 0, j = 0;
        for (int[][] strings : list1) {
            if ((i + 1) == list1.length) {
                continue;
            }
            for (int m = 0; m < strings.length; m++) {
                try {
                    int[] value1 = list1[i][j];
                    int[] value2 = list2[i][j];
                    int k = 0;
                    for (int n = 0; n < value2.length; n++) {
                        if (Math.abs(value1[k] - value2[k]) < 5) {
                            xiangsi++;
                        } else {
                            busi++;
                        }
                    }
                } catch (RuntimeException e) {
                    continue;
                }
                j++;
            }
            i++;
        }
        // list1 = getPX(images[1]);
        // list2 = getPX(images[0]);
        // i = 0;
        // j = 0;
        // for (int[][] strings : list1) {
        //     if ((i + 1) == list1.length) {
        //         continue;
        //     }
        //     for (int m = 0; m < strings.length; m++) {
        //         try {
        //             int[] value1 = list1[i][j];
        //             int[] value2 = list2[i][j];
        //             int k = 0;
        //             for (int n = 0; n < value2.length; n++) {
        //                 if (Math.abs(value1[k] - value2[k]) < 5) {
        //                     xiangsi++;
        //                 } else {
        //                     busi++;
        //                 }
        //             }
        //         } catch (RuntimeException e) {
        //             continue;
        //         }
        //         j++;
        //     }
        //     i++;
        // }
        BigDecimal bxiangsi = new BigDecimal(xiangsi);
        BigDecimal bbusi = new BigDecimal(busi);
        BigDecimal baifen = bxiangsi.divide(bxiangsi.add(bbusi), 4, BigDecimal.ROUND_HALF_UP);
        return new ImageSimilar(xiangsi, busi, baifen);
    }

//    public static void main(String[] args) {
//        // String fileName = getFileName("https://www.baidu.com/img/PCtm_d9c8750bed0b3c7d089fa7d55720d6cf.png");
//        // System.out.println(fileName);
//
//        // byte[] bytes = getBytes("https://www.baidu.com/img/PCtm_d9c8750bed0b3c7d089fa7d55720d6cf.png");
//        // System.out.println(bytes.length);
//
//        System.out.println(terminal_data_file_name(Platform.PS));
//        System.out.println(server_data_file_name(SyncDataType.CLS));
//    }

    /**
     * @description: 同步数据文件名
     **/
    public static String terminal_data_file_name(Platform platform) {
        return terminal_data_file_name(platform.getSign());
    }

    public static String terminal_data_file_name(String sign) {
        return sign + DateUtils.getFormatDate(new Date(), HHMMSSSSS) + DATA_FILE_TYPE;
    }

    /**
     * @param type com.jhscale.common.model.device.DConstant 数据类型
     * @description:
     **/
    public static String server_data_file_name(String type) {
        return Platform.S.getSign() + type + DATA_FILE_TYPE;
    }

    public static String server_data_file_name(SyncDataType type) {
        return server_data_file_name(type.getType());
    }

    /**
     * @description: 文件拷贝
     **/
    public static boolean copy(File source, File target) {
        InputStream inputStream = null;
        OutputStream outputStream = null;
        try {
            if (source == null || !source.exists() || target == null) return false;

            File parentFile = target.getParentFile();
            if (!parentFile.exists()) parentFile.mkdirs();

            inputStream = Files.newInputStream(source.toPath());
            outputStream = Files.newOutputStream(target.toPath());

            byte[] buffer = new byte[1024];
            int len;
            while ((len = inputStream.read(buffer)) != -1) {
                outputStream.write(buffer, 0, len);
            }
            return true;
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException ignored) {
                }
            }
            if (outputStream != null) {
                try {
                    outputStream.close();
                } catch (IOException ignored) {
                }
            }
        }
        return false;
    }

    /**
     * @description: 文件拷贝
     **/
    public static boolean copyFolder(File source, File target) {
        try {
            if (source.isDirectory()) {
                if (!target.exists()) {
                    target.mkdir();
                }

                String[] files = source.list();
                for (String file : files) {
                    File srcFile = new File(source, file);
                    File destFile = new File(target, file);
                    copyFolder(srcFile, destFile);
                }
                return true;
            } else {
                Files.copy(source.toPath(), target.toPath(), StandardCopyOption.REPLACE_EXISTING);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return false;
    }

    /**
     * @description: 文件移动
     **/
    public static boolean remove(File source, File target) {
        boolean result = copy(source, target);
        if (result) source.delete();
        return result;
    }

    /**
     * @description: 文件夹删除
     * @date: 2025-05-22 09:59:23
     **/
    public static boolean delete(File file) {
        if (file != null && file.exists()) {
            if (file.isDirectory()) {
                File[] files = file.listFiles();
                if (files != null) {
                    for (File f : files) {
                        delete(f);
                    }
                }
            }
            return file.delete();
        }
        return false;
    }

    /**
     * 裁剪图片 ,并压缩
     *
     * @param source 源图片地址
     * @param target 目标图片地址
     * @param suffix 文件后缀 png
     * @param w      宽
     * @param h      高
     */
    public static void cropImage(String source, String target, String suffix, int x, int y, int w, int h) {
        cropImage(new File(source), new File(target), suffix, x, y, w, h);
    }

    /**
     * 裁剪图片 ,并压缩
     *
     * @param source 源图片地址
     * @param target 目标图片地址
     * @param suffix 文件后缀 png
     * @param w      宽
     * @param h      高
     */
    public static void cropImage(File source, File target, String suffix, int x, int y, int w, int h) {
        // System.out.printf("裁剪图片  源地址：%s 目标地址：%s%n", source.getAbsoluteFile(), target.getAbsolutePath());
        try {
            BufferedImage bufferedimage = ImageIO.read(source);

            //转换标准jpg图片格式
            BufferedImage newBufferedImage = new BufferedImage(bufferedimage.getWidth(), bufferedimage.getHeight(), BufferedImage.TYPE_INT_RGB);
            newBufferedImage.createGraphics().drawImage(bufferedimage, 0, 0, Color.WHITE, null);
            bufferedimage = newBufferedImage;

            BufferedImage cut1 = PictureUtils.cut(bufferedimage, x, y, w, h);
            PictureUtils.write(cut1, suffix, target);
            System.out.printf("裁剪图片 %s  源地址：%s   目标地址：%s", source.exists(), source.getAbsoluteFile(), target.getAbsolutePath());
        } catch (Exception e) {
            e.printStackTrace();
            System.err.printf("裁剪图片 源地址：%s   目标地址：%s 异常：%s", source.getAbsoluteFile(), target.getAbsolutePath(), e.getMessage());
        }
    }

    private static final int BUFFER_SIZE = 2 * 1024;

    public static void zipDirectory(File sourceDir, File zipFile) {
        zipDirectory(sourceDir.getAbsolutePath(), zipFile.getAbsolutePath());
    }

    // 压缩目录
    public static void zipDirectory(String sourceDir, String zipFile) {
        try (ZipOutputStream zos = new ZipOutputStream(new FileOutputStream(zipFile))) {
            File dir = new File(sourceDir);
            zipDirectory(dir, dir.getName(), zos);
            zos.finish();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    private static void zipDirectory(File dir, String baseName, ZipOutputStream zos) throws IOException {
        File[] files = dir.listFiles();
        if (files != null) {
            for (File file : files) {
                if (file.isDirectory()) {
                    zipDirectory(file, baseName + File.separator + file.getName(), zos);
                } else {
                    try (FileInputStream fis = new FileInputStream(file)) {
                        ZipEntry zipEntry = new ZipEntry(baseName + File.separator + file.getName());
                        zos.putNextEntry(zipEntry);
                        byte[] buffer = new byte[BUFFER_SIZE];
                        int length;
                        while ((length = fis.read(buffer)) >= 0) {
                            zos.write(buffer, 0, length);
                        }
                        zos.closeEntry();
                    }
                }
            }
        }
    }

    public static void unzip(File zipFile, File destDir) {
        unzip(zipFile.getAbsolutePath(), destDir.getAbsolutePath());
    }

    public static void unzip(String zipFile, String destDir) {
        File dir = new File(destDir);
        if (!dir.exists()) {
            dir.mkdirs();
        }

        try (ZipInputStream zis = new ZipInputStream(new FileInputStream(zipFile))) {
            ZipEntry zipEntry;
            while ((zipEntry = zis.getNextEntry()) != null) {
                File newFile = new File(destDir, zipEntry.getName());
                if (zipEntry.isDirectory()) {
                    newFile.mkdirs();
                } else {
                    // 确保父目录存在
                    new File(newFile.getParent()).mkdirs();
                    try (FileOutputStream fos = new FileOutputStream(newFile)) {
                        byte[] buffer = new byte[1024];
                        int length;
                        while ((length = zis.read(buffer)) > 0) {
                            fos.write(buffer, 0, length);
                        }
                    }
                }
                zis.closeEntry();
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
    }


    /**
     * 压缩成ZIP 方法1
     *
     * @param srcDir           压缩文件夹路径
     * @param out              压缩文件输出流
     * @param KeepDirStructure 是否保留原来的目录结构,true:保留目录结构;
     *                         false:所有文件跑到压缩包根目录下(注意：不保留目录结构可能会出现同名文件,会压缩失败)
     * @throws RuntimeException 压缩失败会抛出运行时异常
     */
    public static void toZip(String srcDir, OutputStream out, boolean KeepDirStructure)
            throws RuntimeException {

        long start = System.currentTimeMillis();
        ZipOutputStream zos = null;
        try {
            zos = new ZipOutputStream(out);
            File sourceFile = new File(srcDir);
            compress(sourceFile, zos, sourceFile.getName(), KeepDirStructure);
            long end = System.currentTimeMillis();
            System.out.printf("压缩完成，耗时：%s ms%n", (end - start));
        } catch (Exception e) {
            throw new RuntimeException("zip error from ZipUtils", e);
        } finally {
            if (zos != null) {
                try {
                    zos.close();
                } catch (IOException ignored) {
                }
            }
        }
    }

    /**
     * 压缩成ZIP 方法2
     *
     * @param srcFiles 需要压缩的文件列表
     * @param out      压缩文件输出流
     * @throws RuntimeException 压缩失败会抛出运行时异常
     */
    public static void toZip(List<File> srcFiles, OutputStream out) throws RuntimeException {
        long start = System.currentTimeMillis();
        ZipOutputStream zos = null;
        try {
            zos = new ZipOutputStream(out);
            for (File srcFile : srcFiles) {
                byte[] buf = new byte[BUFFER_SIZE];
                zos.putNextEntry(new ZipEntry(srcFile.getName()));
                int len;
                FileInputStream in = new FileInputStream(srcFile);
                while ((len = in.read(buf)) != -1) {
                    zos.write(buf, 0, len);
                }
                zos.closeEntry();
                in.close();
            }
            long end = System.currentTimeMillis();
            System.out.printf("压缩完成，耗时：%s ms%n", (end - start));
        } catch (Exception e) {
            throw new RuntimeException("zip error from ZipUtils", e);
        } finally {
            if (zos != null) {
                try {
                    zos.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    /**
     * 递归压缩方法
     *
     * @param sourceFile       源文件
     * @param zos              zip输出流
     * @param name             压缩后的名称
     * @param KeepDirStructure 是否保留原来的目录结构,true:保留目录结构;
     *                         false:所有文件跑到压缩包根目录下(注意：不保留目录结构可能会出现同名文件,会压缩失败)
     */
    private static void compress(File sourceFile, ZipOutputStream zos, String name,
                                 boolean KeepDirStructure) throws Exception {
        byte[] buf = new byte[BUFFER_SIZE];
        if (sourceFile.isFile()) {
            // 向zip输出流中添加一个zip实体，构造器中name为zip实体的文件的名字
            zos.putNextEntry(new ZipEntry(name));
            // copy文件到zip输出流中
            int len;
            FileInputStream in = new FileInputStream(sourceFile);
            while ((len = in.read(buf)) != -1) {
                zos.write(buf, 0, len);
            }
            // Complete the entry
            zos.closeEntry();
            in.close();
        } else {
            File[] listFiles = sourceFile.listFiles();
            if (listFiles == null || listFiles.length == 0) {
                // 需要保留原来的文件结构时,需要对空文件夹进行处理
                if (KeepDirStructure) {
                    // 空文件夹的处理
                    zos.putNextEntry(new ZipEntry(name + File.separator));
                    // 没有文件，不需要文件的copy
                    zos.closeEntry();
                }

            } else {
                for (File file : listFiles) {
                    // 判断是否需要保留原来的文件结构
                    if (KeepDirStructure) {
                        // 注意：file.getName()前面需要带上父文件夹的名字加一斜杠,
                        // 不然最后压缩包中就不能保留原来的文件结构,即：所有文件都跑到压缩包根目录下了
                        compress(file, zos, name + File.separator + file.getName(), KeepDirStructure);
                    } else {
                        compress(file, zos, file.getName(), KeepDirStructure);
                    }

                }
            }
        }
    }
}
