package com.jhscale.common.utils;

import com.google.gson.*;
import com.google.gson.reflect.TypeToken;
import com.sun.org.apache.xml.internal.security.exceptions.Base64DecodingException;
import com.sun.org.apache.xml.internal.security.utils.Base64;
import org.apache.commons.lang3.StringUtils;

import java.lang.reflect.Type;
import java.nio.charset.Charset;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;

import static com.jhscale.common.utils.DateUtils.YYYY_MM_DD_HH_MM_SS;


/**
 * @author lie_w
 * @title: JSONUtils
 * @projectName common
 * @description: GJSON 工具类
 * @date 2020-11-0421:42
 */
public class GJSONUtils {

    private static Gson gson = null;

    static {
        gson = new Gson();
    }

    private GJSONUtils() {
    }

    public static class DateAdapter implements JsonSerializer<Date>, JsonDeserializer<Date> {
        private final DateFormat dateFormat = new SimpleDateFormat(YYYY_MM_DD_HH_MM_SS, Locale.getDefault());

        @Override
        public Date deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context)
                throws JsonParseException {
            try {
                return dateFormat.parse(json.getAsString());
            } catch (ParseException e) {
                throw new JsonParseException(e);
            }
        }

        @Override
        public JsonElement serialize(Date src, Type typeOfSrc, JsonSerializationContext context) {
            return new JsonPrimitive(dateFormat.format(src));
        }
    }

    public static void gson_time(){
        gson = new GsonBuilder()
                .registerTypeAdapter(Date.class, new DateAdapter())
                .create();
    }

    /**
     * @description: 判断JSON格式
     **/
    public static boolean isJSON(String json) {
        if (StringUtils.isNotBlank(json)) {
            try {
                JsonParser.parseString(json);
                return true;
            } catch (Exception e) {
            }
        }
        return false;
    }

    /**
     * @description: 对象 转 JSON
     **/
    public static String objectToJSON(Object obj) {
        if (Objects.nonNull(obj))
            return gson.toJson(obj);
        return null;
    }

    /**
     * @description: 字符串转 UTF8 字节流
     **/
    public static byte[] objectToUTF8Bytes(Object obj) {
        String json = objectToJSON(obj);
        return StringUtils.isNotBlank(json) ? json.getBytes(Charset.forName("UTF8")) : null;
    }

    /**
     * @description: 字符串转 UTF8 Base64 字符
     **/
    public static String objectToUTF8Base64Code(Object obj) {
        byte[] jsonbytes = objectToUTF8Bytes(obj);
        return Objects.nonNull(jsonbytes) ? Base64.encode(jsonbytes) : null;
    }

    /**
     * @description: BAS64 UTF8字符串转JSON
     **/
    public static String UTF8Base64CodeToJSON(String base64) {
        if (StringUtils.isNotBlank(base64)) {
            try {
                return new String(Base64.decode(base64), Charset.forName("UTF8"));
            } catch (Base64DecodingException e) {
                System.err.println("数据：{" + base64 + "} 转JSON字符串失败");
            }
        }
        return null;
    }

    /**
     * @description: BAS64 UTF8字符串转 Object
     **/
    public static <T> T UTF8Base64CodeToObject(String base64, Class<T> clazz) {
        String json = UTF8Base64CodeToJSON(base64);
        return StringUtils.isNotBlank(json) ? jsonToObject(json, clazz) : null;
    }

    /**
     * @description: JSON 转 对象
     **/
    public static <T> T jsonToObject(String json, Class<T> clazz) {
        if (StringUtils.isNotBlank(json)) {
            try {
                return gson.fromJson(json, clazz);
            } catch (Exception e) {
                System.err.println("数据：{" + json + "} 转对象失败");
            }
        }
        return null;
    }

    /**
     * @Description: json数据 转 pojo对象list
     */
    public static <T> List<T> jsonToList(String json, Class<T> clazz) {
        if (StringUtils.isNotBlank(json)) {
            try {
                Gson gson = new Gson();
                List<T> list = new ArrayList<T>();
                JsonArray array = new JsonParser().parse(json).getAsJsonArray();
                for (final JsonElement elem : array) {
                    list.add(gson.fromJson(elem, clazz));
                }
                return list;
            } catch (Exception e) {
                System.err.println("数据：" + json + " 转集合失败");
            }
        }
        return null;
    }

    /**
     * @description: 对象 转 对象
     **/
    public static <T> T objectToObject(Object obj, Class<T> clazz) {
        if (Objects.nonNull(obj)) {
            return jsonToObject(objectToJSON(obj), clazz);
        }
        return null;
    }

    /**
     * @Description: json数据 转 pojo对象list
     */
    public static <T> List<T> listToList(List list, Class<T> clazz) {
        if (Objects.nonNull(list) && !list.isEmpty()) {
            try {
                return jsonToList(objectToJSON(list), clazz);
            } catch (Exception e) {
                System.err.println("集合：" + list + " 转集合失败");
            }
        }
        return null;
    }

    /**
     * 转成map的
     *
     * @param json
     * @return
     */
    public static <T> Map<String, T> gsonToMaps(String json) {
        Map<String, T> map = null;
        if (gson != null) {
            map = gson.fromJson(json, new TypeToken<Map<String, T>>() {
            }.getType());
        }
        return map;
    }


    /**
     * @description: Object数据可视化
     **/
    public static String parseObject(Object obj) {
        return JSONUtils.parseJSON(objectToJSON(obj));
    }

    /**
     * @method 通过key获取json数据，获取不到给默认值
     */
    public static String getString(String json, String key, String defval) {
        try {
            JsonElement element = new JsonParser().parse(json);
            if (!element.isJsonNull()) {
                String string = element.getAsJsonObject().get(key).toString();
                return StringUtils.isNotBlank(string) ? string : defval;
            }
        } catch (Exception e) {
            System.err.println("数据：{" + json + "} 并非JSON格式");
        }
        return defval;
    }

    /**
     * @method 通过Key获取json数据，获取不到默认给空字符串
     */
    public static String getString(String json, String key) {
        return getString(json, key, "");
    }

    /**
     * @method 通过key获取json数据，获取不到给默认值
     */
    public static Integer getInt(String json, String key, Integer defval) {
        try {
            JsonElement element = new JsonParser().parse(json);
            if (!element.isJsonNull()) {
                String string = element.getAsJsonObject().get(key).toString();
                return StringUtils.isNotBlank(string) ? Integer.parseInt(string) : defval;
            }
        } catch (Exception e) {
            System.err.println("数据：{" + json + "} 并非JSON格式");
        }
        return defval;
    }

    /**
     * @description: 通过Key获取json数据，获取不到给0
     **/
    public static Integer getInt(String json, String key) {
        return getInt(json, key, 0);
    }

    /**
     * @method 通过key获取json数据，获取不到给默认值
     */
    public static Double getDouble(String json, String key, Double defval) {
        try {
            JsonElement element = new JsonParser().parse(json);
            if (!element.isJsonNull()) {
                String string = element.getAsJsonObject().get(key).toString();
                return StringUtils.isNotBlank(string) ? Double.parseDouble(string) : defval;
            }
        } catch (Exception e) {
            System.err.println("数据：{" + json + "} 并非JSON格式");
        }
        return defval;
    }

    /**
     * @description: 通过Key获取json数据，获取不到给0
     **/
    public static Double getDouble(String json, String key) {
        return getDouble(json, key, (double) 0);
    }
}
