package com.jhscale.common.utils;

import com.alibaba.fastjson.JSONObject;
import com.sun.org.apache.xml.internal.security.exceptions.Base64DecodingException;
import com.sun.org.apache.xml.internal.security.utils.Base64;
import org.apache.commons.lang3.StringUtils;

import java.nio.charset.Charset;
import java.util.List;
import java.util.Objects;


/**
 * @author lie_w
 * @title: JSONUtils
 * @projectName common
 * @description: JSON 工具类
 * @date 2020-11-0421:42
 */
public class JSONUtils {

    private JSONUtils() {
    }

    /**
     * @description: 判断数据是不是JSON格式
     **/
    public static boolean isJSON(String json) {
        if (StringUtils.isNotBlank(json)) {
            try {
                JSONObject.parseObject(json);
                return true;
            } catch (Exception e) {
//                System.err.println("数据：{" + json + "} 非JSON格式");
            }
        }
        return false;
    }

    /**
     * @description: 对象 转 JSON
     **/
    public static String objectToJSON(Object obj) {
        if (Objects.nonNull(obj))
            return JSONObject.toJSONString(obj);
        return null;
    }

    /**
     * @description: 字符串转 UTF8 字节流
     **/
    public static byte[] objectToUTF8Bytes(Object obj) {
        String json = objectToJSON(obj);
        return StringUtils.isNotBlank(json) ? json.getBytes(Charset.forName("UTF8")) : null;
    }

    /**
     * @description: 字符串转 UTF8 Base64 字符
     **/
    public static String objectToUTF8Base64Code(Object obj) {
        byte[] jsonbytes = objectToUTF8Bytes(obj);
        return Objects.nonNull(jsonbytes) ? Base64.encode(jsonbytes) : null;
    }

    /**
     * @description: BAS64 UTF8字符串转JSON
     **/
    public static String UTF8Base64CodeToJSON(String base64) {
        if (StringUtils.isNotBlank(base64)) {
            try {
                return new String(Base64.decode(base64), Charset.forName("UTF8"));
            } catch (Base64DecodingException e) {
                System.err.println("数据：{" + base64 + "} 转JSON字符串失败");
            }
        }
        return null;
    }

    /**
     * @description: BAS64 UTF8字符串转 Object
     **/
    public static <T> T UTF8Base64CodeToObject(String base64, Class<T> clazz) {
        String json = UTF8Base64CodeToJSON(base64);
        return StringUtils.isNotBlank(json) ? jsonToObject(json, clazz) : null;
    }

    /**
     * @description: JSON 转 对象
     **/
    public static <T> T jsonToObject(String json, Class<T> clazz) {
        if (StringUtils.isNotBlank(json)) {
            try {
                return JSONObject.parseObject(json, clazz);
            } catch (Exception e) {
                System.err.println("数据：{" + json + "} 转对象失败");
            }
        }
        return null;
    }

    /**
     * @Description: json数据 转 pojo对象list
     */
    public static <T> List<T> jsonToList(String json, Class<T> clazz) {
        if (StringUtils.isNotBlank(json)) {
            try {
                return JSONObject.parseArray(json, clazz);
            } catch (Exception e) {
                System.err.println("数据：" + json + " 转集合失败");
            }
        }
        return null;
    }

    /**
     * @description: 对象 转 对象
     **/
    public static <T> T objectToObject(Object obj, Class<T> clazz) {
        if (Objects.nonNull(obj)) {
            return jsonToObject(objectToJSON(obj), clazz);
        }
        return null;
    }

    /**
     * @description: 对象 转 对象 (对象不存在创建)
     **/
    public static <T> T objectToObjectNullBuild(Object obj, Class<T> clazz) {
        if (Objects.nonNull(obj)) {
            return jsonToObject(objectToJSON(obj), clazz);
        } else {
            try {
                return clazz.newInstance();
            } catch (Exception e) {
                assert clazz != null;
                System.err.println("Class：" + clazz.getName() + " build fail");
                return null;
            }
        }
    }

    /**
     * @Description: json数据 转 pojo对象list
     */
    public static <T> List<T> listToList(List list, Class<T> clazz) {
        if (Objects.nonNull(list) && !list.isEmpty()) {
            try {
                return JSONObject.parseArray(objectToJSON(list), clazz);
            } catch (Exception e) {
                System.err.println("集合：" + list + " 转集合失败");
            }
        }
        return null;
    }

    /**
     * @description: Object数据可视化
     **/
    public static String parseObject(Object obj) {
        return parseJSON(objectToJSON(obj));
    }

    /**
     * @Description json数据可视化
     */
    public static String parseJSON(String jsonData) {
        int level = 0;
        StringBuffer stringBuffer = new StringBuffer();
        for (int i = 0; i < jsonData.length(); i++) {
            char c = jsonData.charAt(i);

            if (level > 0 && '\n' == stringBuffer.charAt(stringBuffer.length() - 1)) {
                stringBuffer.append(getLevelStr(level));
            }

            switch (c) {
                case '{':
                case '[':
                    stringBuffer.append(c + "\n");
                    level++;
                    break;
                case ',':
                    stringBuffer.append(c + "\n");
                    break;
                case '}':
                case ']':
                    stringBuffer.append("\n");
                    level--;
                    stringBuffer.append(getLevelStr(level));
                    stringBuffer.append(c);
                    break;
                default:
                    stringBuffer.append(c);
                    break;
            }
        }
        return stringBuffer.toString();
    }

    private static String getLevelStr(int level) {
        StringBuffer levelStr = new StringBuffer();
        for (int levelI = 0; levelI < level; levelI++) {
            levelStr.append("\t");
        }
        return levelStr.toString();
    }

    /**
     * @method 通过key获取json数据，获取不到给默认值
     */
    public static String getString(String json, String key, String defval) {
        try {
            String string = JSONObject.parseObject(json).getString(key);
            return StringUtils.isNotBlank(string) ? string : defval;
        } catch (Exception e) {
            System.err.println("数据：{" + json + "} 并非JSON格式");
            return null;
        }
    }

    /**
     * @method 通过Key获取json数据，获取不到默认给空字符串
     */
    public static String getString(String json, String key) {
        return getString(json, key, "");
    }

    /**
     * @method 通过key获取json数据，获取不到给默认值
     */
    public static Integer getInt(String json, String key, Integer defval) {
        try {
            Integer integer = JSONObject.parseObject(json).getInteger(key);
            return Objects.nonNull(integer) ? integer : defval;
        } catch (Exception e) {
            System.err.println("数据：{" + json + "} 并非JSON格式");
            return null;
        }
    }

    /**
     * @description: 通过Key获取json数据，获取不到给0
     **/
    public static Integer getInt(String json, String key) {
        return getInt(json, key, 0);
    }

    /**
     * @method 通过key获取json数据，获取不到给默认值
     */
    public static Double getDouble(String json, String key, Double defval) {
        try {
            Double aDouble = JSONObject.parseObject(json).getDouble(key);
            return Objects.nonNull(aDouble) ? aDouble : defval;
        } catch (Exception e) {
            System.err.println("数据：{" + json + "} 并非JSON格式");
            return null;
        }
    }

    /**
     * @description: 通过Key获取json数据，获取不到给0
     **/
    public static Double getDouble(String json, String key) {
        return getDouble(json, key, (double) 0);
    }
}
