package com.jhscale.common.utils;

import com.jhscale.common.exception.GeneralException;
import com.jhscale.common.exception.ProfessionalException;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.lang.reflect.Field;
import java.text.NumberFormat;
import java.util.*;

/**
 * @author 王列
 * @Package com.ysscale.common.utils
 * @Description: ysscale 集合工具类
 * @date 2018/4/2 0:40
 */
public class ListUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(ListUtil.class);

    private ListUtil() {
    }

    /**
     * @Description: 获取两个String集合的交集
     */
    public static List<String> getIntersection(List<String> list1, List<String> list2) {
        //hash比较
        List<String> list3 = new ArrayList<String>();
        Set<String> hashSet = new HashSet<String>();
        for (String s : list1) {
            hashSet.add(s);
        }
        for (String s : list2) {
            if (hashSet.add(s) == false)
                list3.add(s);
        }
        return list3;
    }

    /**
     * @Description 去除list1 中list2的数据
     */
    public static List getDifferenceSet(List list1, List list2) {
        if (list1 == null) {
            list1 = new ArrayList();
        } else {
            list1 = new ArrayList<>(list1);
        }

        if (list2 == null || list2.isEmpty()) {
            return list1;
        } else {
            list2 = new ArrayList(list2);
        }
        list1.removeAll(list2);
        return list1;
    }

    /**
     * @Description 深度克隆list
     */
    public static <T> List<T> deepCopy(List<T> src) {
        List<T> dest = null;
        try {
            ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
            ObjectOutputStream out = new ObjectOutputStream(byteOut);
            out.writeObject(src);

            ByteArrayInputStream byteIn = new ByteArrayInputStream(byteOut.toByteArray());
            ObjectInputStream in = new ObjectInputStream(byteIn);
            dest = (List<T>) in.readObject();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return dest;
    }

    public static List<Integer> idsChange(String idsString) throws ProfessionalException {
        if (StringUtils.isBlank(idsString)) {
            throw new ProfessionalException("编号不能为空");
        }
        List<Integer> ids = new ArrayList<>();
        try {
            String[] idsStr = idsString.split(",");
            if (idsStr != null && idsStr.length != 0) {
                for (String fid : idsStr) {
                    if (StringUtils.isNotBlank(fid)) {
                        ids.add(Integer.parseInt(fid));
                    }
                }
            }
        } catch (Exception e) {
            LOGGER.error(idsString + "转换异常", e);
            throw new ProfessionalException(idsString + "编号无效");
        }
        return ids;
    }

    /**
     * @Description 去重转换
     */
    public static List<Integer> idsDistinctChange(String idsString) throws ProfessionalException {
        if (StringUtils.isBlank(idsString)) {
            throw new ProfessionalException("编号不能为空");
        }
        List<String> idStrings = new ArrayList<>();
        String[] idsStr = idsString.split(",");
        if (idsStr != null && idsStr.length != 0) {
            for (String fid : idsStr) {
                if (StringUtils.isNotBlank(fid) && !idStrings.contains(fid)) {
                    idStrings.add(fid);
                }
            }
        }
        List<Integer> ids = null;
        try {
            ids = JSONUtils.jsonToList(JSONUtils.objectToJSON(idStrings), Integer.class);
        } catch (Exception e) {
            LOGGER.error(idsString + "转换异常", e);
            throw new ProfessionalException(idsString + "编号无效");
        }
        return ids;
    }

    public static List<Date> datesChange(String dates, boolean type) throws GeneralException {
        if (StringUtils.isBlank(dates)) {
            throw new ProfessionalException("时间字符不能为空");
        }

        List<Date> dateList = new ArrayList<>();

        if (type) {
            //,
            List<String> datesString = Arrays.asList(dates.split(","));
            if (datesString != null && !datesString.isEmpty()) {
                for (String s : datesString) {
                    dateList.add(DateUtils.getDateTime(s, DateUtils.YYYYMMDD));
                }
            }
        } else {
            //-
            List<String> datesString = Arrays.asList(dates.split("-"));
            if (datesString == null || datesString.isEmpty() || datesString.size() != 2) {
                throw new ProfessionalException("给定时间区间不正确[" + dates + "]");
            }
            Date dateStart = DateUtils.getDateTime(datesString.get(0), DateUtils.YYYYMMDD);
            Date dateEnd = DateUtils.getDateTime(datesString.get(1), DateUtils.YYYYMMDD);
            dateList = DateUtils.getMonthBetweenDate(dateStart, dateEnd);
        }
        return dateList;
    }

    /**
     * @Description 手机号码转换
     */
    public static List<String> mobilesChange(String mobiles) throws ProfessionalException {
        if (StringUtils.isBlank(mobiles)) {
            throw new ProfessionalException("手机号码不能为空");
        }
        return Arrays.asList(mobiles.split(","));
    }


    /**
     * 将一个list均分成n个list,主要通过偏移量来实现的
     *
     * @param source
     * @return
     */
    public static <T> List<List<T>> averageAssign(List<T> source, int n) {
        List<List<T>> result = new ArrayList<List<T>>();
        int remaider = source.size() % n;  //(先计算出余数)
        int number = source.size() / n;  //然后是商
        int offset = 0;//偏移量
        for (int i = 0; i < n; i++) {
            List<T> value = null;
            if (remaider > 0) {
                value = source.subList(i * number + offset, (i + 1) * number + offset + 1);
                remaider--;
                offset++;
            } else {
                value = source.subList(i * number + offset, (i + 1) * number + offset);
            }
            result.add(value);
        }
        return result;
    }

    /**
     * 对list的元素按照多个属性名称排序,
     * list元素的属性可以是数字（byte、short、int、long、float、double等，支持正数、负数、0）、char、String、java.util.Date
     *
     * @param list
     * @param sortNameArr list元素的属性名称
     * @param isAsc       true升序，false降序
     */
    public static <E> void sort(List<E> list, final boolean isAsc, final String... sortNameArr) {
        Collections.sort(list, new Comparator<E>() {

            public int compare(E a, E b) {
                int ret = 0;
                try {
                    for (int i = 0; i < sortNameArr.length; i++) {
                        ret = ListUtil.compareObject(sortNameArr[i], isAsc, a, b);
                        if (0 != ret) {
                            break;
                        }
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
                return ret;
            }
        });
    }

    /**
     * 对2个对象按照指定属性名称进行排序
     *
     * @param sortname 属性名称
     * @param isAsc    true升序，false降序
     * @param a
     * @param b
     * @return
     * @throws Exception
     */
    private static <E> int compareObject(final String sortname, final boolean isAsc, E a, E b) throws Exception {
        int ret;
        Object value1 = ListUtil.forceGetFieldValue(a, sortname);
        Object value2 = ListUtil.forceGetFieldValue(b, sortname);
        String str1 = value1.toString();
        String str2 = value2.toString();
        if (value1 instanceof Number && value2 instanceof Number) {
            int maxlen = Math.max(str1.length(), str2.length());
            str1 = ListUtil.addZero2Str((Number) value1, maxlen);
            str2 = ListUtil.addZero2Str((Number) value2, maxlen);
        } else if (value1 instanceof Date && value2 instanceof Date) {
            long time1 = ((Date) value1).getTime();
            long time2 = ((Date) value2).getTime();
            int maxlen = Long.toString(Math.max(time1, time2)).length();
            str1 = ListUtil.addZero2Str(time1, maxlen);
            str2 = ListUtil.addZero2Str(time2, maxlen);
        }
        if (isAsc) {
            ret = str1.compareTo(str2);
        } else {
            ret = str2.compareTo(str1);
        }
        return ret;
    }

    /**
     * 给数字对象按照指定长度在左侧补0.
     * <p>
     * 使用案例: addZero2Str(11,4) 返回 "0011", addZero2Str(-18,6)返回 "-000018"
     *
     * @param numObj 数字对象
     * @param length 指定的长度
     * @return
     */
    public static String addZero2Str(Number numObj, int length) {
        NumberFormat nf = NumberFormat.getInstance();
        // 设置是否使用分组
        nf.setGroupingUsed(false);
        // 设置最大整数位数
        nf.setMaximumIntegerDigits(length);
        // 设置最小整数位数
        nf.setMinimumIntegerDigits(length);
        return nf.format(numObj);
    }

    /**
     * 获取指定对象的指定属性值（去除private,protected的限制）
     *
     * @param obj       属性名称所在的对象
     * @param fieldName 属性名称
     * @return
     * @throws Exception
     */
    public static Object forceGetFieldValue(Object obj, String fieldName) throws Exception {
        Field field = obj.getClass().getDeclaredField(fieldName);
        Object object = null;
        boolean accessible = field.isAccessible();
        if (!accessible) {
            // 如果是private,protected修饰的属性，需要修改为可以访问的
            field.setAccessible(true);
            object = field.get(obj);
            // 还原private,protected属性的访问性质
            field.setAccessible(accessible);
            return object;
        }
        object = field.get(obj);
        return object;
    }
}
