package com.jhscale.common.utils;

import java.time.YearMonth;

/**
 * @author lie_w
 * @title: LongKidUtils
 * @projectName Rely-On-Utils
 * @description: TODO
 * @date 2020/3/2316:11
 */
public class LongKidUtils {

    // ==============================Fields===========================================
    /**
     * 开始时间截 (2015-01-01)
     */
    private static final long TWEPOCH = 1420041600000L;

    /**
     * 机器id所占的位数
     */
    private static final long WORKERIDBITS = 5L;

    /**
     * 数据标识id所占的位数
     */
    private static final long DATACENTERIDBITS = 5L;

    /**
     * 支持的最大机器id，结果是31 (这个移位算法可以很快的计算出几位二进制数所能表示的最大十进制数)
     */
    private static final long MAXWORKERID = -1L ^ (-1L << WORKERIDBITS);

    /**
     * 支持的最大数据标识id，结果是31
     */
    private static final long MAXDATACENTERID = -1L ^ (-1L << DATACENTERIDBITS);

    /**
     * 序列在id中占的位数
     */
    private static final long SEQUENCEBITS = 12L;

    /**
     * 机器ID向左移12位
     */
    private static final long WORKERIDSHIFT = SEQUENCEBITS;

    /**
     * 数据标识id向左移17位(12+5)
     */
    private static final long DATACENTERIDSHIFT = SEQUENCEBITS + WORKERIDBITS;

    /**
     * 时间截向左移22位(5+5+12)
     */
    private static final long TIMESTAMPLEFTSHIFT = SEQUENCEBITS + WORKERIDBITS + DATACENTERIDBITS;

    /**
     * 生成序列的掩码，这里为4095 (0b111111111111=0xfff=4095)
     */
    private static final long SEQUENCEMASK = -1L ^ (-1L << SEQUENCEBITS);

    /**
     * 工作机器ID(0~31)
     */
    private static long WORKERID = 29;

    /**
     * 数据中心ID(0~31)
     */
    private static long DATACENTERID = 24;

    /**
     * 毫秒内序列(0~4095)
     */
    private static long SEQUENCE = 0L;

    /**
     * 上次生成ID的时间截
     */
    private static long LASTTIMESTAMP = -1L;

    /**
     * 私有化构造器，防止初始化
     */
    private LongKidUtils() {
    }

    // ==============================Methods==========================================

    /**
     * 获得下一个ID (该方法是线程安全的)
     *
     * @return SnowflakeId
     */
    public static synchronized long nextId() {
        long timestamp = timeGen();

        // 如果当前时间小于上一次ID生成的时间戳，说明系统时钟回退过这个时候应当抛出异常
        if (timestamp < LASTTIMESTAMP) {
            throw new RuntimeException(
                    String.format(
                            "Clock moved backwards.  Refusing to generate id for %d milliseconds",
                            LASTTIMESTAMP - timestamp));
        }

        // 如果是同一时间生成的，则进行毫秒内序列
        if (LASTTIMESTAMP == timestamp) {
            SEQUENCE = (SEQUENCE + 1) & SEQUENCEMASK;
            // 毫秒内序列溢出
            if (SEQUENCE == 0) {
                // 阻塞到下一个毫秒,获得新的时间戳
                timestamp = tilNextMillis(LASTTIMESTAMP);
            }
        }
        // 时间戳改变，毫秒内序列重置
        else {
            SEQUENCE = 0L;
        }

        // 上次生成ID的时间截
        LASTTIMESTAMP = timestamp;

        // 移位并通过或运算拼到一起组成64位的ID
        Long Id =
                ((timestamp - TWEPOCH) << TIMESTAMPLEFTSHIFT)
                        | (DATACENTERID << DATACENTERIDSHIFT)
                        | (WORKERID << WORKERIDSHIFT)
                        | SEQUENCE;
        return Long.parseLong(YearMonth.now().toString().replace("-", "") + Id.toString().substring(5, Id.toString().length()));
    }

    /**
     * 获得下一个ID (该方法是线程安全的)
     *
     * @return SnowflakeId
     */
    public static synchronized String nextIdStr() {
        long timestamp = timeGen();

        // 如果当前时间小于上一次ID生成的时间戳，说明系统时钟回退过这个时候应当抛出异常
        if (timestamp < LASTTIMESTAMP) {
            throw new RuntimeException(
                    String.format(
                            "Clock moved backwards.  Refusing to generate id for %d milliseconds",
                            LASTTIMESTAMP - timestamp));
        }

        // 如果是同一时间生成的，则进行毫秒内序列
        if (LASTTIMESTAMP == timestamp) {
            SEQUENCE = (SEQUENCE + 1) & SEQUENCEMASK;
            // 毫秒内序列溢出
            if (SEQUENCE == 0) {
                // 阻塞到下一个毫秒,获得新的时间戳
                timestamp = tilNextMillis(LASTTIMESTAMP);
            }
        }
        // 时间戳改变，毫秒内序列重置
        else {
            SEQUENCE = 0L;
        }

        // 上次生成ID的时间截
        LASTTIMESTAMP = timestamp;

        // 移位并通过或运算拼到一起组成64位的ID
        Long Id =
                ((timestamp - TWEPOCH) << TIMESTAMPLEFTSHIFT)
                        | (DATACENTERID << DATACENTERIDSHIFT)
                        | (WORKERID << WORKERIDSHIFT)
                        | SEQUENCE;
        return YearMonth.now().toString().replace("-", "") + Id.toString().substring(5, Id.toString().length());
    }

    /**
     * 阻塞到下一个毫秒，直到获得新的时间戳
     *
     * @param lastTimestamp 上次生成ID的时间截
     * @return 当前时间戳
     */
    protected static long tilNextMillis(long lastTimestamp) {
        long timestamp = timeGen();
        while (timestamp <= lastTimestamp) {
            timestamp = timeGen();
        }
        return timestamp;
    }

    /**
     * 返回以毫秒为单位的当前时间
     *
     * @return 当前时间(毫秒)
     */
    protected static long timeGen() {
        return System.currentTimeMillis();
    }

}
