package com.jhscale.common.utils;

public class MathUtils {

    private MathUtils() {
    }

    /**
     * @description: 平均值
     **/
    public static float average(float[] a) {
        return sum(a) / a.length;
    }

    /**
     * @description: 求和
     **/
    public static float sum(float[] a) {
        float sum = 0;
        for (int i = 0; i < a.length; i++) {
            sum += a[i];
        }
        return sum;
    }

    /**
     * 数据归一化，缩小数据的误差
     *
     * @param a 传入参数
     * @return 结果
     */
    public static float[] normalize(float[] a) {
        float sum = sum(a);
        if (sum == 0) return a;
        for (int i = 0; i < a.length; i++) {
            a[i] = a[i] / sum;
        }
        return a;
    }

    /**
     * 计算数据的平均值
     *
     * @param array 二维数组
     * @return 一位数组平均值
     */
    public static float[] calculateColumnAverages(float[][] array) {
        int numRows = array.length;
        int numCols = array[0].length;
        float[] averages = new float[numCols];
        for (int j = 0; j < numCols; j++) {
            float sum = 0.0f;
            for (int i = 0; i < numRows; i++) {
                sum += array[i][j];
            }
            averages[j] = sum / numRows;
        }
        return averages;
    }

    /**
     * @description: 向target 添加 调整平均值(叠加)
     **/
    public static float[] calculateColumnAverages(float[] target, int weight, float[] source) {
        float[] result = null;
        if (target == null) {
            result = source;
        } else {
            int length = Math.max(target.length, source.length);
            result = new float[length];
            for (int i = 0; i < length; i++) {
                float x = 0;
                try {
                    x = target[i];
                } catch (Exception e) {
                }
                float y = 0;
                try {
                    y = source[i];
                } catch (Exception e) {
                }
                result[i] = (x * weight + y) / (weight + 1);
            }
        }
        return result;
    }

    /**
     * @description: 向target 添加 调整平均值（叠减）
     **/
    public static float[] calculateColumnAverages(float[] target, int target_weight, float[] source, int source_weight) {
        float[] result = null;
        if (target == null) {
            result = source;
        } else {
            int length = Math.max(target.length, source.length);
            result = new float[length];
            for (int i = 0; i < length; i++) {
                float x = 0;
                try {
                    x = target[i];
                } catch (Exception e) {
                }
                float y = 0;
                try {
                    y = source[i];
                } catch (Exception e) {
                }
                result[i] = (x * target_weight - y * source_weight) / (target_weight - source_weight);
            }
        }
        return result;
    }

    /**
     * @description: 欧几里得距离 欧式距离
     * 欧几里得距离(Euclidean Distance)：计算两个向量之间的欧几里得距离，即两点之间的直线距离。
     * 公式：d = sqrt((x2 - x1)^2 + (y2 - y1)^2)
     **/
    public static float calculate0(float[] vector1, float[] vector2) {
        float sum = 0;
        for (int i = 0; i < vector1.length; i++) {
            sum += Math.pow(vector1[i] - vector2[i], 2);
        }
        return (float) Math.sqrt(sum);
    }

    /**
     * @description: 曼哈顿距离
     * 曼哈顿距离(Manhattan Distance)：计算两个向量之间的曼哈顿距离，即两点之间的城市街区距离。
     * 公式：d = |x2 - x1| + |y2 - y1|
     **/
    public static float calculate1(float[] vector1, float[] vector2) {
        float sum = 0;
        for (int i = 0; i < vector1.length; i++) {
            sum += Math.abs(vector1[i] - vector2[i]);
        }
        return sum;
    }

    /**
     * @description: 切比雪夫距离
     * 切比雪夫距离(Chebyshev Distance)：计算两个向量之间的切比雪夫距离，即两点之间的棋盘距离。
     * 公式：d = max(|x2 - x1|, |y2 - y1|)
     **/
    public static float calculate2(float[] vector1, float[] vector2) {
        float max = 0;
        for (int i = 0; i < vector1.length; i++) {
            float diff = Math.abs(vector1[i] - vector2[i]);
            if (diff > max) {
                max = diff;
            }
        }
        return max;
    }

    /**
     * @description: 余弦相似度
     * 余弦相似度(Cosine Similarity)：计算两个向量之间的余弦相似度，即两个向量的夹角的余弦值。
     * 公式：cosθ = A·B / (||A|| ||B||)
     **/
    public static float calculate3(float[] vector1, float[] vector2) {
        float dotProduct = 0;
        float normA = 0;
        float normB = 0;
        for (int i = 0; i < vector1.length; i++) {
            dotProduct += vector1[i] * vector2[i];
            normA += Math.pow(vector1[i], 2);
            normB += Math.pow(vector2[i], 2);
        }
        return dotProduct / (float) (Math.sqrt(normA) * Math.sqrt(normB));
    }

    /**
     * @description: float 开方
     **/
    public static float calculate31(float[] feature) {
        float norm = 0;
        for (int i = 0; i < feature.length; i++) {
            norm += feature[i] * feature[i];
        }
        return norm;
    }

    /**
     * @description: 皮尔逊相关系数
     * 皮尔逊相关系数(Pearson Correlation Coefficient)：计算两个向量之间的皮尔逊相关系数，即两个向量之间的线性相关性。
     * 公式：r = (nΣxy - ΣxΣy) / sqrt((nΣx^2 - (Σx)^2)(nΣy^2 - (Σy)^2))
     **/
    public static float calculate4(float[] vector1, float[] vector2) {
        float mean1 = average(vector1);
        float mean2 = average(vector2);
        float sumXY = 0;
        float sumX = 0;
        float sumY = 0;
        float sumXSquare = 0;
        float sumYSquare = 0;
        int n = vector1.length;
        for (int i = 0; i < n; i++) {
            sumXY += (vector1[i] - mean1) * (vector2[i] - mean2);
            sumX += vector1[i];
            sumY += vector2[i];
            sumXSquare += Math.pow(vector1[i], 2);
            sumYSquare += Math.pow(vector2[i], 2);
        }
        return (n * sumXY - sumX * sumY) / (float) (Math.sqrt((n * sumXSquare - Math.pow(sumX, 2)) * (n * sumYSquare - Math.pow(sumY, 2))));
    }
}
