package com.jhscale.common.utils;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.imageio.IIOImage;
import javax.imageio.ImageIO;
import javax.imageio.ImageWriteParam;
import javax.imageio.ImageWriter;
import javax.imageio.stream.FileImageOutputStream;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.*;
import java.util.Iterator;

/**
 * @author wang lie
 * @title: PictureUtils
 * @projectName common
 * @description: 图片处理工具类
 * @date 2023/1/2317:30
 */
public class PictureUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(FileUtils.class);

    private PictureUtils() {
    }

    /**
     * @description: 图片写操作
     **/
    public static boolean write(BufferedImage source, String formatName, File file) {
        if (StringUtils.isNotBlank(formatName) && file != null && !file.isDirectory()) {
            Iterator<ImageWriter> iter = ImageIO.getImageWritersByFormatName(formatName);
            if (iter.hasNext()) {
                ImageWriter writer = iter.next();
                ImageWriteParam param = writer.getDefaultWriteParam();

                param.setCompressionMode(ImageWriteParam.MODE_EXPLICIT);
                param.setCompressionQuality(1f);

                FileImageOutputStream out = null;
                try {
                    if (!file.exists()) file.createNewFile();
                    out = new FileImageOutputStream(file);
                    writer.setOutput(out);
                    writer.write(null, new IIOImage(source, null, null), param);
                    writer.dispose();
                    return true;
                } catch (Exception ex) {
                    System.err.println("BufferedImage Write 异常");
                } finally {
                    if (out != null) {
                        try {
                            out.close();
                        } catch (IOException e) {
                        }
                    }
                }
            }
        }
        return false;
    }

    /**
     * @description: image 大小
     **/
    public static int size(BufferedImage image, String type) {
        return image2InputStream(image, type).getLength();
    }

    /**
     * @description: image 转 inputStream
     **/
    public static BufferedImageInputStream image2InputStream(BufferedImage image, String type) {
        ByteArrayOutputStream bos = null;
        try {
            bos = new ByteArrayOutputStream();
            ImageIO.write(image, type, bos);
            return new BufferedImageInputStream(new ByteArrayInputStream(bos.toByteArray()), bos.size());
        } catch (IOException e) {
            LOGGER.error("获取文件大小失败：{}", e.getMessage(), e);
            return new BufferedImageInputStream(-1);
        } finally {
            if (bos != null) {
                try {
                    bos.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    public static class BufferedImageInputStream {
        /**
         * @description: 输入流
         **/
        private InputStream inputStream;

        /**
         * @description:
         **/
        private int length;

        public BufferedImageInputStream() {
        }

        public BufferedImageInputStream(InputStream inputStream, int length) {
            this.inputStream = inputStream;
            this.length = length;
        }

        public BufferedImageInputStream(int length) {
            this.length = length;
        }

        public InputStream getInputStream() {
            return inputStream;
        }

        public void setInputStream(InputStream inputStream) {
            this.inputStream = inputStream;
        }

        public int getLength() {
            return length;
        }

        public void setLength(int length) {
            this.length = length;
        }
    }

    /**
     * @description: 图片比例缩放
     **/
    public static BufferedImage compress(BufferedImage source, double ratio) {
        return compress(source, 0, 0, ratio);
    }

    /**
     * @description: 指定宽度高度缩放
     **/
    public static BufferedImage compress(BufferedImage source, int width, int height) {
        return compress(source, width, height, 0);
    }

    /**
     * @param source 源图片文件
     * @param width  图片宽度
     * @param height 图片高度
     * @param rate   缩放比例
     * @description: 图片压缩处理
     **/
    public static BufferedImage compress(BufferedImage source, int width, int height, double rate) {
        if (rate > 0) {
            width = (int) (source.getWidth() * rate);
            height = (int) (source.getHeight() * rate);
        }
        BufferedImage target = new BufferedImage(width, height, source.getType());
        // Graphics对象封装了 Java 支持的基本渲染操作所需的状态信息,可以理解为是一支画笔
        Graphics graphics = target.getGraphics();
        // 绘制图像  getScaledInstance表示创建此图像的缩放版本，返回一个新的缩放版本Image,按指定的width,height呈现图像
        // Image.SCALE_SMOOTH,选择图像平滑度比缩放速度具有更高优先级的图像缩放算法。
        graphics.drawImage(source.getScaledInstance(width, height, Image.SCALE_SMOOTH), 0, 0, null);
        //处理此图形上下文并释放它正在使用的任何系统资源。 调用dispose后无法使用Graphics对象。
        graphics.dispose();
        return target;
    }

    /**
     * @param source 源图片文件
     * @param angel  旋转角度
     * @description: 图片旋转
     **/
    public static BufferedImage rotate(BufferedImage source, int angel) {
        // 计算旋转后图片的尺寸
        // System.out.println("Rotate Image...");
        Rectangle rect_des = CalcRotatedSize(new Rectangle(new Dimension(
                source.getWidth(), source.getHeight())), angel);
        BufferedImage targetImage = new BufferedImage(rect_des.width, rect_des.height,
                BufferedImage.TYPE_INT_RGB);
        Graphics2D g2 = targetImage.createGraphics();
        // 进行转换
        g2.translate((rect_des.width - source.getWidth()) / 2,
                (rect_des.height - source.getHeight()) / 2);
        g2.rotate(Math.toRadians(angel), source.getWidth() / 2, source.getHeight() / 2);
        g2.drawImage(source, null, null);
        // System.out.println("Rotate Image Success.");
        return targetImage;
    }

    /**
     * @description: 图片裁剪
     **/
    public static BufferedImage cut(BufferedImage source, int x, int y, int width, int height) {
        return source.getSubimage(x, y, width, height);
    }

    /**
     * @param source   源图片文件
     * @param text     水印文本
     * @param position 方位 0-左上 1-右上 2-左下 3-右下
     * @description: 添加水印(方位添加)
     **/
    public static BufferedImage water_mark_position(BufferedImage source, String text, int position) {
        int width = source.getWidth();
        int height = source.getHeight();

        // 字体颜色
        String FONT_NAME = "微软雅黑";
        // 字体样式
        int FONT_STYLE = Font.BOLD;
        // 字体大小
        int FONT_SIZE = (width + height) / 40;
        // 字体颜色
        Color FONT_COLOR = Color.black;
        // 透明度
        float ALPHA = 0.3F;
        // 旋转角度
        float ROTATE = 0.0F;

        // 创建图片缓存对象
        BufferedImage target = new BufferedImage(width, height, source.getType());
        Graphics2D graphics = target.createGraphics();
        graphics.drawImage(source, 0, 0, width, height, null);
        graphics.setColor(FONT_COLOR);
        graphics.setFont(new Font(FONT_NAME, FONT_STYLE, FONT_SIZE));
        graphics.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_ATOP, ALPHA));
        graphics.rotate(ROTATE);
        switch (position) {
            case 0:
                graphics.drawString(text, 0, FONT_SIZE);
                break;
            case 1:
                graphics.drawString(text, width - textWidth(text, FONT_SIZE), FONT_SIZE);
                break;
            case 2:
                graphics.drawString(text, 0, height);
                break;
            case 3:
                graphics.drawString(text, width - textWidth(text, FONT_SIZE), height);
                break;
        }
        graphics.dispose();
        return target;
    }


    /**
     * @param source 源图片文件
     * @param text   水印文本
     * @param rotate 旋转角度
     * @description: 添加水印(斜满图)
     **/
    public static BufferedImage water_mark_oblique(BufferedImage source, String text, float rotate) {
        int width = source.getWidth();
        int height = source.getHeight();

        // 字体颜色
        String FONT_NAME = "微软雅黑";
        // 字体样式
        int FONT_STYLE = Font.BOLD;
        // 字体大小
        int FONT_SIZE = (width + height) / 40;
        // 字体颜色
        Color FONT_COLOR = Color.black;
        // 透明度
        float ALPHA = 0.3F;
        // 旋转角度
        float ROTATE = rotate;

        // 创建图片缓存对象
        BufferedImage target = new BufferedImage(width, height, source.getType());
        Graphics2D graphics = target.createGraphics();
        graphics.drawImage(source, 0, 0, width, height, null);
        graphics.setColor(FONT_COLOR);
        graphics.setFont(new Font(FONT_NAME, FONT_STYLE, FONT_SIZE));
        graphics.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_ATOP, ALPHA));
        graphics.rotate(ROTATE);
        // 间隔
        int split = FONT_SIZE * 2;
        // 文字占用的宽度
        int xWidth = textWidth(text, FONT_SIZE);
        // x,y可以绘制的数量,多加一个补充空白
        int xCanNum = width / xWidth + 1;
        int yCanNum = height / FONT_SIZE + 1;
        for (int i = 1; i <= yCanNum; i++) {
            int y = FONT_SIZE * i + split * i;
            for (int j = 0; j < xCanNum; j++) {
                int x = xWidth * j + split * j;
                graphics.drawString(text, x, y - (FONT_SIZE + split) * j);
            }
        }
        return target;
    }

    /**
     * 计算旋转后的图片
     *
     * @param src   被旋转的图片
     * @param angel 旋转角度
     * @return 旋转后的图片
     */
    private static Rectangle CalcRotatedSize(Rectangle src, int angel) {
        // 如果旋转的角度大于90度做相应的转换
        if (angel >= 90) {
            if (angel / 90 % 2 == 1) {
                int temp = src.height;
                src.height = src.width;
                src.width = temp;
            }
            angel = angel % 90;
        }

        double r = Math.sqrt(src.height * src.height + src.width * src.width) / 2;
        double len = 2 * Math.sin(Math.toRadians(angel) / 2) * r;
        double angel_alpha = (Math.PI - Math.toRadians(angel)) / 2;
        double angel_dalta_width = Math.atan((double) src.height / src.width);
        double angel_dalta_height = Math.atan((double) src.width / src.height);

        int len_dalta_width = (int) (len * Math.cos(Math.PI - angel_alpha
                - angel_dalta_width));
        int len_dalta_height = (int) (len * Math.cos(Math.PI - angel_alpha
                - angel_dalta_height));
        int des_width = src.width + len_dalta_width * 2;
        int des_height = src.height + len_dalta_height * 2;
        return new Rectangle(new Dimension(des_width, des_height));
    }

    /**
     * @param text     字符串
     * @param fontSize 文字大小
     * @description: 获取字符串占用的宽度
     **/
    private static int textWidth(String text, int fontSize) {
        char[] chars = text.toCharArray();
        fontSize = fontSize % 2 == 0 ? fontSize : (fontSize + 1);
        int fontSize2 = fontSize / 2 + 3;
        int width = 0;
        for (char c : chars) {
            int len = String.valueOf(c).getBytes().length;
            // 汉字为3,其余1
            // 可能还有一些特殊字符占用2等等,统统计为汉字
            if (len != 1) {
                width += fontSize;
            } else {
                width += fontSize2;
            }
        }
        return width;
    }

}
