package com.jhscale.common.utils;

import com.jhscale.common.model.simple.RLocation;
import org.apache.commons.lang3.StringUtils;

import java.io.*;
import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.*;

/**
 * @author lie_w
 * @title: SystemtUtils
 * @projectName common
 * @description: TODO
 * @date 2022/5/710:21
 */
public class SystemtUtils {

    private static final String WIN_PING_FORMAT = "ping %s";
    private static final String CENTOS_PING_FROMAT = "ping %s -w %s";

    /**
     * @Description: 判断系统是不是WINDOWS
     */
    public static boolean isWindows() {
        return System.getProperties().getProperty("os.name").toUpperCase().indexOf("WINDOWS") != -1;
    }

    /**
     * @param url URL
     * @return 网络连接信息
     */
    public static NetWork checkNetwork(String url) {
        Runtime runtime = Runtime.getRuntime();
        Process process;
        InputStream is = null;
        InputStreamReader isr = null;
        BufferedReader br = null;
        NetWork netWork = null;
        try {
            process = runtime.exec(String.format(WIN_PING_FORMAT, url));
            is = process.getInputStream();
            isr = new InputStreamReader(is, "GBK");
            br = new BufferedReader(isr);
            String line = null;
            StringBuffer sb = new StringBuffer();
            while ((line = br.readLine()) != null) {
                sb.append(line);
            }
            if (StringUtils.isNotBlank(sb) && sb.indexOf("TTL") > 0) {
                netWork = new NetWork();
                String connectStr = sb.toString();
                connectStr = connectStr.split("数据包: 已发送 =")[1];
                netWork.setSendCount(Integer.parseInt(connectStr.substring(0, connectStr.indexOf("，")).trim()));
                connectStr = connectStr.split("已接收 =")[1];
                netWork.setAcceptCount(Integer.parseInt(connectStr.substring(0, connectStr.indexOf("，")).trim()));
                connectStr = connectStr.split("丢失 =")[1];
                netWork.setLoseCount(Integer.parseInt(connectStr.substring(0, connectStr.indexOf("(")).trim()));
                connectStr = connectStr.split("最短 =")[1];
                netWork.setMinTime(Integer.parseInt(connectStr.substring(0, connectStr.indexOf("ms")).trim()));
                connectStr = connectStr.split("最长 =")[1];
                netWork.setMaxTime(Integer.parseInt(connectStr.substring(0, connectStr.indexOf("ms")).trim()));
                connectStr = connectStr.split("平均 =")[1];
                netWork.setAvgTime(Integer.parseInt(connectStr.substring(0, connectStr.indexOf("ms")).trim()));
            }
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            close(is, isr, br);
        }
        return netWork;
    }

    /**
     * @description: 检查百度连接状态
     **/
    public static boolean checkBaiduState() {
        return checkNetworkState("baidu.com");
    }

    /**
     * @description: 检查网络状态
     **/
    public static boolean checkNetworkState(String target_name) {
        return checkNetworkState(target_name, 10);
    }

    /**
     * @description: 检查网络状态
     **/
    public static boolean checkNetworkState(String target_name, int out_time) {
        return isWindows() ? checkWinNetworkState(target_name) : checkCentosNetworkState(target_name, out_time);
    }

    /**
     * @param target_name Windows 检查URL连接状态
     * @return
     */
    public static boolean checkWinNetworkState(String target_name) {

        Runtime runtime = Runtime.getRuntime();
        Process process;
        boolean connect = false;
        InputStream is = null;
        InputStreamReader isr = null;
        BufferedReader br = null;
        try {
            process = runtime.exec(String.format(WIN_PING_FORMAT, target_name));
            is = process.getInputStream();
            isr = new InputStreamReader(is, "GBK");
            br = new BufferedReader(isr);
            String line = null;
            while ((line = br.readLine()) != null) {
                if (line.contains("TTL")) return true;
            }
            return false;
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            close(is, isr, br);
        }
        return connect;
    }

    /**
     * @description: Centos 检查网络状态
     **/
    public static boolean checkCentosNetworkState(String target_name, int out_time) {
        Runtime runtime = Runtime.getRuntime();
        BufferedReader bufferedReader = null;
        try {
            Process process = runtime.exec(String.format(CENTOS_PING_FROMAT, target_name, out_time));
            if (null == process)
                return false;
            bufferedReader = new BufferedReader(
                    new InputStreamReader(process.getInputStream(), "UTF-8"));// windows下编码默认是GBK，Linux是UTF-8
            String line = null;
            while (null != (line = bufferedReader.readLine())) {
                System.out.println(line);
                if (line.startsWith("bytes from", 3))
                    return true;
                if (line.startsWith("from"))
                    return true;
            }
            return false;
        } catch (IOException e) {
            e.printStackTrace();
            return false;
        } finally {
            if (bufferedReader != null) {
                try {
                    bufferedReader.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    /**
     * @description: 关闭连接
     **/
    private static void close(InputStream is, InputStreamReader isr, BufferedReader br) {
        try {
            if (is != null)
                is.close();
            if (isr != null)
                isr.close();
            if (br != null)
                br.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * @description: 线程休眠不打印日志
     **/
    public static void sleep(int time) {
        sleep(time, false);
    }

    /**
     * @description: 线程休眠
     **/
    public static void sleep(int time, boolean log) {
        int min = 0;
        do {
            min++;
            try {
                if (min % 10 == 0 && log)
                    System.out.println("休眠[ " + time + "s ]-第[" + min + "]秒");
                Thread.sleep(1000);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        } while (min < time);
    }

    /**
     * @description: 线程休眠不打印日志
     **/
    public static void millis_sleep(long time) {
        millis_sleep(time, false);
    }

    /**
     * @description: 线程休眠
     **/
    public static void millis_sleep(long time, boolean log) {
        int millis = 0;
        do {
            millis++;
            try {
                if (millis % 10000 == 0 && log)
                    System.out.println("休眠[ " + time + "s ]-第[" + millis + "]毫秒");
                Thread.sleep(1);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        } while (millis < time);
    }

    /**
     * @description: 获取局域网IP
     **/
    public static List<NetworkInfo> network_info() {
        List<NetworkInfo> networkInfo = new ArrayList<>();
        try {
            Enumeration<NetworkInterface> nifs = NetworkInterface.getNetworkInterfaces();
            while (nifs.hasMoreElements()) {
                NetworkInterface nif = nifs.nextElement();
                Enumeration<InetAddress> address = nif.getInetAddresses();
                while (address.hasMoreElements()) {
                    InetAddress addr = address.nextElement();
                    if (addr instanceof Inet4Address) {
                        networkInfo.add(new NetworkInfo(nif.getName(), addr));
                    }
                }
            }
        } catch (SocketException e) {
            e.printStackTrace();
        }
        return networkInfo;
    }

    /**
     * @description: 获取本机有效IP
     **/
    public static String lan_ip() {
        String ip = null;
        try {
            List<NetworkInfo> networkInfoList = network_info();
            if (networkInfoList != null && !networkInfoList.isEmpty()) {
                for (NetworkInfo networkInfo : networkInfoList) {
                    String hostAddress = networkInfo.getAddress().getHostAddress();
                    if (!hostAddress.endsWith(".1")) {
                        ip = hostAddress;
                        break;
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return ip;
    }

    /**
     * @description: 局域网路由IP
     **/
    public static String route_ip() {
        String lan_ip = lan_ip();
        String route_ip = null;
        if (StringUtils.isNotBlank(lan_ip)) {
            int index = lan_ip.lastIndexOf(".");
            route_ip = lan_ip.substring(0, index) + ".1";
        }
        return route_ip;
    }

    /**
     * @description: 局域网广播IP
     **/
    public static String route_broadcast() {
        String lan_ip = lan_ip();
        String route_broadcast = null;
        if (StringUtils.isNotBlank(lan_ip)) {
            int index = lan_ip.lastIndexOf(".");
            route_broadcast = lan_ip.substring(0, index) + ".255";
        }
        return route_broadcast;
    }

    /**
     * @description: 获取输入
     **/
    public static String scanner(String description) {
        Scanner scanner = new Scanner(System.in);
        System.out.print(description + ": ");
        return scanner.nextLine();
    }

    /**
     * @description: command 指令
     **/
    public static String command(String... commands) {
        return isWindows() ? win_command(commands) : linux_command(commands);
    }

    public static String command(File directory, String... commands) {
        return isWindows() ? win_command(directory, commands) : linux_command(directory, commands);
    }


    public static String win_command(String... commands) {
        Process process = null;
        BufferedReader reader = null;
        StringBuilder builder = new StringBuilder();
        try {
            List<String> commandList = new ArrayList<>();
            commandList.add("cmd.exe");
            commandList.add("/c");
            commandList.addAll(Arrays.asList(commands));
            ProcessBuilder processBuilder = new ProcessBuilder(commandList.<String>toArray(new String[0]));
            processBuilder.redirectErrorStream(true);
            process = processBuilder.start();
            reader = new BufferedReader(new InputStreamReader(process.getInputStream(), "GBK"));
            String line;
            while ((line = reader.readLine()) != null)
                builder.append(line);
            int exitCode = process.waitFor();
            if (exitCode != 0)
                throw new RuntimeException("Command exited with non-zero status code: " + exitCode);
        } catch (IOException | InterruptedException e) {
            throw new RuntimeException("Failed to execute command: " + e.getMessage());
        } finally {
            if (reader != null)
                try {
                    reader.close();
                } catch (IOException iOException) {
                }
            if (process != null)
                process.destroy();
        }
        return builder.toString();
    }

    public static String win_command(File directory, String... commands) {
        Process process = null;
        BufferedReader reader = null;
        StringBuilder builder = new StringBuilder();
        try {
            List<String> commandList = new ArrayList<>();
            commandList.add("cmd.exe");
            commandList.add("/c");
            commandList.addAll(Arrays.asList(commands));
            ProcessBuilder processBuilder = new ProcessBuilder(new String[0]);
            processBuilder.directory(directory);
            processBuilder.command(commandList.<String>toArray(new String[0]));
            processBuilder.redirectErrorStream(true);
            process = processBuilder.start();
            reader = new BufferedReader(new InputStreamReader(process.getInputStream(), "GBK"));
            String line;
            while ((line = reader.readLine()) != null)
                builder.append(line);
            int exitCode = process.waitFor();
            if (exitCode != 0)
                throw new RuntimeException("Command exited with non-zero status code: " + exitCode);
        } catch (IOException | InterruptedException e) {
            throw new RuntimeException("Failed to execute command: " + e.getMessage());
        } finally {
            if (reader != null)
                try {
                    reader.close();
                } catch (IOException iOException) {
                }
            if (process != null)
                process.destroy();
        }
        return builder.toString();
    }

    public static String linux_command(String... commands) {
        Process process = null;
        BufferedReader reader = null;
        StringBuilder sb = new StringBuilder();
        try {
            List<String> commandList = new ArrayList<>();
            commandList.add("/bin/sh");
            commandList.add("-c");
            commandList.addAll(Arrays.asList(commands));
            ProcessBuilder processBuilder = new ProcessBuilder(commandList.<String>toArray(new String[0]));
            processBuilder.redirectErrorStream(true);
            process = processBuilder.start();
            reader = new BufferedReader(new InputStreamReader(process.getInputStream()));
            String line;
            while ((line = reader.readLine()) != null)
                sb.append(line);
            int exitCode = process.waitFor();
            if (exitCode != 0)
                throw new RuntimeException("Command exited with non-zero status code: " + exitCode);
        } catch (IOException | InterruptedException e) {
            throw new RuntimeException("Failed to execute command: " + e.getMessage());
        } finally {
            if (reader != null)
                try {
                    reader.close();
                } catch (IOException iOException) {
                }
            if (process != null)
                process.destroy();
        }
        return sb.toString();
    }

    public static String linux_command(File directory, String... commands) {
        Process process = null;
        BufferedReader reader = null;
        StringBuilder sb = new StringBuilder();
        try {
            List<String> commandList = new ArrayList<>();
            commandList.add("/bin/sh");
            commandList.add("-c");
            commandList.addAll(Arrays.asList(commands));
            ProcessBuilder processBuilder = new ProcessBuilder(new String[0]);
            processBuilder.directory(directory);
            processBuilder.command(commandList.<String>toArray(new String[0]));
            processBuilder.redirectErrorStream(true);
            process = processBuilder.start();
            reader = new BufferedReader(new InputStreamReader(process.getInputStream()));
            String line;
            while ((line = reader.readLine()) != null)
                sb.append(line);
            int exitCode = process.waitFor();
            if (exitCode != 0)
                throw new RuntimeException("Command exited with non-zero status code: " + exitCode);
        } catch (IOException | InterruptedException e) {
            throw new RuntimeException("Failed to execute command: " + e.getMessage());
        } finally {
            if (reader != null)
                try {
                    reader.close();
                } catch (IOException iOException) {
                }
            if (process != null)
                process.destroy();
        }
        return sb.toString();
    }

    /**
     * @description: IP
     **/
    public static class NetworkInfo {
        // 网卡
        private String card;

        // IP
        private InetAddress address;

        public NetworkInfo() {
        }

        public NetworkInfo(String card, InetAddress address) {
            this.card = card;
            this.address = address;
        }

        public String getCard() {
            return card;
        }

        public void setCard(String card) {
            this.card = card;
        }

        public InetAddress getAddress() {
            return address;
        }

        public void setAddress(InetAddress address) {
            this.address = address;
        }
    }

    /**
     * @description: 网络状态信息
     **/
    public static class NetWork {
        // 已发送次数
        private int sendCount;

        // 已接受次数
        private int acceptCount;

        // 丢失次数
        private int loseCount;

        // 最大连接时间
        private int maxTime;

        // 最小连接时间
        private int minTime;

        // 平均连接时间
        private int avgTime;

        public int getSendCount() {
            return sendCount;
        }

        public void setSendCount(int sendCount) {
            this.sendCount = sendCount;
        }

        public int getAcceptCount() {
            return acceptCount;
        }

        public void setAcceptCount(int acceptCount) {
            this.acceptCount = acceptCount;
        }

        public int getLoseCount() {
            return loseCount;
        }

        public void setLoseCount(int loseCount) {
            this.loseCount = loseCount;
        }

        public int getMaxTime() {
            return maxTime;
        }

        public void setMaxTime(int maxTime) {
            this.maxTime = maxTime;
        }

        public int getMinTime() {
            return minTime;
        }

        public void setMinTime(int minTime) {
            this.minTime = minTime;
        }

        public int getAvgTime() {
            return avgTime;
        }

        public void setAvgTime(int avgTime) {
            this.avgTime = avgTime;
        }
    }

    /**
     * @description: 计算经纬度距离 千米
     * @date: 2024-09-11 13:53:03
     **/
    private static final double EARTH_RADIUS = 6371.0;

    /**
     * @description: 计算距离 千米
     * @date: 2024-08-13 17:13:23
     **/
    public static double haversineKilometreDistance(RLocation r1, RLocation r2) {
        double dLat = Math.toRadians(r2.getLat() - r1.getLat());
        double dLon = Math.toRadians(r2.getLon() - r1.getLon());
        double a = Math.sin(dLat / 2) * Math.sin(dLat / 2) +
                Math.cos(Math.toRadians(r1.getLat())) * Math.cos(Math.toRadians(r2.getLat())) *
                        Math.sin(dLon / 2) * Math.sin(dLon / 2);
        double c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
        return EARTH_RADIUS * c;
    }

    /**
     * @description: 计算距离 米
     * @date: 2024-08-13 17:13:23
     **/
    public static double haversineMetreDistance(RLocation r1, RLocation r2) {
        return haversineKilometreDistance(r1, r2) * 1000;
    }
}
