package com.jhscale.common.utils;

import com.jhscale.common.exception.GeneralException;
import com.jhscale.common.model.inter.JSONModel;
import org.apache.commons.lang3.StringUtils;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * @author lie_w
 * @title: UniqueUtils
 * @projectName common
 * @description: TODO
 * @date 2022/6/918:03
 */
public class UniqueUtils {

    private static final Map<String, BoardStrategy> boardStrategyMap;

    static {
        boardStrategyMap = new HashMap<>();
        boardStrategyMap.put(Board.YSA.getSimple(), new YSA_Board());
        boardStrategyMap.put(Board.YSB.getSimple(), new YSB_Board());
    }

    /**
     * @description: 主板序列号获取
     **/
    public static MainBoard unique(MainBoard board, BoardStrategy otherStrategy) throws GeneralException {
        BoardStrategy boardStrategy = boardStrategyMap.get(board.getBoard());
        String unique = null;
        if (boardStrategy != null) {
            unique = boardStrategy.strategy(board);
        }
        if (StringUtils.isBlank(unique) && otherStrategy != null) {
            unique = otherStrategy.strategy(board);
        }
        board.setUnique(unique);
        return board;
    }

    public interface BoardStrategy {

        /**
         * @description: Unique策略执行
         **/
        String strategy(MainBoard mainBoard) throws GeneralException;
    }

    /**
     * @description: YSA主板（紫竹）
     **/
    public static class YSA_Board implements BoardStrategy {

        /**
         * @param mainBoard
         * @description: Unique策略执行
         */
        @Override
        public String strategy(MainBoard mainBoard) {
            return (StringUtils.isNotBlank(mainBoard.getSerialBoard()) ? mainBoard.getSerialBoard() : "")
                    +
                    (StringUtils.isNotBlank(mainBoard.getLan()) ? mainBoard.getLan() : "");
        }
    }

    /**
     * @description: YSB主板
     **/
    public static class YSB_Board implements BoardStrategy {

        /**
         * @description: Unique策略执行
         */
        @Override
        public String strategy(MainBoard mainBoard) {
            return StringUtils.isNotBlank(mainBoard.getSerialBoard()) ? mainBoard.getSerialBoard() : "";
        }
    }

    /**
     * @description: 主板信息
     **/
    public static class MainBoard implements JSONModel {

        // 生产版本
        private int serial;

        // 主板
        private String board;

        // 主板序列号
        private String serialBoard;

        // 系统类型
        private String sysType;

        // 系统版本
        private String sysVersion;

        // 网卡序列号
        private String lan;

        // 最终设备序列号
        private String unique;

        // 生产日期
        private Date prod = new Date();

        public MainBoard() {
        }

        public MainBoard(String board, String serialBoard, String sysType, String sysVersion, String lan) {
            this.board = board;
            this.serialBoard = serialBoard;
            this.sysType = sysType;
            this.sysVersion = sysVersion;
            this.lan = lan;
        }

        public MainBoard(String board, String serialBoard, String sysType, String sysVersion, String lan, Date prod) {
            this.board = board;
            this.serialBoard = serialBoard;
            this.sysType = sysType;
            this.sysVersion = sysVersion;
            this.lan = lan;
            this.prod = prod;
        }

        public int getSerial() {
            return serial;
        }

        public void setSerial(int serial) {
            this.serial = serial;
        }

        public String getBoard() {
            return board;
        }

        public void setBoard(String board) {
            this.board = board;
        }

        public String getSerialBoard() {
            return serialBoard;
        }

        public void setSerialBoard(String serialBoard) {
            this.serialBoard = serialBoard;
        }

        public String getSysType() {
            return sysType;
        }

        public void setSysType(String sysType) {
            this.sysType = sysType;
        }

        public String getSysVersion() {
            return sysVersion;
        }

        public void setSysVersion(String sysVersion) {
            this.sysVersion = sysVersion;
        }

        public String getLan() {
            return lan;
        }

        public void setLan(String lan) {
            this.lan = lan;
        }

        public String getUnique() {
            return unique;
        }

        public void setUnique(String unique) {
            this.unique = unique;
        }

        public Date getProd() {
            return prod;
        }

        public void setProd(Date prod) {
            this.prod = prod;
        }
    }

    /**
     * @description: 主板类型
     **/
    public enum Board {

        YSA("YSA", ""),
        YSB("YSB", "");

        private String simple;

        private String name;

        Board(String simple, String name) {
            this.simple = simple;
            this.name = name;
        }

        public static Board board(String board) {
            if (StringUtils.isNotBlank(board)) {
                for (Board board1 : Board.values()) {
                    if (board.equals(board1.getSimple())) return board1;
                }
            }
            return null;
        }

        public String getSimple() {
            return simple;
        }

        public String getName() {
            return name;
        }
    }
}
