package com.jhscale.common.utils;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.lang3.StringUtils;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.xml.sax.InputSource;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import java.io.ByteArrayInputStream;
import java.io.StringReader;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.util.*;

/**
 * @author lie_w
 * @title: XMLUtils
 * @projectName common
 * @description: XML 工具类
 * @date 2020-11-0510:28
 */
public class XMLUtils {

    private XMLUtils() {
    }

    /**
     * @description: 判断数据是不是XML格式
     **/
    public static boolean isXML(String xml) {
        try {
            DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder = documentBuilderFactory.newDocumentBuilder();
            builder.parse(new InputSource(new StringReader(xml)));
            return false;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return false;
    }

    /**
     * 将String类型的xml转换成对象
     */
    public static <T> T xmlToObject(String xml, Class<T> clazz) {
        Object xmlObject = null;
        try {
            JAXBContext context = JAXBContext.newInstance(clazz);
            // 进行将Xml转成对象的核心接口
            Unmarshaller unmarshaller = context.createUnmarshaller();
            StringReader sr = new StringReader(xml);
            xmlObject = unmarshaller.unmarshal(sr);
        } catch (JAXBException e) {
            e.printStackTrace();
        }
        return (T) xmlObject;
    }

    /**
     * @description: JSONObject 转 XML内部XML
     **/
    public static String jsonObjectToXML(Object obj) {
        if (Objects.isNull(obj)) return null;
        StringBuilder xml = new StringBuilder();
        if (obj instanceof JSONObject) {
            JSONObject jsonObject = (JSONObject) obj;
            jsonObject.getInnerMap().forEach((k, v) -> {
                if (Objects.nonNull(v)) {
                    xml.append("<").append(k).append(">")
                            .append(objToXML(v))
                            .append("</").append(k).append(">");
                }
            });
        } else if (obj instanceof JSONArray) {
            JSONArray jsonArray = (JSONArray) obj;
            if (jsonArray.size() > 0) {
                jsonArray.forEach(jsonObject -> {
                    if (Objects.nonNull(jsonObject)) {
                        String name = jsonObject.getClass().getSimpleName();
                        name = StringUtil.firstToLowerCase(name);
                        xml.append("<").append(name).append(">")
                                .append(objToXML(jsonObject))
                                .append("</").append(name).append(">");
                    }
                });
            }
        } else {
            xml.append(obj);
        }
        return xml.toString();
    }

    /**
     * @description: Object 转 XML内部XML
     **/
    private static String objToXML(Object obj) {
        if (Objects.isNull(obj)) return null;
        StringBuilder xml = new StringBuilder();
        if (obj instanceof List) {
            List list = (List) obj;
            if (!list.isEmpty()) {
                list.forEach(o -> {
                    if (Objects.nonNull(o)) {
                        String name = o.getClass().getSimpleName();
                        name = StringUtil.firstToLowerCase(name);
                        xml.append("<").append(name).append(">")
                                .append(objToXML(o))
                                .append("</").append(name).append(">");
                    }
                });
            }
        } else if (obj instanceof Map) {
            ((Map) obj).forEach((k, v) -> {
                if (Objects.nonNull(v)) {
                    xml.append("<").append(k).append(">")
                            .append(objToXML(v))
                            .append("</").append(k).append(">");
                }
            });
        } else if (obj instanceof String || obj instanceof Integer
                || obj instanceof Long || obj instanceof Short
                || obj instanceof Double || obj instanceof Boolean
                || obj instanceof BigDecimal || obj instanceof Byte) {
            xml.append(obj.toString());
        } else {
            Field[] fields = obj.getClass().getDeclaredFields();
            if (fields != null && fields.length > 0) {
                for (Field field : fields) {
                    try {
                        field.setAccessible(true);
                        Object o = field.get(obj);
                        if (o != null) {
                            xml.append("<").append(field.getName()).append(">")
                                    .append(objToXML(o))
                                    .append("</").append(field.getName()).append(">");
                        }
                    } catch (IllegalAccessException e) {
                        e.printStackTrace();
                    }
                }
            }
        }
        return xml.toString();
    }

    /**
     * 将Object 中所有字段转 XML字符
     */
    public static String objectAllToXML(Object bean) {
        if (Objects.isNull(bean)) return null;
        StringBuilder xml = new StringBuilder();
        xml.append("<xml>");
        String objToXML = objToXML(bean);
        if (StringUtils.isNotBlank(objToXML)) {
            xml.append(objToXML);
        }
        xml.append("</xml>");
        return xml.toString();
    }

    /**
     * 将Map<String,String>类型的xml转换成对象
     */
    public static String objectToXML(Object bean) {
        if (Objects.isNull(bean)) return null;
        Map map = JSONUtils.objectToObject(bean, Map.class);
        if (Objects.isNull(map) || map.isEmpty()) return null;
        StringBuilder xml = new StringBuilder();
        xml.append("<xml>");
        Iterator var2 = map.entrySet().iterator();

        while (var2.hasNext()) {
            Map.Entry<String, String> entry = (Map.Entry) var2.next();
            String key = entry.getKey();
            String value = entry.getValue();
            if (StringUtils.isNotBlank(value)) {
                xml.append("<").append(key).append(">");
                xml.append(entry.getValue());
                xml.append("</").append(key).append(">");
            }
        }
        xml.append("</xml>");
        return xml.toString();
    }

    /**
     * @description: XML转MAP
     **/
    public static Map<String, Object> xmlToMap(String xml) {
        try {
            SAXReader saxReader = new SAXReader();
            Document document = saxReader.read(new ByteArrayInputStream(xml.getBytes()));
            return Dom2Map(document);
        } catch (DocumentException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * @description: 将Document对象转为Map（String→Document→Map）
     **/
    public static Map<String, Object> Dom2Map(Document doc) {
        Map<String, Object> map = new HashMap<String, Object>();
        if (doc == null)
            return map;
        Element root = doc.getRootElement();
        for (Iterator iterator = root.elementIterator(); iterator.hasNext(); ) {
            Element e = (Element) iterator.next();
            if (e.elements().size() > 0) {
                map.put(e.getName(), Dom2Map(e));
            } else
                map.put(e.getName(), e.getText());
        }
        return map;
    }

    /**
     * @description: 将Element对象转为Map（String→Document→Element→Map）
     **/
    private static Map Dom2Map(Element e) {
        Map map = new HashMap();
        List list = e.elements();
        if (list.size() > 0) {
            for (int i = 0; i < list.size(); i++) {
                Element iter = (Element) list.get(i);
                List mapList = new ArrayList();
                if (iter.elements().size() > 0) {
                    Map m = Dom2Map(iter);
                    if (map.get(iter.getName()) != null) {
                        Object obj = map.get(iter.getName());
                        if (!obj.getClass().getName().equals("java.util.ArrayList")) {
                            mapList = new ArrayList();
                            mapList.add(obj);
                            mapList.add(m);
                        }
                        if (obj.getClass().getName().equals("java.util.ArrayList")) {
                            mapList = (List) obj;
                            mapList.add(m);
                        }
                        map.put(iter.getName(), mapList);
                    } else
                        map.put(iter.getName(), m);
                } else {
                    if (map.get(iter.getName()) != null) {
                        Object obj = map.get(iter.getName());
                        if (!obj.getClass().getName().equals("java.util.ArrayList")) {
                            mapList = new ArrayList();
                            mapList.add(obj);
                            mapList.add(iter.getText());
                        }
                        if (obj.getClass().getName().equals("java.util.ArrayList")) {
                            mapList = (List) obj;
                            mapList.add(iter.getText());
                        }
                        map.put(iter.getName(), mapList);
                    } else
                        map.put(iter.getName(), iter.getText());//公共map resultCode=0
                }
            }
        } else {
            map.put(e.getName(), e.getText());
        }
        return map;
    }
}
