package com.jhscale.meter.call.service.impl;

import com.alibaba.fastjson.JSON;
import com.jhscale.meter.call.agreement.Agreement;
import com.jhscale.meter.call.entity.Request;
import com.jhscale.meter.call.entity.Response;
import com.jhscale.meter.call.link.CallMessenger;
import com.jhscale.meter.call.link.ICallBack;
import com.jhscale.meter.call.link.UpgradeParam;
import com.jhscale.meter.call.model.BCCache;
import com.jhscale.meter.call.service.IPort;
import com.jhscale.meter.call.service.IWeight;
import com.jhscale.meter.em.WeighingMode;
import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.exp.MeterStateEnum;
import com.jhscale.meter.protocol.print.entity.ScheduleState;
import com.jhscale.meter.utils.AConstant;
import com.jhscale.meter.utils.ByteUtils;
import com.jhscale.meter.utils.FileUtils;

import java.io.File;
import java.io.InputStream;
import java.util.Arrays;
import java.util.List;

/**
 * @author Loveven
 * @title: BCPort
 * @projectName jhscale-agreement
 * @description: TODO
 * @date 2019/9/152:05
 */
public class BCPort implements IPort, IWeight {

    private Request request;

    private String cmd;

    public BCPort() {
    }

    public BCPort(Request request) {
        this.request = request;
    }

    public BCPort(String cmd) {
        this.cmd = cmd;
    }

    @Override
    public String assemble() {
        IPort port = new Agreement(request);
        return port.assemble();
    }

    @Override
    public String paese() {
        IPort port = new Agreement(cmd);
        return port.paese();
    }

    /**
     * @param responseStr
     * @description: 切取
     */
    @Override
    public List<String> split(String responseStr) {
        return new Agreement().split(responseStr);
    }

    /**
     * @param request
     * @description: 数据拼装
     */
    @Override
    public String assemble(Request request) {
        return new Agreement(request).assemble();
    }

    /**
     * @param cmd
     * @description: 数据内容解析
     */
    @Override
    public String parse(String cmd) {
        return new Agreement(cmd).paese();
    }

    /**
     * @param weighingMode
     * @description: 设置秤重模式
     */
    @Override
    public String setModel(WeighingMode weighingMode) {
        return new Agreement(new Request("A", weighingMode.getVal())).assemble();
    }

    /**
     * @description: 置零操作
     **/
    @Override
    public String setZero() {
        return new Agreement(new Request("B")).assemble();
    }

    /**
     * @description: 去皮操作
     **/
    @Override
    public String setPeeling() {
        return new Agreement(new Request("C")).assemble();
    }

    /**
     * @param weight
     * @description: 设置皮重
     */
    @Override
    public String setPeeling(String weight) {
        return new Agreement(new Request("I", weight)).assemble();
    }

    /**
     * @description: 关机操作
     **/
    @Override
    public String powerOff() {
        return new Agreement(new Request("H")).assemble();
    }

    /**
     * @description: 读取重量命令
     **/
    @Override
    public String readWeight() {
        return new Agreement(new Request("C")).assemble();
    }

    /**
     * @param responseStr
     * @description: 响应数据解析 同时可以解析秤重
     */
    @Override
    public String parseResponse(String responseStr) {
        return new Agreement(responseStr).paese();
    }

    /**
     * @param cmd
     * @description: 读取重量解析
     */
    @Override
    public String readWeight(String cmd) {
        return new Agreement(cmd).paese();
    }

    /**
     * @description: 读取MAC(蓝牙通讯时支持此操作)
     **/
    @Override
    public String readMac() {
        return new Agreement(new Request("B")).assemble();
    }

    /**
     * @description: 读取版本
     **/
    @Override
    public String readVersion() {
        return new Agreement(new Request("G")).assemble();
    }

    private UpgradeParam param = new UpgradeParam();

    /**
     * @description: 获取升级参数
     **/
    @Override
    public UpgradeParam getUpgradeParam() {
        return param;
    }

    /**
     * @param param
     * @description: 设置升级参数
     */
    @Override
    public void setUpgradeParam(UpgradeParam param) {
        this.param = param;
    }

    // 握手通讯器
    private CallMessenger messenger;
    private static final String VERSION_CHECK = "55555555555555555555555555555555";

    /**
     * @param callMessenger
     * @param callBack
     * @description: 握手
     */
    @Override
    public void shake(CallMessenger callMessenger, ICallBack callBack) throws MeterException {
        if (callMessenger == null || callBack == null) {
            throw new MeterException(MeterStateEnum.升级握手失败);
        }

        BCCache.getInstance().clearCache();
        ICallBack iCallBack = new ICallBack() {
            @Override
            public void response(Response response) {
//                System.out.println("收到通讯器回调信息：" + JSON.toJSONString(response));
                BCCache.getInstance().addCache(response.hexToStr());
            }

            @Override
            public void schedule(ScheduleState scheduleState) {
            }
        };

        int count = 0;
        while (count <= this.param.getShakeTryCount()) {
            String[] split = BCCache.getInstance().getCache().split("\r\n");
            if (split != null && split.length >= 2) {
                String version = split[split.length - 2];
                if (version.startsWith("U")) {
                    this.messenger = callMessenger;
                    if (callBack != null)
                        callBack.response(Response.success(version));
                    return;
                }
            }
            callMessenger.sendCmd(VERSION_CHECK, iCallBack);

            try {
                Thread.sleep(this.param.getShakeSleep());
                count++;
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
        throw new MeterException(MeterStateEnum.升级握手超时);
    }

    /**
     * @param file
     * @description: 文件升级
     */
    @Override
    public void upgrade(File file, ICallBack callBack) throws MeterException {
        long start = System.currentTimeMillis();
        byte[] bytes = FileUtils.upgradeBytes(FileUtils.fileToBytes(file));
        System.out.println(String.format("文件转换耗时：%s", (System.currentTimeMillis() - start)));
        upgrade(bytes, callBack);
    }

    /**
     * @param inputStream
     * @description: 文件流升级
     */
    @Override
    public void upgrade(InputStream inputStream, ICallBack callBack) throws MeterException {
        byte[] bytes = FileUtils.upgradeBytes(FileUtils.inputStreamToBytes(inputStream));
        upgrade(bytes, callBack);
    }

    /**
     * @param bytes
     * @description: 升级
     */
    private void upgrade(byte[] bytes, ICallBack callBack) throws MeterException {
        if (messenger == null)
            throw new MeterException(MeterStateEnum.未完成升级握手);

        System.out.println("Upgrade Starting......");
//        System.out.println("升级文件数据：" + ByteUtils.toHexString(bytes));
        int size = param.getSize();
        int length = bytes.length % size == 0 ? bytes.length / size : bytes.length / size + 1;
        ICallBack iCallBack = new ICallBack() {
            @Override
            public void response(Response response) {
                System.out.println("升级响应收取：" + JSON.toJSONString(response));
            }

            @Override
            public void schedule(ScheduleState scheduleState) {
            }
        };

        for (int i = 0; i < length; i++) {
            int start = i * param.getSize();
            int end = (i + 1) * param.getSize();
            end = end > bytes.length ? bytes.length : end;
//            System.out.println(String.format("i:%s start:%s end:%s length:%s", i, start, end, bytes.length));
            String sendMsg = ByteUtils.toHexString(Arrays.copyOfRange(bytes, start, end));
//            System.out.println("发送数据：" + sendMsg);
            messenger.sendCmd(sendMsg, iCallBack);

            if (callBack != null && i % 256 == 0)
                callBack.schedule(new ScheduleState(i + 1, length));

            try {
                if (param.getShakeSleep() > 0)
                    Thread.sleep(param.getUpgradeSleep());
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
        System.out.println("Upgrade Ending......");
        if (callBack != null)
            callBack.response(Response.success(AConstant.SUCCESS_STR));
    }
}
