package com.jhscale.meter.io;

import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.exp.MeterStateEnum;
import com.jhscale.meter.io.control.ClientControl;
import com.jhscale.meter.io.control.DeviceControl;
import com.jhscale.meter.io.listener.DeviceClientEventListener;
import com.jhscale.meter.io.listener.DeviceDiscoverEventListener;
import com.jhscale.meter.io.listener.SocketClientEventListener;
import com.jhscale.meter.log.JLog;
import com.jhscale.meter.log.PortLogBack;
import com.jhscale.meter.model.device.Device;
import com.jhscale.meter.model.device.InitDeviceEntity;
import com.jhscale.meter.protocol.model.GlobalPara;
import com.jhscale.meter.protocol.model.ReadResponse;
import com.jhscale.meter.utils.ByteUtils;
import com.jhscale.meter.utils.MeterConstant;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.*;

/**
 * @author lie_w
 * @title: PortManager
 * @projectName jh-meter
 * @description: 通讯管理
 * @date 2021/6/412:34
 */
public class PortManager<T extends Device> {

    static final int STATE_NONE = 0;
    static final int STATE_LISTEN = 1;
    static final int STATE_CONNECTING = 2;
    static final int STATE_CONNECTED = 3;

    /**
     * @description: 默认读取字节流长度
     **/
    private Integer readBytesLength = MeterConstant.DEFAULT_READ_STREAM_LENGTH; // 默认读取字节流长度

    private DeviceControl control;

    private DeviceDiscoverEventListener discoverEventListener; // 发现监听
    private Set<T> devices; // 发现列表
    private boolean discovered = false; // false-未开启|true-已开启

    private T device; // 连接的设备
    private DeviceClientEventListener clientEventListener; // 连接监听器

    private InputStream inputStream;
    private OutputStream outputStream;
    /**
     * @description: 多连接管理器
     **/
    private long max_time_out = 10000;// 默认超时时间10s
    protected Thread thread;// 异步线程
    private PortManager server;
    private Map<String, PortManager> clientMaps = Collections.synchronizedMap(new HashMap<>());


    // private Map<String, ListenerThread> moreLinkThread = new ConcurrentHashMap<>(); // 多连接缓冲区
    // private String sign;// 单次标识
    // private U onceLink;


    /**
     * @description: 日志开关
     **/
    private boolean log = false;
    private boolean logBack = false;
    private PortLogBack portLogBack;

    /**
     * @description: 开启状态
     **/
    private boolean status;

    public PortManager() {
    }

    public PortManager(DeviceControl control, DeviceDiscoverEventListener discoverEventListener) throws MeterException {
        this.control = control;
        this.discoverEventListener = discoverEventListener;
        this.devices = new HashSet<T>();
        this.control.initDevice();
    }

    public PortManager(DeviceControl control, T device) throws MeterException {
        this.control = control;
        this.device = device;
        this.control.initDevice();
    }

    public PortManager(DeviceControl control, T device, DeviceClientEventListener clientEventListener) throws MeterException {
        this.control = control;
        this.device = device;
        this.clientEventListener = clientEventListener;
        this.control.initDevice();
    }

    public PortManager(DeviceControl control, T device, PortManager server) throws MeterException {
        this.control = control;
        this.device = device;
        this.clientEventListener = server.clientEventListener;
        this.server = server;
        this.control.initDevice();
    }

    /**
     * @description: 此控制器
     **/
    public PortManager _this() {
        return this;
    }

    /**
     * @description: 通讯初始化
     **/
    public void initPort(InitDeviceEntity entity) throws MeterException {
        this.control.initDevice();
        if (Objects.nonNull(entity)) {
            if (Objects.nonNull(entity.getStreamLength()))
                this.readBytesLength = entity.getStreamLength();
            this.control.initParam(entity);
        }
    }

    /**
     * @description: 开启发现设备
     **/
    public boolean discovery() throws MeterException {
        if (this.devices == null) this.devices = new HashSet<T>();
        this.devices.clear();
        if (this.discoverEventListener != null)
            this.control.addListener(this.discoverEventListener, this.devices);
        this.discovered = this.control.discovery();
        return this.discovered;
    }

    /**
     * @description: 关闭发现设备
     **/
    public boolean cancelDiscovery(boolean clear) throws MeterException {
        if (this.control.cancelDiscovery()) {
            if (clear) {
                if (this.devices == null) this.devices = new HashSet<T>();
                this.devices.clear();
            }
            this.discovered = false;
            return true;
        }
        return false;
    }

    /**
     * @description: 打开端口 (Override) 先打开端口再添加监听
     **/
    public boolean openPort() throws MeterException {
        if (this.device == null) return false;
        if (this.status) return true;
        this.control.openPort(device);
        if (this.clientEventListener != null)
            this.control.addListener(this.clientEventListener, this);
        this.status = this.getInputStream() != null && this.getOutputStream() != null;
        return this.status;
    }

    /**
     * @description: 打开端口 前置添加监听 先添加监听再打开端口
     **/
    public boolean openPortAddEventListenerBefore() throws MeterException {
        if (this.device == null) return false;
        if (this.status) return true;
        if (this.clientEventListener != null)
            this.control.addListener(this.clientEventListener, this);
        this.getControl().openPort(device);
        this.setStatus(true);
        return this.isStatus();
    }

    /**
     * @description: 日志开关
     **/
    public PortManager log(boolean log) {
        this.log = log;
        return this;
    }

    /**
     * @description: 日志回调开关
     **/
    public PortManager logBack(boolean logBack, PortLogBack portLogBack) {
        this.logBack = logBack;
        this.portLogBack = portLogBack;
        return this;
    }

    /**
     * @description: 超时时间
     **/
    public PortManager time_out(long time_out) {
        this.max_time_out = time_out;
        return this;
    }

    /**
     * @description: 写字节流信息
     **/
    public void writeDataImmediately(Vector<Byte> data) throws MeterException {
        this.writeDataImmediately(convertVectorByteToBytes(data));
    }

    /**
     * @description: 流写出
     **/
    public void writeDataImmediately(byte[] bytes) throws MeterException {
        this.writeDataImmediately(bytes, 0, bytes.length);
    }

    public void writeDataImmediatelyWithoutFlush(byte[] bytes) throws MeterException {
        this.write_log(bytes);
        try {
            if (bytes.length > 0) this.getOutputStream().write(bytes, 0, bytes.length);
        } catch (IOException e) {
            JLog.error("PortManager write error：{}", e.getMessage(), e);
            throw new MeterException(MeterStateEnum.流写入错误);
        }
    }

    public void writeDataImmediatelyFlush() throws MeterException {
        try {
            this.getOutputStream().flush();
        } catch (IOException e) {
            JLog.error("PortManager write error：{}", e.getMessage(), e);
            throw new MeterException(MeterStateEnum.流写入错误);
        }
    }

    /**
     * @description: 发送同步响应
     **/
    public byte[] sendWithAccept(byte[] request) throws MeterException {
        long start = System.currentTimeMillis();
        this.writeDataImmediately(request);
        long start1 = System.currentTimeMillis();
        byte[] response = new byte[0];
        while ((System.currentTimeMillis() - start1) <= this.max_time_out) {
            try {
                int available = this.getInputStream().available();
                if (available > 0) {
                    response = this.readData(available);
                    break;
                }
            } catch (IOException e) {
                JLog.error("PortManager write error：{}", e.getMessage(), e);
                throw new MeterException(MeterStateEnum.流读取错误);
            }
        }
        if (this.log) {
            if ((response == null || response.length == 0)) {
                System.out.println(String.format("Max Time Out:[%s]ms,未接到响应信息", this.max_time_out));
            } else {
                System.out.println(String.format("Time Total:[%s]ms，Send:[%s]ms，Accept:[%s]ms",
                        (System.currentTimeMillis() - start), (start1 - start), (System.currentTimeMillis() - start1)));
            }
        }

        if (GlobalPara.getInstance().isRunLog()) {
            if ((response == null || response.length == 0)) {
                System.out.println(String.format("Max Time Out:[%s]ms,未接到响应信息", this.max_time_out));
            } else {
                System.out.println(String.format("Time Total:[%s]ms，Send:[%s]ms，Accept:[%s]ms",
                        (System.currentTimeMillis() - start), (start1 - start), (System.currentTimeMillis() - start1)));
            }
        }
        return response;
    }

    /**
     * @description: 从长度开始写指定长度字节流
     **/
    public void writeDataImmediately(Vector<Byte> data, int offset, int len) throws MeterException {
        this.writeDataImmediately(convertVectorByteToBytes(data), offset, len);
    }

    /**
     * @description: 流写出 (Override)
     **/
    public void writeDataImmediately(byte[] bytes, int offset, int len) throws MeterException {
        this.write_log(bytes);
        if (bytes.length == 0) return;
        byte[] buffer = new byte[2048];
        try {
            while (offset < len) {
                int remaining = len - offset;
                int toWrite = Math.min(remaining, buffer.length);
                System.arraycopy(bytes, offset, buffer, 0, toWrite);
                this.getOutputStream().write(buffer, 0, toWrite);
                offset += toWrite;
            }
            this.getOutputStream().flush();
        } catch (IOException e) {
            e.printStackTrace();
            JLog.error("PortManager write error：{}", e.getMessage(), e);
            throw new MeterException(MeterStateEnum.流写入错误);
        }
    }

    /**
     * @description: 写打印日志
     **/
    protected void write_log(byte[] bytes) {
        if (bytes != null && bytes.length > 0) {
            if ((log || logBack)) {
                byte[] buffer = new byte[1024];
                int min = Math.min(buffer.length, bytes.length);
                System.arraycopy(bytes, 0, buffer, 0, min);

                String logStr = "Write:[ " + ByteUtils.toHexString(Arrays.copyOfRange(buffer, 0, min)) + " ]";
                if (this.log) System.out.println(log);
                if (this.logBack && this.portLogBack != null) this.portLogBack.logBack(logStr);
            }
            if (GlobalPara.getInstance().isRunLog()) {
                byte[] buffer = new byte[1024];
                int min = Math.min(buffer.length, bytes.length);
                System.arraycopy(bytes, 0, buffer, 0, min);
                System.out.println("Write:[ " + ByteUtils.toHexString(Arrays.copyOfRange(buffer, 0, min)) + " ]");
            }

        }
    }

    /**
     * @description: 读取有效长度
     **/
    public byte[] readData() throws MeterException {
        byte[] accept = new byte[this.readBytesLength];
        ReadResponse response = this.readData(accept);
        return invalidBytes(accept, response.getLength());
    }

    /**
     * @description: 读取最大长度
     **/
    public byte[] readData(int length) throws MeterException {
        byte[] accept = new byte[length];
        ReadResponse response = this.readData(accept);
        return invalidBytes(accept, response.getLength());
    }

    /**
     * @description: 读取字节流 (Override)
     **/
    public ReadResponse readData(byte[] bytes) throws MeterException {
        try {
            int len = this.getInputStream() != null && this.getInputStream().available() > 0 ?
                    this.getInputStream().read(bytes) : 0;
            this.read_log(bytes, len);
            return new ReadResponse(len, bytes);
        } catch (IOException e) {
            JLog.error("PortManager read error：{}", e.getMessage(), e);
            throw new MeterException(MeterStateEnum.流读取错误);
        }
    }

    /**
     * @description: 读取日志
     **/
    protected void read_log(byte[] bytes, int len) {
        byte[] content = invalidBytes(bytes, len);
        if (content != null && content.length > 0) {
            String logStr = "Read:[ " + ByteUtils.toHexString(content) + " ]";

            if ((log || logBack) && bytes != null && len > 0) {
                if (this.log) System.out.println(logStr);
                if (this.logBack && this.portLogBack != null) this.portLogBack.logBack(logStr);
            }
            // if (GlobalPara.getInstance().isRunLog()) System.out.println(logStr);
        }
    }

    /**
     * @description: 关闭端口 (Override)
     **/
    public boolean closePort() throws MeterException {
        if (!this.status) return true;
        try {
            JLog.info("{} close [ {} ] start...", this.device.getClass().getName(), this.device.getDevice());
            if (this.clientMaps != null && !this.clientMaps.isEmpty()) {
                Iterator<String> iterator = this.clientMaps.keySet().iterator();
                while (iterator.hasNext()) {
                    String next = iterator.next();
                    PortManager portManager = this.clientMaps.remove(next);
                    portManager.closePort();
                }
                this.clientMaps.clear();
            }

            if (this.thread != null) {
                this.thread.isInterrupted();
                this.thread.destroy();
                this.thread = null;
            }

            if (this.getInputStream() != null) {
                try {
                    this.getInputStream().close();
                } catch (Exception e) {
                }
            }

            if (this.getOutputStream() != null) {
                try {
                    this.getOutputStream().close();
                } catch (Exception e) {
                }
            }

            if (this.control != null) {
                this.control.close();
            }

            if (clientEventListener != null) {
                this.clientEventListener.stop();
            }
            JLog.info("{} close [ {} ] success", this.device.getClass().getName(), this.device.getDevice());
            this.status = false;
            return true;
        } catch (Exception e) {
            JLog.error("{} close [ {} ] error：{}", this.device.getClass().getName(), this.device.getDevice(), e.getMessage(), e);
            return false;
        }
    }

    /**
     * @description: 连接端口名称
     **/
    public String portname() {
        return Objects.nonNull(this.device) ? this.device.getDevice() : null;
    }

    /**
     * @description: 获取设备列表
     **/
    public Set<T> getDevices() {
        return this.devices;
    }

    /**
     * @description: 获取输入流
     **/
    public InputStream getInputStream() throws MeterException {
        return this.inputStream != null ? this.inputStream : this.control.getInputStream();
    }

    /**
     * @description: 获取输出流
     **/
    public OutputStream getOutputStream() throws MeterException {
        return this.outputStream != null ? this.outputStream : this.control.getOutputStream();
    }

    public DeviceControl getControl() {
        return control;
    }

    public void setControl(DeviceControl control) {
        this.control = control;
    }

    public DeviceDiscoverEventListener getDiscoverEventListener() {
        return discoverEventListener;
    }

    public void setDiscoverEventListener(DeviceDiscoverEventListener discoverEventListener) {
        this.discoverEventListener = discoverEventListener;
    }

    public T getDevice() {
        return device;
    }

    public void setDevice(T device) {
        this.device = device;
    }

    public DeviceClientEventListener getClientEventListener() {
        return clientEventListener;
    }

    public void setClientEventListener(DeviceClientEventListener clientEventListener) {
        this.clientEventListener = clientEventListener;
    }

    public void setInputStream(InputStream inputStream) {
        this.inputStream = inputStream;
    }

    public void setOutputStream(OutputStream outputStream) {
        this.outputStream = outputStream;
    }

    public boolean isStatus() {
        return status;
    }

    public void setStatus(boolean status) {
        this.status = status;
    }

    /**
     * @description: Vector 转 Byte
     **/
    protected static byte[] convertVectorByteToBytes(Vector<Byte> data) {
        byte[] sendData = new byte[data.size()];
        if (data.size() > 0) {
            for (int i = 0; i < data.size(); ++i) {
                sendData[i] = data.get(i);
            }
        }
        return sendData;
    }

    /**
     * @description: Byte 转 Vector
     **/
    protected static Vector<Byte> convertBytesToVectorByte(byte[] bytes) {
        Vector<Byte> vector = new Vector<Byte>();
        for (byte b : bytes) {
            vector.add(b);
        }
        return vector;
    }

    /**
     * @description: 获取有效长度字节流
     **/
    public static byte[] invalidBytes(byte[] source, int length) {
        try {
            return Arrays.copyOf(source, length);
        } catch (Exception e) {
            e.printStackTrace();
            return new byte[0];
        }
    }

    /**
     * @description: 客户端异步线程
     **/
    protected class ClientThread extends Thread {

        private ClientControl control;
        private long start = System.currentTimeMillis();
        private long count = 0;
        private boolean run = true;
        private PortManager portManager;

        public ClientThread(ClientControl control) {
            super(String.format("%s_%s_Thread", device.getDevice(), System.currentTimeMillis()));
            this.control = control;
            this.portManager = _this();
            if (clientEventListener != null && clientEventListener instanceof SocketClientEventListener) {
                server.clientMaps.put(this.portManager.portname(), this.portManager);
                ((SocketClientEventListener) clientEventListener).onClientLink(this.control.getAddress(), this.control.getPort());
            }
        }

        /**
         * @description: 检查运行状态
         **/
        private boolean doCheck() {
            long count = (System.currentTimeMillis() - start) / 1000;
            if (this.count != count && count % 10 == 0) {
                this.count = count;
                return true;
            } else {
                return false;
            }
        }

        @Override
        public void run() {
            while (!isInterrupted() && run) {

                if (this.doCheck()) {
                    if (!this.control.checkLink()) break;
                }

                try {
                    byte[] accept = readData();
                    if (clientEventListener != null && clientEventListener instanceof SocketClientEventListener) {
                        byte[] send = ((SocketClientEventListener) clientEventListener).onClientEventResponse(this.portManager, accept);
                        if (send != null && send.length > 0) writeDataImmediately(send);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                    if (log)
                        System.err.printf("单线程读取监听异常：%s%n", e.getMessage());

                    if (clientEventListener != null) {
                        if (e instanceof MeterException) {
                            clientEventListener.onClientEventExp((MeterException) e);
                        } else {
                            clientEventListener.onClientEventExp(new MeterException(MeterStateEnum.多连接通讯读写异常));
                        }
                    }
                }

                try {
                    Thread.sleep(1);
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }

            if (clientEventListener != null && clientEventListener instanceof SocketClientEventListener) {
                clientMaps.remove(this.portManager.portname());
                ((SocketClientEventListener) clientEventListener).onClientBreak(this.control.getAddress(), this.control.getPort());
            }
        }

        /**
         * Throws {@link NoSuchMethodError}.
         *
         * @throws NoSuchMethodError always
         * @deprecated This method was originally designed to destroy this
         * thread without any cleanup. Any monitors it held would have
         * remained locked. However, the method was never implemented.
         * If if were to be implemented, it would be deadlock-prone in
         * much the manner of {@link #suspend}. If the target thread held
         * a lock protecting a critical system resource when it was
         * destroyed, no thread could ever access this resource again.
         * If another thread ever attempted to lock this resource, deadlock
         * would result. Such deadlocks typically manifest themselves as
         * "frozen" processes. For more information, see
         * <a href="{@docRoot}/../technotes/guides/concurrency/threadPrimitiveDeprecation.html">
         * Why are Thread.stop, Thread.suspend and Thread.resume Deprecated?</a>.
         */
        @Override
        public void destroy() {
            this.run = false;
        }
    }
}
