package com.jhscale.meter.io.control.win;

import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.exp.MeterStateEnum;
import com.jhscale.meter.io.BluetoothPort;
import com.jhscale.meter.io.PortManager;
import com.jhscale.meter.io.control.DeviceControl;
import com.jhscale.meter.io.listener.DeviceClientEventListener;
import com.jhscale.meter.io.listener.DeviceDiscoverEventListener;
import com.jhscale.meter.log.JLog;
import com.jhscale.meter.model.device.BlueDevice;
import com.jhscale.meter.model.device.InitDeviceEntity;
import com.jhscale.meter.utils.BluecoveUtils;

import javax.bluetooth.*;
import javax.microedition.io.Connector;
import javax.microedition.io.StreamConnection;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Set;

/**
 * @author lie_w
 * @title: WBluetoothControl
 * @projectName meter-jar
 * @description: Win 蓝牙控制器
 * @date 2021/6/618:31
 */
public class WBluetoothControl implements DeviceControl<BlueDevice> {

    // 蓝牙发现监听器
    private WBluetoothDiscoverEventListener discoverEventListener;
    // 连接后监听器
    private WBluetoothClientListener clientListener;
    // 流连接
    private StreamConnection streamConnection;

    /**
     * @description: 初始化设备控制器
     */
    @Override
    public void initDevice() throws MeterException {
        // 本机驱动检查
        BluecoveUtils.doctorDevice();
    }

    /**
     * @param entity
     * @description: 初始化控制器参数
     */
    @Override
    public void initParam(InitDeviceEntity entity) {

    }

    /**
     * @param discoverEventListener
     * @param devicesDiscovered
     * @description: 添加查找监听
     */
    @Override
    public void addListener(DeviceDiscoverEventListener discoverEventListener, Set<BlueDevice> devicesDiscovered) {
        this.discoverEventListener = new WBluetoothDiscoverEventListener(discoverEventListener, devicesDiscovered);
    }

    /**
     * @description: 开启查找
     **/
    @Override
    public boolean discovery() throws MeterException {
        try {
            JLog.debug("{} discovery 本机蓝牙名称: [ {} ] 开启设备查找...", BluetoothPort.TAG, LocalDevice.getLocalDevice().getFriendlyName());
            return LocalDevice.getLocalDevice().getDiscoveryAgent().startInquiry(DiscoveryAgent.GIAC, this.discoverEventListener);
        } catch (BluetoothStateException e) {
            JLog.error("{} discovery 搜索蓝牙设备查找开启异常：{}", BluetoothPort.TAG, e.getMessage(), e);
            throw new MeterException(MeterStateEnum.蓝牙开启查找设备异常);
        }
    }

    /**
     * @description: 关闭查找
     **/
    @Override
    public boolean cancelDiscovery() throws MeterException {
        try {
            JLog.debug("{} cancelDiscovery 本机蓝牙名称: [ {} ] 关闭设备查找...", BluetoothPort.TAG, LocalDevice.getLocalDevice().getFriendlyName());
            LocalDevice.getLocalDevice().getDiscoveryAgent().cancelInquiry(this.discoverEventListener);
            this.discoverEventListener = null;
            return true;
        } catch (BluetoothStateException e) {
            JLog.error("{} cancelDiscovery 搜索蓝牙设备查找关闭异常：{}", BluetoothPort.TAG, e.getMessage(), e);
            throw new MeterException(MeterStateEnum.蓝牙关闭查找设备异常);
        }
    }

    /**
     * @param device
     * @description: 开启连接
     */
    @Override
    public void openPort(BlueDevice device) throws MeterException {
        try {
            JLog.debug("{} openport 打开蓝牙连接: [ {} ] ", BluetoothPort.TAG, device.getDevice());
            streamConnection = (StreamConnection) Connector.open("btspp://" + device.getDevice() + ":1");
        } catch (IOException e) {
            JLog.error("{} openport 打开蓝牙连接失败：{}", BluetoothPort.TAG, e.getMessage(), e);
            throw new MeterException(MeterStateEnum.蓝牙打开失败);
        }
    }

    /**
     * @param clientEventListener
     * @param portManager
     * @description: 添加连接后监听器
     */
    @Override
    public void addListener(DeviceClientEventListener clientEventListener, PortManager portManager) throws MeterException {
        this.clientListener = new WBluetoothClientListener(clientEventListener, portManager);
        this.clientListener.start();
    }

    /**
     * @description: 输出流
     **/
    @Override
    public InputStream getInputStream() throws MeterException {
        if (!this.checkLink()) throw new MeterException(MeterStateEnum.蓝牙未打开);
        try {
            return this.streamConnection.openDataInputStream();
        } catch (IOException e) {
            throw new MeterException(MeterStateEnum.获取蓝牙输入流失败);
        }
    }

    /**
     * @description: 输入流
     **/
    @Override
    public OutputStream getOutputStream() throws MeterException {
        if (!this.checkLink()) throw new MeterException(MeterStateEnum.蓝牙未打开);
        try {
            return this.streamConnection.openDataOutputStream();
        } catch (IOException e) {
            throw new MeterException(MeterStateEnum.获取蓝牙输出流失败);
        }
    }

    /**
     * @description: 关闭操作
     **/
    @Override
    public void close() throws MeterException {
        if (this.streamConnection != null) {
            this.clientListener.clientState = false;
            try {
                Thread.sleep(100);
                this.streamConnection.close();
            } catch (InterruptedException | IOException e) {
                JLog.error("{} close 关闭蓝牙连接失败：{}", BluetoothPort.TAG, e.getMessage(), e);
                throw new MeterException(MeterStateEnum.蓝牙关闭异常);
            }
            this.streamConnection = null;
            this.clientListener = null;
        }
    }

    /**
     * @description: 连接检查
     **/
    @Override
    public boolean checkLink() {
        return this.streamConnection != null;
    }

//    /**
//     * @param bytes
//     * @param offset
//     * @param len
//     * @description: 写出数据
//     */
//    @Override
//    public void writeData(byte[] bytes, int offset, int len) throws MeterException {
//        try {
//            this.getOutputStream().write(bytes, 0, len);
//            this.getOutputStream().flush();
//        } catch (IOException e) {
//            JLog.error("serial write error：{}", e.getMessage(), e);
//            throw new MeterException(MeterStateEnum.蓝牙写入错误);
//        }
//    }
//
//    /**
//     * @param bytes
//     * @description: 读取数据
//     */
//    @Override
//    public int readData(byte[] bytes) throws MeterException {
//        try {
//            return this.getInputStream() != null && this.getInputStream().available() > 0 ?
//                    this.getInputStream().read(bytes) : 0;
//        } catch (IOException e) {
//            JLog.error("serial read error：{}", e.getMessage(), e);
//            throw new MeterException(MeterStateEnum.蓝牙读取错误);
//        }
//    }

    /**
     * @description: BLUECOVE 发现监听
     **/
    private static class WBluetoothDiscoverEventListener implements DiscoveryListener {

        private DeviceDiscoverEventListener discoverEventListener;
        private Set<BlueDevice> devicesDiscovered;

        public WBluetoothDiscoverEventListener(DeviceDiscoverEventListener discoverEventListener, Set<BlueDevice> devicesDiscovered) {
            this.discoverEventListener = discoverEventListener;
            this.devicesDiscovered = devicesDiscovered;
        }

        public void inquiryCompleted(int discType) {
            JLog.debug("{} RemoteDeviceDiscovery 搜索完成", BluetoothPort.TAG);
            if (discoverEventListener != null)
                discoverEventListener.onDiscoverSuccessEvent(devicesDiscovered);
        }

        @Override
        public void deviceDiscovered(RemoteDevice remoteDevice, DeviceClass deviceClass) {
            BlueDevice blueDevice = BluecoveUtils.deviceInfo(remoteDevice);
            JLog.debug("{} RemoteDeviceDiscovery 发现设备: [ {} ]", BluetoothPort.TAG, blueDevice.getBlueName());
            if (discoverEventListener != null)
                discoverEventListener.onDiscoverEvent(blueDevice);
            if (devicesDiscovered != null)
                devicesDiscovered.add(blueDevice);
        }

        @Override
        public void servicesDiscovered(int arg0, ServiceRecord[] arg1) {
            JLog.debug("{} RemoteDeviceDiscovery servicesDiscovered", BluetoothPort.TAG);
        }

        @Override
        public void serviceSearchCompleted(int arg0, int arg1) {
            JLog.debug("{} RemoteDeviceDiscovery serviceSearchCompleted", BluetoothPort.TAG);
        }
    }

    /**
     * @description: 默认蓝牙连接监听器
     **/
    private static class WBluetoothClientListener extends Thread {

        private DeviceClientEventListener clientEventListener;
        private PortManager portManager;
        private boolean clientState = true;

        public WBluetoothClientListener(DeviceClientEventListener clientEventListener, PortManager portManager) {
            super("WBluetoothClientListener_Thread");
            this.clientEventListener = clientEventListener;
            this.portManager = portManager;
        }

        @Override
        public void run() {
            while (clientState) {
                try {
                    if (this.clientEventListener != null)
                        clientEventListener.onClientEvent(portManager.readData());
                    Thread.sleep(100);
                } catch (MeterException e) {
                    JLog.error("{} Listener Read [ {} ] error:{}", BluetoothPort.TAG, portManager.portname(), e.getMessage(), e);
                    if (this.clientEventListener != null)
                        clientEventListener.onClientEventExp(e);
                } catch (InterruptedException e) {
                    JLog.error("{} Listener Read Time [ {} ] error:{}", BluetoothPort.TAG, portManager.portname(), e.getMessage(), e);
                }
            }
        }
    }
}
