package com.jhscale.meter.io.control.win;

import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.exp.MeterStateEnum;
import com.jhscale.meter.io.PortManager;
import com.jhscale.meter.io.control.DeviceControl;
import com.jhscale.meter.io.listener.DeviceClientEventListener;
import com.jhscale.meter.io.listener.DeviceDiscoverEventListener;
import com.jhscale.meter.log.JLog;
import com.jhscale.meter.model.device.InitDeviceEntity;
import com.jhscale.meter.model.device.SerialDevice;
import com.jhscale.meter.protocol.model.GlobalPara;
import com.jhscale.meter.utils.ByteUtils;
import gnu.io.*;

import java.io.InputStream;
import java.io.OutputStream;
import java.util.Enumeration;
import java.util.Set;
import java.util.TooManyListenersException;

/**
 * @author lie_w
 * @title: WSerialPortControl
 * @projectName meter-jar
 * @description: Win 串口控制器
 * @date 2021/6/617:03
 */
public class WSerialPortControl implements DeviceControl<SerialDevice> {

    // 串口通讯
    private SerialPort serialPort;

    // 发现监听
    private WSerialDiscoverEventListener discoverEventListener;

    /**
     * @description: 初始化设备控制器
     */
    @Override
    public void initDevice() {

    }

    /**
     * @param entity
     * @description: 初始化控制器参数
     */
    @Override
    public void initParam(InitDeviceEntity entity) {

    }

    /**
     * @param discoverEventListener
     * @param devicesDiscovered
     * @description: 添加查找监听
     */
    @Override
    public void addListener(DeviceDiscoverEventListener discoverEventListener, Set<SerialDevice> devicesDiscovered) {
        this.discoverEventListener = new WSerialDiscoverEventListener(discoverEventListener, devicesDiscovered);
    }

    /**
     * @description: 开启查找
     **/
    @Override
    public boolean discovery() {
        if (this.discoverEventListener != null)
            this.discoverEventListener.start();
        return true;
    }

    /**
     * @description: 关闭查找
     **/
    @Override
    public boolean cancelDiscovery() {
        if (this.discoverEventListener != null)
            this.discoverEventListener.discoverEventState = false;
        this.discoverEventListener = null;
        return true;
    }

    /**
     * @param device
     * @description: 开启连接
     */
    @Override
    public void openPort(SerialDevice device) throws MeterException {
        try {
            JLog.info("{} open [ {} ] start...", com.jhscale.meter.io.SerialPort.TAG, device.getDevice());
            // 通过端口名识别端口 打开端口，并给端口名字和一个timeout（打开操作的超时时间）
            CommPort commPort = CommPortIdentifier.getPortIdentifier(device.getDevice()).open(device.getDevice(), 2000);
            if (commPort instanceof SerialPort) {
                this.serialPort = (SerialPort) commPort;
                //设置一下串口的波特率等参数
                serialPort.setSerialPortParams(device.getBaudrate(), SerialPort.DATABITS_8, SerialPort.STOPBITS_1, SerialPort.PARITY_NONE);
                JLog.info("{} open [ {} ] success", com.jhscale.meter.io.SerialPort.TAG, device.getDevice());
            } else {
                throw new MeterException(MeterStateEnum.端口指向设备不是串口类型);
            }
        } catch (NoSuchPortException e) {
            throw new MeterException(MeterStateEnum.没有该端口对应的串口设备);
        } catch (PortInUseException e) {
            throw new MeterException(MeterStateEnum.端口已被占用);
        } catch (UnsupportedCommOperationException e) {
            throw new MeterException(MeterStateEnum.设置串口参数失败);
        }
    }

    /**
     * @param clientEventListener
     * @param portManager
     * @description: 添加连接后监听器
     */
    @Override
    public void addListener(DeviceClientEventListener clientEventListener, PortManager portManager) throws MeterException {
        if (!this.checkLink()) throw new MeterException(MeterStateEnum.串口未打开);
        try {
            //给串口添加监听器
            this.serialPort.addEventListener(new SerialPortEventListener() {
                /**
                 * @description: SerialPortEvent.BI: // 10 通讯中断
                 * SerialPortEvent.OE: // 7 溢位（溢出）错误
                 * SerialPortEvent.FE: // 9 帧错误
                 * SerialPortEvent.PE: // 8 奇偶校验错误
                 * SerialPortEvent.CD: // 6 载波检测
                 * SerialPortEvent.CTS: // 3 清除待发送数据
                 * SerialPortEvent.DSR: // 4 待发送数据准备好了
                 * SerialPortEvent.RI: // 5 振铃指示
                 * SerialPortEvent.OUTPUT_BUFFER_EMPTY: // 2 输出缓冲区已清空
                 * SerialPortEvent.DATA_AVAILABLE: // 1 串口存在可用数据
                 **/
                @Override
                public void serialEvent(SerialPortEvent serialPortEvent) {
                    try {

                        int eventType = serialPortEvent.getEventType();
                        byte[] bytes = portManager.readData();
                        // if (GlobalPara.getInstance().isRunLog())
                        //     System.out.printf("EventType: %s ,Content: %s %n", eventType, ByteUtils.toHexString(bytes));

                        if (clientEventListener != null)
                            clientEventListener.onClientEvent(SerialPortEvent.DATA_AVAILABLE == eventType ? bytes : new byte[0]);
                    } catch (MeterException e) {
                        JLog.error("{} Listener Read [ {} ] error:{}", com.jhscale.meter.io.SerialPort.TAG, portManager.portname(), e.getMessage(), e);
                        clientEventListener.onClientEventExp(e);
                    }
                }
            });
            //设置当有数据到达时唤醒监听接收线程
            this.serialPort.notifyOnDataAvailable(true);
            //设置当通信中断时唤醒中断线程
            this.serialPort.notifyOnBreakInterrupt(true);
        } catch (TooManyListenersException e) {
            JLog.error("{} addListener [ {} ] error：{}", com.jhscale.meter.io.SerialPort.TAG, portManager.portname(), e.getMessage(), e);
            throw new MeterException(MeterStateEnum.监听类对象过多);
        }
    }

    /**
     * @description: 输出流
     **/
    @Override
    public InputStream getInputStream() throws MeterException {
        if (!this.checkLink()) throw new MeterException(MeterStateEnum.串口未打开);
        try {
            return this.serialPort.getInputStream();
        } catch (Exception e) {
            throw new MeterException(MeterStateEnum.获取串口输入流失败);
        }
    }

    /**
     * @description: 输入流
     **/
    @Override
    public OutputStream getOutputStream() throws MeterException {
        if (!this.checkLink()) throw new MeterException(MeterStateEnum.串口未打开);
        try {
            return this.serialPort.getOutputStream();
        } catch (Exception e) {
            throw new MeterException(MeterStateEnum.获取串口输出流失败);
        }
    }

    /**
     * @description: 关闭操作
     **/
    @Override
    public void close() throws MeterException {
        if (this.serialPort != null) {
            this.serialPort.close();
            this.serialPort = null;
        }
    }

    /**
     * @description: 连接检查
     **/
    @Override
    public boolean checkLink() {
        return this.serialPort != null;
    }

//    /**
//     * @param bytes
//     * @param offset
//     * @param len
//     * @description: 写出数据
//     */
//    @Override
//    public void writeData(byte[] bytes, int offset, int len) throws MeterException {
//        try {
//            this.getOutputStream().write(bytes, 0, len);
//            this.getOutputStream().flush();
//        } catch (IOException e) {
//            JLog.error("serial write error：{}", e.getMessage(), e);
//            throw new MeterException(MeterStateEnum.串口写入错误);
//        }
//    }
//
//    /**
//     * @param bytes
//     * @description: 读取数据
//     */
//    @Override
//    public int readData(byte[] bytes) throws MeterException {
//        try {
//            return this.getInputStream() != null && this.getInputStream().available() > 0 ?
//                    this.getInputStream().read(bytes) : 0;
//        } catch (IOException e) {
//            JLog.error("serial read error：{}", e.getMessage(), e);
//            throw new MeterException(MeterStateEnum.串口读取错误);
//        }
//    }

    /**
     * @description: 串口发现监听
     **/
    private static class WSerialDiscoverEventListener extends Thread {
        private DeviceDiscoverEventListener deviceDiscoverEventListener;
        private Set<SerialDevice> devicesDiscovered;
        private boolean discoverEventState = true;

        public WSerialDiscoverEventListener(DeviceDiscoverEventListener deviceDiscoverEventListener, Set<SerialDevice> devicesDiscovered) {
            super("WSerialDiscoverEventListener_Thread");
            this.deviceDiscoverEventListener = deviceDiscoverEventListener;
            this.devicesDiscovered = devicesDiscovered;
        }

        @Override
        public void run() {
            try {
                Enumeration<CommPortIdentifier> portList = CommPortIdentifier.getPortIdentifiers();
                while (discoverEventState && portList.hasMoreElements()) {
                    CommPortIdentifier commPortIdentifier = portList.nextElement();
                    SerialDevice<CommPortIdentifier> serialDevice = new SerialDevice<>(commPortIdentifier.getName(), commPortIdentifier);
                    if (devicesDiscovered != null)
                        devicesDiscovered.add(serialDevice);
                    if (deviceDiscoverEventListener != null)
                        deviceDiscoverEventListener.onDiscoverEvent(serialDevice);
                }
                if (deviceDiscoverEventListener != null)
                    deviceDiscoverEventListener.onDiscoverSuccessEvent(devicesDiscovered);
            } catch (Exception e) {
                JLog.error("{} discovery error：{}", com.jhscale.meter.io.SerialPort.TAG, e.getMessage(), e);
            }
        }
    }
}
