package com.jhscale.meter.io.control.win;

import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.exp.MeterStateEnum;
import com.jhscale.meter.io.PortManager;
import com.jhscale.meter.io.control.SocketControl;
import com.jhscale.meter.io.listener.DeviceClientEventListener;
import com.jhscale.meter.io.listener.SocketClientEventListener;
import com.jhscale.meter.model.device.UDPDevice;
import com.jhscale.meter.protocol.model.SocketReadResponse;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.DatagramPacket;
import java.net.DatagramSocket;

/**
 * @author lie_w
 * @title: WTCPServerControl
 * @projectName meter-jar
 * @description: Win TCP 客户端 控制器
 * @date 2022/12/3010:09
 */
public class WUDPControl implements SocketControl<UDPDevice> {

    // UDP设备信息
    private UDPDevice udpDevice;
    // Socket 接受对象
    private DatagramSocket socket;
    // 监听器
    private DeviceClientEventListener clientEventListener;
    // 端口控制器
    private PortManager portManager;
    // UDP异步读取线程
    private WUDPThread thread;

    /**
     * @param device
     * @description: 开启连接
     */
    @Override
    public void openPort(UDPDevice device) throws MeterException {
        try {
            this.udpDevice = device;
            this.socket = new DatagramSocket(device.getListener());
            this.thread = new WUDPThread(portManager, clientEventListener);
            this.thread.start();
        } catch (IOException e) {
            throw new MeterException(MeterStateEnum.UDP打开失败);
        }
    }

    /**
     * @param clientEventListener
     * @param portManager
     * @description: 添加连接后监听器
     */
    @Override
    public void addListener(DeviceClientEventListener clientEventListener, PortManager portManager) throws MeterException {
        this.clientEventListener = clientEventListener;
        this.portManager = portManager;
    }

    /**
     * @description: 输出流
     **/
    @Override
    public InputStream getInputStream() throws MeterException {
        return null;
        // throw new MeterException(MeterStateEnum.UDP不支持IO输入流);
    }

    /**
     * @description: 输入流
     **/
    @Override
    public OutputStream getOutputStream() throws MeterException {
        return null;
        // throw new MeterException(MeterStateEnum.UDP不支持IO输出流);
    }

    /**
     * @description: 关闭操作
     **/
    @Override
    public void close() throws MeterException {
        if (this.socket != null) {
            this.thread.close();
            this.socket.close();
            this.socket = null;
            this.thread = null;
        }
    }

    /**
     * @param bytes
     * @description: 通讯控制器直接写出
     */
    @Override
    public synchronized void write(byte[] bytes) throws MeterException {
        if (this.socket != null) {
            try {
                DatagramPacket packet = new DatagramPacket(bytes, bytes.length,
                        this.udpDevice.broadcastAddress(), this.udpDevice.getBroadcast());
                this.socket.send(packet);
            } catch (IOException e) {
                e.printStackTrace();
                throw new MeterException(MeterStateEnum.UDP写数据失败);
            }
        }
    }

    /**
     * @param bytes
     * @description: 通讯器直接读取
     */
    @Override
    public SocketReadResponse read(byte[] bytes) throws MeterException {
        if (this.socket != null) {
            try {
                DatagramPacket packet = new DatagramPacket(bytes, bytes.length);
                // 3.等待接受数据
                socket.receive(packet);
                // 4.取出数据
                // String accept = new String(bytes, 0, packet.getLength());
                // System.out.println(String.format("收到来自:%s,对方端口号为:%s,消息内容：%s",
                //         packet.getAddress(), packet.getPort(), accept));
                return new SocketReadResponse(packet.getLength(),
                        bytes,
                        packet.getAddress(),
                        packet.getPort());
            } catch (IOException e) {
                throw new MeterException(MeterStateEnum.UDP读数据失败);
            }
        }
        return new SocketReadResponse(-1);
    }

    /**
     * @description: UDP 异步线程
     **/
    private static class WUDPThread extends Thread {

        private PortManager portManager;
        private DeviceClientEventListener clientEventListener;

        public WUDPThread(PortManager portManager, DeviceClientEventListener clientEventListener) {
            this.portManager = portManager;
            this.clientEventListener = clientEventListener;
        }

        /**
         * If this thread was constructed using a separate
         * <code>Runnable</code> run object, then that
         * <code>Runnable</code> object's <code>run</code> method is called;
         * otherwise, this method does nothing and returns.
         * <p>
         * Subclasses of <code>Thread</code> should override this method.
         *
         * @see #start()
         * @see #stop()
         */
        @Override
        public void run() {
            while (!isInterrupted() && this.portManager != null) {
                try {
                    byte[] buffer = new byte[1024];
                    SocketReadResponse response = (SocketReadResponse) this.portManager.readData(buffer);
                    if (this.clientEventListener != null
                            && this.clientEventListener instanceof SocketClientEventListener) {
                        byte[] bytes = ((SocketClientEventListener) this.clientEventListener).onClientEvent(
                                this.portManager, response.getFrom_address(),
                                response.getFrom_port(), response.invalidBytes());
                        if (bytes != null && bytes.length > 0)
                            this.portManager.writeDataImmediately(bytes);
                    }
                } catch (Exception e) {
                    this.close();
                }
            }
        }

        /**
         * @description: 停止异步线程
         **/
        synchronized void close() {
            if (this.portManager != null) {
                this.interrupt();
                this.portManager = null;
            }
        }
    }
}
