package com.jhscale.meter.io.control.win;

import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.exp.MeterStateEnum;
import com.jhscale.meter.io.BluetoothPort;
import com.jhscale.meter.io.PortManager;
import com.jhscale.meter.io.USBPort;
import com.jhscale.meter.io.control.DeviceControl;
import com.jhscale.meter.io.entity.JEndpointInputStream;
import com.jhscale.meter.io.entity.JEndpointOutputStream;
import com.jhscale.meter.io.listener.DeviceClientEventListener;
import com.jhscale.meter.io.listener.DeviceDiscoverEventListener;
import com.jhscale.meter.log.JLog;
import com.jhscale.meter.model.device.InitDeviceEntity;
import com.jhscale.meter.model.device.USBDevice;

import javax.usb.*;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;
import java.util.Objects;
import java.util.Set;

/**
 * @author lie_w
 * @title: WUSBPortControl
 * @projectName meter-jar
 * @description: Win USB控制器
 * @date 2021/6/619:24
 */
public class WUSBControl implements DeviceControl<USBDevice> {

    private Short VID;
    private Short PID;
    private UsbInterface iface;
    // 发现监听器
    private WUSBDeviceDiscoverEventListener discoverEventListener;
    // 连接后监听器
    private WUSBClientListener clientListener;
    private JEndpointInputStream inputStream;
    private JEndpointOutputStream outputStream;


    /**
     * @description: 初始化设备控制器
     */
    @Override
    public void initDevice() throws MeterException {

    }

    /**
     * @param entity
     * @description: 初始化控制器参数
     */
    @Override
    public void initParam(InitDeviceEntity entity) {
        this.VID = entity.getVID();
        this.PID = entity.getPID();
    }

    /**
     * @param discoverEventListener
     * @param devicesDiscovered
     * @description: 添加查找监听
     */
    @Override
    public void addListener(DeviceDiscoverEventListener discoverEventListener, Set<USBDevice> devicesDiscovered) {
        this.discoverEventListener = new WUSBDeviceDiscoverEventListener(this.VID, this.PID, discoverEventListener, devicesDiscovered);
    }

    /**
     * @description: 开启查找
     **/
    @Override
    public boolean discovery() throws MeterException {
        if (this.discoverEventListener != null)
            this.discoverEventListener.start();
        return true;
    }

    /**
     * @description: 关闭查找
     **/
    @Override
    public boolean cancelDiscovery() throws MeterException {
        this.discoverEventListener = null;
        return true;
    }

    /**
     * @param device
     * @description: 开启连接
     */
    @Override
    public void openPort(USBDevice device) throws MeterException {
        try {
            UsbDevice usbDevice = null;
            if (device.getDeviceInfo() != null) {
                usbDevice = (UsbDevice) device.getDeviceInfo();
            } else {
                usbDevice = WUSBDeviceDiscoverEventListener.findDevice(UsbHostManager.getUsbServices().getRootUsbHub(), device.getVid(), device.getPid());
                if (usbDevice != null) {
                    device.setDevice(usbDevice.toString());
                    device.setDeviceInfo(usbDevice);
                }
            }
            if (Objects.isNull(usbDevice))
                throw new MeterException(MeterStateEnum.未找到USB接口);
            UsbConfiguration configuration = usbDevice.getActiveUsbConfiguration();
            UsbInterface iface = null;
            if (configuration.getUsbInterfaces().size() > 0) {
                iface = (UsbInterface) configuration.getUsbInterfaces().get(0);
            } else {
                throw new MeterException(MeterStateEnum.USB连接失败);
            }
            iface.claim();
            this.iface = iface;
        } catch (UsbException e) {
            JLog.error("{} openPort USB连接失败：{}", BluetoothPort.TAG, e.getMessage(), e);
            throw new MeterException(MeterStateEnum.USB连接失败);
        }
    }

    /**
     * @param clientEventListener
     * @param portManager
     * @description: 添加连接后监听器
     */
    @Override
    public void addListener(DeviceClientEventListener clientEventListener, PortManager portManager) throws MeterException {
        this.clientListener = new WUSBClientListener(clientEventListener, portManager);
        this.clientListener.start();
    }

    /**
     * @description: 输出流
     **/
    @Override
    public InputStream getInputStream() throws MeterException {
        if (!this.checkLink()) throw new MeterException(MeterStateEnum.USB未打开);
        try {
            if (this.inputStream == null) {
                UsbEndpoint usbEndpoint = (UsbEndpoint) this.iface.getUsbEndpoints().get(0);
                if (usbEndpoint.getUsbEndpointDescriptor().toString().contains("OUT")) {
                    usbEndpoint = (UsbEndpoint) this.iface.getUsbEndpoints().get(1);
                }
                this.inputStream = new JEndpointInputStream(usbEndpoint);
            }
            return this.inputStream;
        } catch (Exception e) {
            throw new MeterException(MeterStateEnum.获取串口输入流失败);
        }
    }

    /**
     * @description: 输入流
     **/
    @Override
    public OutputStream getOutputStream() throws MeterException {
        if (!this.checkLink()) throw new MeterException(MeterStateEnum.USB未打开);
        try {
            if (this.outputStream == null) {
                UsbEndpoint usbEndpoint = (UsbEndpoint) this.iface.getUsbEndpoints().get(0);
                if (!usbEndpoint.getUsbEndpointDescriptor().toString().contains("OUT")) {
                    usbEndpoint = (UsbEndpoint) this.iface.getUsbEndpoints().get(1);
                }
                this.outputStream = new JEndpointOutputStream(usbEndpoint);
            }
            return this.outputStream;
        } catch (Exception e) {
            throw new MeterException(MeterStateEnum.获取串口输入流失败);
        }
    }

    /**
     * @description: 关闭操作
     **/
    @Override
    public void close() throws MeterException {
        try {
            if (this.iface != null) {
                this.inputStream.close();
                this.inputStream = null;
                this.outputStream.close();
                this.outputStream = null;
                this.iface = null;
                this.clientListener.clientState = false;
                Thread.sleep(100);
                this.clientListener = null;
            }
        } catch (Exception e) {
            JLog.error("{} close 关闭USB连接失败：{}", USBPort.TAG, e.getMessage(), e);
            throw new MeterException(MeterStateEnum.USB关闭异常);
        }
    }

    /**
     * @description: 连接检查
     **/
    @Override
    public boolean checkLink() {
        return this.iface != null;
    }

//    /**
//     * @param bytes
//     * @param offset
//     * @param len
//     * @description: 写出数据
//     */
//    @Override
//    public void writeData(byte[] bytes, int offset, int len) throws MeterException {
//
//    }
//
//    /**
//     * @param bytes
//     * @description: 读取数据
//     */
//    @Override
//    public int readData(byte[] bytes) throws MeterException {
//        return 0;
//    }

    /**
     * @description: USB发现监听
     **/
    private static class WUSBDeviceDiscoverEventListener extends Thread {
        private Short VID;
        private Short PID;
        private DeviceDiscoverEventListener deviceDiscoverEventListener;
        private Set<USBDevice> devicesDiscovered;

        public WUSBDeviceDiscoverEventListener(Short VID, Short PID,
                                               DeviceDiscoverEventListener deviceDiscoverEventListener,
                                               Set<USBDevice> devicesDiscovered) {
            this.VID = VID;
            this.PID = PID;
            this.deviceDiscoverEventListener = deviceDiscoverEventListener;
            this.devicesDiscovered = devicesDiscovered;
        }

        @Override
        public void run() {
            try {
                findDevice(UsbHostManager.getUsbServices().getRootUsbHub());
                if (this.deviceDiscoverEventListener != null)
                    this.deviceDiscoverEventListener.onDiscoverSuccessEvent(devicesDiscovered);
            } catch (UsbException e) {
                JLog.error("{} discovery error：{}", USBPort.TAG, e.getMessage(), e);
                if (this.deviceDiscoverEventListener != null)
                    this.deviceDiscoverEventListener.onDiscoverEventExp(new MeterException(MeterStateEnum.USB设别查找异常));
            }
        }

        /**
         * @description: 查询设别列表
         **/
        private void findDevice(UsbHub hub) {
            List<UsbDevice> list = (List<UsbDevice>) hub.getAttachedUsbDevices();
            for (UsbDevice device : list) {
                UsbDeviceDescriptor desc = device.getUsbDeviceDescriptor();
                USBDevice<UsbDevice> usbDevice = new USBDevice<>(device.toString(), device);
                if (Objects.nonNull(this.VID) && Objects.nonNull(this.PID)) {
                    if (desc.idVendor() == this.VID && desc.idProduct() == this.PID) {
                        if (this.deviceDiscoverEventListener != null)
                            this.deviceDiscoverEventListener.onDiscoverEvent(usbDevice);
                        if (this.devicesDiscovered != null)
                            this.devicesDiscovered.add(usbDevice);
                    }
                } else {
                    if (this.deviceDiscoverEventListener != null)
                        this.deviceDiscoverEventListener.onDiscoverEvent(usbDevice);
                    if (this.devicesDiscovered != null)
                        this.devicesDiscovered.add(usbDevice);
                }
                if (device.isUsbHub())
                    findDevice((UsbHub) device);
            }
        }

        /**
         * @description: 查找USB设备
         **/
        private static UsbDevice findDevice(UsbHub hub, short vid, short pid) {
            UsbDevice device = null;
            List list = (List) hub.getAttachedUsbDevices();
            for (int i = 0; i < list.size(); i++) {
                device = (UsbDevice) list.get(i);
                UsbDeviceDescriptor desc = device.getUsbDeviceDescriptor();
                if (desc.idVendor() == vid && desc.idProduct() == pid) {
                    return device;
                }
                if (device.isUsbHub()) {
                    device = findDevice((UsbHub) device, vid, pid);
                    if (device != null) return device;
                }
            }
            return null;
        }
    }

    /**
     * @description: 默认USB连接监听器
     **/
    private static class WUSBClientListener extends Thread {

        private DeviceClientEventListener clientEventListener;
        private PortManager portManager;
        private boolean clientState = true;

        public WUSBClientListener(DeviceClientEventListener clientEventListener, PortManager portManager) {
            this.clientEventListener = clientEventListener;
            this.portManager = portManager;
        }

        @Override
        public void run() {
            while (clientState) {
                try {
                    if (this.clientEventListener != null)
                        clientEventListener.onClientEvent(portManager.readData());
                    Thread.sleep(100);
                } catch (MeterException e) {
                    JLog.error("{} Listener Read [ {} ] error:{}", USBPort.TAG, portManager.portname(), e.getMessage(), e);
                    if (this.clientEventListener != null)
                        clientEventListener.onClientEventExp(e);
                } catch (InterruptedException e) {
                    JLog.error("{} Listener Read Time [ {} ] error:{}", USBPort.TAG, portManager.portname(), e.getMessage(), e);
                }
            }
        }
    }
}
