package com.jhscale.meter.io.entity;

import com.jhscale.meter.exp.MeterStateEnum;
import com.jhscale.meter.io.PortManager;
import com.jhscale.meter.io.USBPort;
import com.jhscale.meter.log.JLog;

import javax.usb.UsbEndpoint;
import javax.usb.UsbException;
import javax.usb.UsbPipe;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * @author lie_w
 * @title: EndpointInputstream
 * @projectName meter-jar
 * @description: TODO
 * @date 2021/6/710:42
 */
public class JEndpointInputStream extends InputStream {

    private List<Byte> bytes;
    private USBReadThread readThread;

    public JEndpointInputStream(UsbEndpoint usbEndpoint) throws UsbException {
        UsbPipe usbPipe = usbEndpoint.getUsbPipe();
        usbPipe.open();
        this.bytes = Collections.synchronizedList(new ArrayList<>());
        this.readThread = new USBReadThread(usbPipe, bytes);
        this.readThread.start();
    }

    /**
     * @description: USB读取
     **/
    private static class USBReadThread extends Thread {
        private UsbPipe usbPipe;
        private List<Byte> bytes;
        private boolean run = true;

        public USBReadThread(UsbPipe usbPipe, List<Byte> bytes) {
            super("USBReadThread");
            this.usbPipe = usbPipe;
            this.bytes = bytes;
        }

        @Override
        public void run() {
            byte[] b1 = new byte[128];
            while (run) {
                try {
                    int length = usbPipe.syncSubmit(b1);//阻塞
                    byte[] b2 = PortManager.invalidBytes(b1, length);
                    if (b2 != null && b2.length > 0) {
                        for (byte b : b2) {
                            this.bytes.add(b);
                        }
                    }
                } catch (UsbException e) {
                    JLog.error("{} InputStream error：{}", USBPort.TAG, e.getMessage(), e);
                }
            }
        }
    }

    /**
     * Reads the next byte of data from the input stream. The value byte is
     * returned as an <code>int</code> in the range <code>0</code> to
     * <code>255</code>. If no byte is available because the end of the stream
     * has been reached, the value <code>-1</code> is returned. This method
     * blocks until input data is available, the end of the stream is detected,
     * or an exception is thrown.
     *
     * <p> A subclass must provide an implementation of this method.
     *
     * @return the next byte of data, or <code>-1</code> if the end of the
     * stream is reached.
     * @throws IOException if an I/O error occurs.
     */
    @Override
    public int read() throws IOException {
        return Byte.toUnsignedInt(this.bytes.remove(0));
    }

    /**
     * Returns an estimate of the number of bytes that can be read (or
     * skipped over) from this input stream without blocking by the next
     * invocation of a method for this input stream. The next invocation
     * might be the same thread or another thread.  A single read or skip of this
     * many bytes will not block, but may read or skip fewer bytes.
     *
     * <p> Note that while some implementations of {@code InputStream} will return
     * the total number of bytes in the stream, many will not.  It is
     * never correct to use the return value of this method to allocate
     * a buffer intended to hold all data in this stream.
     *
     * <p> A subclass' implementation of this method may choose to throw an
     * {@link IOException} if this input stream has been closed by
     * invoking the {@link #close()} method.
     *
     * <p> The {@code available} method for class {@code InputStream} always
     * returns {@code 0}.
     *
     * <p> This method should be overridden by subclasses.
     *
     * @return an estimate of the number of bytes that can be read (or skipped
     * over) from this input stream without blocking or {@code 0} when
     * it reaches the end of the input stream.
     * @throws IOException if an I/O error occurs.
     */
    public int available() throws IOException {
        return this.bytes.size();
    }

    /**
     * Closes this input stream and releases any system resources associated
     * with the stream.
     *
     * <p> The <code>close</code> method of <code>InputStream</code> does
     * nothing.
     *
     * @throws IOException if an I/O error occurs.
     */
    public void close() throws IOException {
        try {
            this.readThread.run = false;
            this.readThread.usbPipe.close();
            Thread.sleep(100);
            this.readThread = null;
            this.bytes = null;
        } catch (Exception e) {
            throw new IOException(MeterStateEnum.USB_IN关闭异常.toString());
        }
    }
}
