package com.jhscale.meter.lora;

import com.jhscale.common.utils.ByteUtils;
import com.jhscale.meter.entity.MeterQueue;
import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.exp.MeterStateEnum;
import com.jhscale.meter.io.PortManager;
import com.jhscale.meter.io.control.DeviceControl;
import com.jhscale.meter.io.listener.DeviceClientEventListener;
import com.jhscale.meter.model.device.Device;
import io.swagger.annotations.ApiModelProperty;

import java.math.BigDecimal;
import java.util.*;

public class LoRaHost implements DeviceClientEventListener {

    public static BigDecimal GetNumS32(int value) {
        return new BigDecimal(value >> 3).movePointLeft(value & 0x07);
    }

    public static int GetU16(byte[] input, int offset) {
        return (((int) input[offset]) & 0xff) + (((int) input[offset + 1]) & 0xff) * 0x100;
    }

    public static int GetS32(byte[] input, int offset) {
        return (((int) input[offset]) & 0xff) + (((int) input[offset + 1]) & 0xff) * 0x100 + (((int) input[offset + 2]) & 0xff) * 0x10000 + (((int) input[offset + 3]) & 0xff) * 0x1000000;
    }

    public static BigDecimal GetNumS32(byte[] input, int offset) {
        if (input.length < offset + 4)
            return null;
        return GetNumS32((((int) input[offset]) & 0xff) + (((int) input[offset + 1]) & 0xff) * 0x100 + (((int) input[offset + 2]) & 0xff) * 0x10000 + (((int) input[offset + 3]) & 0xff) * 0x1000000);
    }

    private static final int HostAddr = 256;

    private LoRaHost() {
    }

    private static class SingleLoRaHost {
        private static final LoRaHost SINGLETON = new LoRaHost();
    }

    public static LoRaHost getInstance() {
        return SingleLoRaHost.SINGLETON;
    }

    @ApiModelProperty(value = "从设备列表", name = "received_Valid")
    private Map<Integer, LoRaDevice> devices = new HashMap<>();
    @ApiModelProperty(value = "LoRa参数", name = "para")
    public LoRaPara para = new LoRaPara();

    // 通讯器
    private PortManager messenger;
    // 发送器
    private LoRaThread thread;

    /**
     * @description: 构建通讯器(默认超时2秒)
     **/
    public boolean Init_Manager(DeviceControl control, Device device, long timeout) throws MeterException {
        this.thread = new LoRaThread(timeout);
        this.thread.start();
        this.messenger = new PortManager<>(control, device, this);
        this.messenger.openPort();
        return true;
    }

    /**
     * @description: 停止通讯器
     **/
    public boolean Stop_Manager() throws MeterException {
        if (this.thread != null) {
            this.thread.run = false;
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {
            }
            this.thread = null;
        }
        if (this.messenger != null) {
            this.messenger.closePort();
            this.messenger = null;
        }
        return true;
    }

    /**
     * @description: 添加发送
     **/
    public void Send(LoRaSend send) {
        this.thread.addSend(send);
    }

    public LoRaDevice getDevice(int id) {
        return this.devices.get(id);
    }

    /**
     * @description: 添加设备
     **/
    public boolean addDevice(LoRaDevice... devices) {
        if (devices != null && devices.length > 0) {
            return this.addDevice(Arrays.asList(devices));
        }
        return false;
    }

    /**
     * @description: 添加设备
     **/
    public boolean addDevice(List<LoRaDevice> devices) {
        if (devices != null && !devices.isEmpty()) {
            List<LoRaDevice> temp = new ArrayList<>();
            for (LoRaDevice device : devices) {
                int addr = device.getAddr();
                if ((addr >= 1) && (addr <= 31)) {
                    temp.add(device);
                }
            }
            if (!temp.isEmpty()) {
                temp.stream().forEach(d -> {
                    this.devices.put(d.getAddr(), d);
                });
                return true;
            }
        }
        return false;
    }

    public void clearDevices() {
        devices.clear();
    }

    private Map<Integer, LoRaResponse> taskbuf = new HashMap<>();

    private void TaskBuf_Clear() {
        taskbuf.clear();
    }

    private void TaskBuf_Put(int addrCmd, LoRaResponse lrp) {
        taskbuf.put(addrCmd, lrp);
    }

    /**
     * @description: 响应结果数据
     **/
    private void Receive_Deal(int addr, int command, byte[] data) {
        LoRaReceive receive = new LoRaReceive(addr, command, data);
        LoRaResponse response = taskbuf.get(receive.getAddrCmd());
        if (response != null) {
            response.response(receive);
        } else if (this.thread.waitACK && (this.thread.loRaSend != null) && (this.thread.loRaSend.getResponse() != null) && (this.thread.loRaSend.getCmd() == receive.getCmd()) && ((this.thread.loRaSend.getAddr() == 0) || (this.thread.loRaSend.getAddr() == receive.getAddr()))) {
            this.thread.loRaSend.getResponse().response(receive);
            this.thread.waitACK = false;
        }
    }

    private short[] buffer = new short[65536];
    private int buffer_stt = 0;
    private int buffer_end = 0;

    public void Receive(byte[] bytes) {
        for (int loop = 0; loop < bytes.length; loop++) {
            int nextpos;
            buffer[buffer_end] = (short) (((int) bytes[loop]) & 0xff);
            nextpos = (buffer_end + 1) & 0xffff;
            if (nextpos != buffer_stt)
                buffer_end = nextpos;
            else
                break;
        }

        while (true) {
            int length = (buffer_end - buffer_stt) & 0xffff;
            if (length < 4)
                break;
            if (buffer[buffer_stt] != 0xf2)
                buffer_stt = (buffer_stt + 1) & 0xffff;
            else {
                int type = buffer[(buffer_stt + 1) & 0xffff];
                int cnlng = buffer[(buffer_stt + 2) & 0xffff];
                if (length < cnlng + 4)
                    break;
                else if (buffer[(buffer_stt + cnlng + 3) & 0xffff] != 0xf3)
                    buffer_stt = (buffer_stt + 1) & 0xffff;
                else {
                    if (type == 'R') {
                        int head = buffer[(buffer_stt + 3) & 0xffff];
                        if (((head & 0xc0) == 0) && ((head & 0x1f) > 0)) {
                            int pos;
                            int cmd = buffer[(buffer_stt + 4) & 0xffff];
                            if ((cmd & 0x80) != 0) {
                                cmd = (cmd & 0x7f) | (buffer[(buffer_stt + 5) & 0xffff] << 7) + 0x80;
                                pos = 3;
                            } else {
                                pos = 2;
                            }
                            int cmdlng = cnlng - pos;
                            if (cmdlng >= 0) {
                                byte[] cmdbuf = new byte[cmdlng];
                                for (int loop = 0; loop < cmdlng; loop++)
                                    cmdbuf[loop] = (byte) buffer[(buffer_stt + pos + 3 + loop) & 0xffff];

                                //远端响应数据
                                Receive_Deal(head, cmd, cmdbuf);

                            }
                        }
                    } else if (type == 'Q') {
                        if (cnlng >= 1) {
                            int cmd = buffer[(buffer_stt + 3) & 0xffff];
                            int cmdlng = cnlng - 1;
                            byte[] cmdbuf = new byte[cmdlng];
                            for (int loop = 0; loop < cmdlng; loop++)
                                cmdbuf[loop] = (byte) buffer[(buffer_stt + 4 + loop) & 0xffff];

                            // 模块响应数据
                            Receive_Deal(this.HostAddr, cmd, cmdbuf);
                        } else {
                            Init_Again();
                        }
                    }
                    buffer_stt = (buffer_stt + cnlng + 4) & 0xffff;
                }
            }
        }
    }

    public void Task_Clear() {
        LoRaHost.getInstance().Send(new LoRaSend(new byte[]{(byte) 0xf2, 'T', 0, (byte) 0xf3}));
        LoRaHost.getInstance().TaskBuf_Clear();
    }

    public void Task_All() {
        for (LoRaDevice value : LoRaHost.getInstance().devices.values()) {
            value.Task_Routine();
        }
    }

    public void CMD_Version() {
        for (LoRaDevice value : LoRaHost.getInstance().devices.values()) {
            value.CMD_Version();
        }
    }

    public void CMD_ChangeDevicePara(int addr_old, int addr_new, LoRaPara para, boolean checkInfo) {
        LoRaHost.getInstance().Send(
                new LoRaSend(addr_old, 0x8000, null, new LoRaResponse() {
                    @Override
                    public void response(LoRaReceive receive) {
                        int addr_mid;
                        addr_mid = receive.getAddr();
                        if (checkInfo) {
                            LoRaDevice device = LoRaHost.getInstance().getDevice(addr_new);

//                            if(device instanceof LoRaDevice_EZY) {
//                            } else {
//                            }

                            if (device == null)
                                System.out.println(String.format("失败：未知地址"));
                            if (!device.FirmwareIsOK(receive.getData()))
                                System.out.println(String.format("失败：设备信息不匹配"));
                        }

                        LoRaHost.getInstance().Send(
                                new LoRaSend(addr_mid, 0x8001, para.getPara(addr_new), new LoRaResponse() {
                                    @Override
                                    public void response(LoRaReceive receive) {
                                        System.out.println(String.format("成功：修改设备信息"));
                                    }

                                    @Override
                                    public void exp(MeterException e) {
                                        System.out.println(String.format("请求异常：[%s]", e.getMessage()));
                                    }
                                })
                        );
                    }

                    @Override
                    public void exp(MeterException e) {
                        System.out.println(String.format("请求异常：[%s]", e.getMessage()));
                    }
                })
        );
    }

    private byte[] initPara = null;
    private LoRaAction initAction = null;

    public void Init_RunAction() {
        if (initAction != null)
            initAction.execute();
    }

    /**
     * @description: 以相应参数初始化
     **/
    public void Init_AsPara(byte[] para, LoRaAction action) {
        initPara = para;
        initAction = action;
        LoRaHost.getInstance().Send(
                new LoRaSend(LoRaHost.HostAddr, 'I', para, new LoRaResponse() {
                    @Override
                    public void response(LoRaReceive receive) {
                        if (LoRaHost.getInstance().para.setPara(receive.getData())) {
                            System.out.println(String.format("成功：启动Host"));
                            LoRaHost.getInstance().Init_RunAction();
                        } else
                            System.out.println(String.format("失败：启动Host返参不详"));
                    }

                    @Override
                    public void exp(MeterException e) {
                        System.out.println(String.format("请求异常：[%s]", e.getMessage()));
                    }
                })
        );
    }

    /**
     * @description: 以默认参数在指定频段初始化
     **/
    public void Init_AsWorking(int freqType, LoRaAction action) {
        Init_AsPara(new byte[]{(byte) (freqType & 0xff), (byte) ((freqType >> 8) & 0xff)}, action);
    }

    /**
     * @description: 在配对信道初始化
     **/
    public void Init_AsMatching(LoRaAction action) {
        Init_AsPara(null, action);
    }

    /**
     * @description: 以上次的参数初始化
     **/
    public void Init_Again() {
        Init_AsPara(initPara, initAction);
    }

    /**
     * @param bytes
     * @description: 连接监听
     */
    @Override
    public void onClientEvent(byte[] bytes) throws MeterException {
        System.out.println(String.format("监听结果数据：[%s]", ByteUtils.bytes2HexString(bytes)));
        this.Receive(bytes);
    }

    /**
     * @param e
     * @description: 连接异常监听
     */
    @Override
    public void onClientEventExp(MeterException e) {
        System.out.println(String.format("监听结果异常：[%s]", e.getMessage()));
    }

    /**
     * @description: 发送线程
     **/
    public class LoRaThread extends Thread {
        private boolean run = true;
        private boolean waitACK = false;
        //private MeterQueue<LoRaReceive> received;
        private MeterQueue<LoRaSend> sending;
        private LoRaSend loRaSend;
        private long time_out = 2000;// 超时时间 默认2s

        public LoRaThread(long timeout) {
            this.sending = new MeterQueue<>();
            //this.received = new MeterQueue<>();
            if (timeout > 0) this.time_out = timeout;
        }

        /**
         * @description: 添加发送指令
         **/
        public void addSend(LoRaSend send) {
            this.sending.enqueue(send);
        }

        @Override
        public void run() {
            while (this.run) {
                if (!this.waitACK) {
                    // 队列取数据发送
                    if (!this.sending.isEmpty()) {
                        LoRaSend dequeue = this.sending.dequeue();
                        if (dequeue == null) continue;
                        // 回调器
                        LoRaResponse response = dequeue.getResponse();

                        try {
                            if (response != null) {
                                if (dequeue.getTimeout() >= 0) {
                                    this.waitACK = true;
                                    this.loRaSend = dequeue;
                                } else {
                                    TaskBuf_Put(dequeue.getAddrCmd(), response);
                                }
                            }

                            System.out.println(String.format("发送内容数据：[%s]", ByteUtils.bytes2HexString(dequeue.getData())));
                            messenger.writeDataImmediately(dequeue.getData());
                        } catch (MeterException e) {
                            e.printStackTrace();
                            if (response != null)
                                response.exp(e);
                            this.waitACK = false;
                        }
                    }
                } else {
                    if (this.loRaSend.isTimeout()) {
                        this.loRaSend.getResponse().exp(new MeterException(MeterStateEnum.通讯响应超时));
                        this.waitACK = false;
                    }
                }

                // 线程休眠5ms
                try {
                    Thread.sleep(5);
                } catch (InterruptedException e) {
                }
            }
        }
    }

}
