package com.jhscale.meter.mqtt;

import com.jhscale.common.model.simple.JSONModel;
import com.jhscale.meter.log.JLog;
import com.jhscale.meter.mqtt.annotations.COrder;
import com.jhscale.meter.mqtt.em.Encoding;
import com.jhscale.meter.mqtt.entity.Mark;
import com.jhscale.meter.utils.ByteUtils;
import com.jhscale.meter.utils.MQTTUtils;
import org.apache.commons.lang3.StringUtils;

import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.nio.charset.Charset;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author lie_w
 * @title: IMQTT
 * @projectName meter-jar
 * @description: MQTT协议
 * @date 2021/12/23:48
 */
public class IMQTT extends JSONModel {

    /**
     * @description: 字符集
     **/
    private Charset charset;

    /**
     * @description: 内容体
     **/
    private StringBuffer buffer;

    /**
     * @description: 组装构造函数
     **/
    public IMQTT() {
        this.buffer = new StringBuffer();
    }

    public IMQTT(Charset charset) {
        this.charset = charset;
        this.buffer = new StringBuffer();
    }

    /**
     * @description: 解析构造函数
     **/
    public IMQTT(String innerContent) {
        this(Encoding.GBK.getCharset(), innerContent);
    }

    public IMQTT(Charset charset, String innerContent) {
        this.charset = charset;
        this.buffer = new StringBuffer(innerContent);
    }

    /**
     * @description: 字符集 内容 拷贝
     **/
    protected void copy(IMQTT imqtt) {
        this.charset = imqtt.charset;
        this.buffer = imqtt.buffer;
    }

    /**
     * @description: 字符集 内容 拷贝
     **/
    protected void copy(Charset charset, String data) {
        this.charset = charset;
        this.buffer = new StringBuffer(data);
    }

    /**
     * @description: 缓冲区清空
     **/
    protected void clear() {
        this.buffer.setLength(0);
    }

    /**
     * @description: 存在下个
     **/
    public boolean hasNext() {
        return this.buffer.length() > 0;
    }

    /**
     * @description: 内容体组装
     **/
    public IMQTT packSubContent() {
        return this;
    }

    /**
     * @param mark
     * @description: 内容体解析
     */
    public IMQTT unpackSubContent(Mark mark) {
        return null;
    }

    /**
     * @description: 解析填充
     **/
    public void fillingSub() {
    }

    /**
     * @description: 字符串缓冲区截取
     **/
    private String substring(int index) {
        String substring = this.buffer.substring(0, index);
        this.buffer = new StringBuffer(this.buffer.substring(index));
        return substring;
    }

    /**
     * @description: Integer to Sys Hex
     **/
    public String int2SHex(Integer val, int length) {
        return ByteUtils.convert(ByteUtils.appendFill(Integer.toHexString(val), length));
    }

    /**
     * @description: Sys Hex to Integer
     **/
    public Integer shex2Int(String hex) {
        if (StringUtils.isBlank(hex)) hex = "00";
        if (hex.length() % 2 != 0) hex = "0" + hex;
        return Integer.parseInt(ByteUtils.convert(hex), 16);
    }

    /**
     * @description: 组装不定长文本
     **/
    public IMQTT append(String text) {
        text = StringUtils.isBlank(text) ? "" : text;
        byte[] bytes = Objects.isNull(this.charset) ? text.getBytes() : text.getBytes(this.charset);
        String hexText = StringUtils.isBlank(text) ? "" : ByteUtils.toHexAscii(bytes);
        this.buffer.append(hexText).append("00");
        return this;
    }

    /**
     * @description: 组装不定长文本
     **/
    public IMQTT appendNull(String text) {
        return StringUtils.isNotBlank(text) ? this.append(text) : this;
    }

    /**
     * @description: 解析不定长文本
     **/
    public String parseText() {
        int index = ByteUtils.indexOf(this.buffer.toString(), "00");
        if (index == -1) return "";
        return new String(ByteUtils.fromHexAscii(this.substring(index + 2)), this.charset).trim();
    }

    /**
     * @description: 组装时间
     **/
    public IMQTT append(Date date) {
        long timeDefault = 946656000000L;
        long timeNow = Objects.nonNull(date) ? date.getTime() : 0;
        long second = (timeNow - timeDefault) / 1000;
        String hexTime = ByteUtils.completeBytes(Long.toHexString(second), 8);
        this.buffer.append(hexTime);
        return this;
    }

    /**
     * @description: 解析时间
     **/
    public Date parseDate() {
        return ByteUtils.convertTime(this.substring(8));
    }

    /**
     * @description: 组装特殊浮点数
     **/
    public IMQTT append(BigDecimal val) {
        val = Objects.nonNull(val) ? val : BigDecimal.ZERO;
        int i = ByteUtils.toSPF2(val);
        this.buffer.append(ByteUtils.completeBytes(Long.toHexString(i), 8));
        return this;
    }

    /**
     * @description: 组装特殊浮点数
     **/
    public IMQTT appendNull(BigDecimal val) {
        return Objects.nonNull(val) ? this.append(val) : this;
    }

    /**
     * @description: 解析特殊浮点数
     **/
    public BigDecimal parseBigDecimal() {
        return ByteUtils.convertBigDecimal(this.substring(8));
    }

    /**
     * @description: 组装字节
     **/
    public IMQTT append(Integer val, int len) {
        return Objects.nonNull(val) ? this.append(Integer.toHexString(val), len) : this;
    }

    /**
     * @description: 添加字符
     **/
    public IMQTT append(char character) {
        return this.append(character, 2);
    }

    /**
     * 添加指定长度字节
     */
    private IMQTT append(long val, int length) {
        this.buffer.append(ByteUtils.completeBytes(Long.toHexString(val), length));
        return this;
    }

    /**
     * @description: 组装字节
     **/
    public IMQTT append(String hex, int len) {
        return this.appendCovert(ByteUtils.appendFill(hex, len));
    }

    /**
     * @description: 转义颠倒HEX
     **/
    public IMQTT appendCovert(String hex) {
        return this.appendHex(ByteUtils.convert(hex));
    }

    /**
     * @description: 直接组装指定长度Hex
     **/
    public IMQTT appendHex(String hex) {
        if (StringUtils.isNotBlank(hex))
            this.buffer.append(hex);
        return this;
    }

    /**
     * @description: 直接组装指定长度Hex
     **/
    public IMQTT appendTopHex(String hex) {
        if (StringUtils.isNotBlank(hex))
            this.buffer.insert(0, hex);
        return this;
    }

    /**
     * @description: 添加BIN 指令长度
     **/
    public IMQTT appendBin(String bin, int len) {
        this.buffer.append(this.int2SHex(MQTTUtils.bit2Int(bin), len));
        return this;
    }

    /**
     * @description: 解析指令长度Bin
     **/
    protected String parseBin(int len) {
        String bit = MQTTUtils.hex2Bit(this.parseCovert(len));
        return ByteUtils.appendFill(bit, 16);
    }

    /**
     * @description: 添加命令
     **/
    public IMQTT appendCmd(String cmd) {
        return this.appendCovert(cmd);
    }

    /**
     * @description: 添加命令
     **/
    public String parseCmd() {
        return this.parseCovert(4);
    }

    /**
     * @description: 转义HEX
     **/
    public String parseCovert(int len) {
        return ByteUtils.convert(this.parseHex(len));
    }

    /**
     * @description: 直接获取指定长度Hex
     **/
    public String parseHex(int len) {
        return this.substring(len);
    }

    public String parseHex() {
        return this.substring(this.buffer.length());
    }

    /**
     * @description: 解析字节
     **/
    public int parseInt(int len) {
        return Integer.parseInt(this.parseCovert(len), 16);
    }

    /**
     * @description: 解析字符
     **/
    public char parseChar() {
        return (char) this.parseInt(2);
    }

    /**
     * @description: 解析二进制
     **/
    public String parseBit() {
        return MQTTUtils.hex2Bit(this.parseHex(2));
    }

    /**
     * @description: 组装浮动标志
     **/
    public IMQTT appendFloat() {
        List<Character> chars1 = new ArrayList<>();
        List<Character> chars2 = new ArrayList<>();
        List<Character> chars3 = new ArrayList<>();
        List<Character> chars4 = new ArrayList<>();
        List<Character> chars5 = new ArrayList<>();
        List<Character> chars6 = new ArrayList<>();
        List<Character> chars7 = new ArrayList<>();
        List<Character> chars8 = new ArrayList<>();
        List<Character> chars9 = new ArrayList<>();

        List<String> params = this.paramArray();

        for (int i = 0; i < 63; i++) {
            if (i < 7) {
                MQTTUtils.bitPart(chars1, params, i);
            }
            if (i >= 7 && i < 15) {
                MQTTUtils.bitPart(chars3, params, i);
            }
            if (i >= 15 && i < 23) {
                MQTTUtils.bitPart(chars4, params, i);
            }
            if (i >= 23 && i < 31) {
                MQTTUtils.bitPart(chars5, params, i);
            }
            if (i >= 31 && i < 39) {
                MQTTUtils.bitPart(chars6, params, i);
            }
            if (i >= 39 && i < 47) {
                MQTTUtils.bitPart(chars7, params, i);
            }
            if (i >= 47 && i < 55) {
                MQTTUtils.bitPart(chars8, params, i);
            }
            if (i >= 55) {
                MQTTUtils.bitPart(chars9, params, i);
            }
        }

        MQTTUtils.collect(chars2, chars3);
        MQTTUtils.collect(chars2, chars4);
        MQTTUtils.collect(chars2, chars5);
        MQTTUtils.collect(chars2, chars6);
        MQTTUtils.collect(chars2, chars7);
        MQTTUtils.collect(chars2, chars8);
        MQTTUtils.collect(chars2, chars9);
        chars2.add('0');
        MQTTUtils.collect(chars1, chars2);

        StringBuffer sb = new StringBuffer();
        sb.append(ByteUtils.dataFormat(MQTTUtils.floatArr2Hex(chars1)));
        if (chars2.contains('1')) {
            sb.append(ByteUtils.dataFormat(MQTTUtils.floatArr2Hex(chars2)));
        }

        if (chars3.contains('1')) {
            sb.append(ByteUtils.dataFormat(MQTTUtils.floatArr2Hex(chars3)));
        }

        if (chars4.contains('1')) {
            sb.append(ByteUtils.dataFormat(MQTTUtils.floatArr2Hex(chars4)));
        }

        if (chars5.contains('1')) {
            sb.append(ByteUtils.dataFormat(MQTTUtils.floatArr2Hex(chars5)));
        }

        if (chars6.contains('1')) {
            sb.append(ByteUtils.dataFormat(MQTTUtils.floatArr2Hex(chars6)));
        }

        if (chars7.contains('1')) {
            sb.append(ByteUtils.dataFormat(MQTTUtils.floatArr2Hex(chars7)));
        }

        if (chars8.contains('1')) {
            sb.append(ByteUtils.dataFormat(MQTTUtils.floatArr2Hex(chars8)));
        }

        if (chars9.contains('1')) {
            sb.append(ByteUtils.dataFormat(MQTTUtils.floatArr2Hex(chars9)));
        }

        this.buffer.append(sb);
        return this;
    }

    /**
     * @description: 解析浮动标志
     **/
    public char[] parseFloat() {
        char[] chars = {'0', '0', '0', '0', '0', '0', '0', '0',
                '0', '0', '0', '0', '0', '0', '0', '0',
                '0', '0', '0', '0', '0', '0', '0', '0',
                '0', '0', '0', '0', '0', '0', '0', '0',
                '0', '0', '0', '0', '0', '0', '0', '0',
                '0', '0', '0', '0', '0', '0', '0', '0',
                '0', '0', '0', '0', '0', '0', '0', '0',
                '0', '0', '0', '0', '0', '0', '0', '0',
                '0', '0', '0', '0', '0', '0', '0', '0'};

        String bit1 = MQTTUtils.hex2Bit(this.substring(2));
        for (int i = 0; i < bit1.length(); i++) {
            chars[i] = bit1.charAt(i);
        }

        if (bit1.charAt(bit1.length() - 1) == '1') {

            String bit2 = MQTTUtils.hex2Bit(this.substring(2));

            for (int i = 0; i < bit2.length(); i++) {
                char c = bit2.charAt(i);
                chars[8 + i] = c;
                if (c == '1') {
                    String bit3 = MQTTUtils.hex2Bit(this.substring(2));
                    for (int j = 0; j < bit3.length(); j++) {
                        chars[8 * (i + 2) + j] = bit3.charAt(j);
                    }
                }
            }
        }
        return chars;
    }

    /**
     * @description: 浮动标志 参数内容
     **/
    public List<String> paramArray() {
        List<String> ignoreParams = this.ignoreParams();
        List<Field> fields = super.allFieldList(ignoreParams.toArray(new String[ignoreParams.size()]));

        if (fields == null || fields.isEmpty()) return new ArrayList<>();

        List<Field> orderFields = new ArrayList<>();
        for (Field field : fields) {
            COrder annotation = field.getAnnotation(COrder.class);
            if (annotation != null) {
                orderFields.add(field);
            }
        }

        if (orderFields == null || orderFields.isEmpty()) return new ArrayList<>();

        Collections.sort(orderFields, new Comparator<Field>() {
            @Override
            public int compare(Field o1, Field o2) {
                return o1.getAnnotation(COrder.class).order() - o2.getAnnotation(COrder.class).order();
            }
        });

        String[] array = new String[orderFields.size()];
        for (int i = 0; i < orderFields.size(); i++) {
            Field field = orderFields.get(i);
            String str = null;
            try {
                field.setAccessible(true);
                Object val = field.get(this);
                str = Objects.isNull(val) ? null : String.valueOf(val);
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            }
            array[i] = str;
        }
        return Stream.of(array).collect(Collectors.toList());
    }

    /**
     * @description: 浮动参数忽略的参数
     **/
    public List<String> ignoreParams() {
        return new ArrayList<>();
    }

    /**
     * @description: 组装完成
     **/
    public String over() {
        JLog.debug("IMQTT buffer:{}", this.buffer);
        return this.buffer.toString();
    }

    public Charset getCharset() {
        return charset;
    }

    public void setCharset(Charset charset) {
        this.charset = charset;
    }

    public StringBuffer getBuffer() {
        return buffer;
    }

    public void setBuffer(StringBuffer buffer) {
        this.buffer = buffer;
    }
}
