package com.jhscale.meter.mqtt;

import com.jhscale.common.utils.Base64Utils;
import com.jhscale.common.utils.JSONUtils;
import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.exp.MeterStateEnum;
import com.jhscale.meter.log.JLog;
import com.jhscale.meter.mqtt.em.*;
import com.jhscale.meter.mqtt.entity.MQTTClient;
import com.jhscale.meter.mqtt.entity.Mark;
import com.jhscale.meter.mqtt.entity.ModuleInfo;
import com.jhscale.meter.utils.AesHexUtils;
import com.jhscale.meter.utils.ByteUtils;
import com.jhscale.meter.utils.CrcUtils;
import com.jhscale.meter.utils.MQTTUtils;
import org.apache.commons.lang3.StringUtils;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

/**
 * @author lie_w
 * @title: MQTTPackBuilder
 * @projectName meter-jar
 * @description: TODO
 * @date 2021/12/313:02
 */
public class MQTTPackBuilder extends MQTTBuilder {

    // 加密状态
    private Encrypt encrypt;

    // 请求响应标志
    private ACK ack;

    // 请求响应状态
    private Confirm confirm;

    // 指令标识
    private CD cd;

    // 缓存Key
    private String cacheKey;

    // 客户端
    private MQTTClient mqttClient;

    // MQTT加载器
    private MQTTLoader loader;

    // 指令缓冲区
    private StringBuffer buffer;

    // 指令缓冲区
    private List<IMQTT> imqtts;

    public MQTTPackBuilder() {
        buffer = new StringBuffer();
    }

    /**
     * @param encrypt 加密状态
     * @param ack     请求响应方式
     * @param confirm 指令状态
     * @param cd      指令类型
     * @description: Mark
     **/
    public MQTTPackBuilder mark(Encrypt encrypt, ACK ack, Confirm confirm, CD cd) {
        this.encrypt = encrypt;
        this.ack = ack;
        this.confirm = confirm;
        this.cd = cd;
        return this;
    }

    /**
     * @param mark 头信息
     * @description: Mark
     **/
    public MQTTPackBuilder mark(Mark mark) {
        this.encrypt = mark.getEncrypt();
        this.ack = mark.getAck();
        this.confirm = mark.getConfirm();
        this.cd = mark.getCd();
        return this;
    }

    /**
     * @param encrypt 加密状态
     * @param answer  应答状态 true 期待应答 false 不期待应答
     * @param cd      指令类型
     * @description: 请求Mark
     **/
    public MQTTPackBuilder request(Encrypt encrypt, boolean answer, CD cd) {
        return this.mark(encrypt, ACK.Request, answer ? Confirm.WAIT : Confirm.NWAIT, cd);
    }

    /**
     * @param encrypt 加密状态
     * @param result  处理结果 true 成功 false 失败
     * @param cd      指令类型
     * @description: 响应Mark
     **/
    public MQTTPackBuilder response(Encrypt encrypt, boolean result, CD cd) {
        return this.mark(encrypt, ACK.Response, result ? Confirm.SUCCESS : Confirm.FAIL, cd);
    }

    /**
     * @param mark 请求Mark信息
     * @description: 响应Mark
     **/
    public MQTTPackBuilder response(Mark mark, boolean result) {
        return this.mark(mark.getEncrypt(), ACK.Response, result ? Confirm.SUCCESS : Confirm.FAIL, mark.getCd());
    }

    /**
     * @description: MQTT客户端
     **/
    public MQTTPackBuilder mqttClient(MQTTClient mqttClient) {
        this.mqttClient = mqttClient;
        return this;
    }

    /**
     * @description: MQTT加载器
     **/
    public MQTTPackBuilder loader(MQTTLoader loader) {
        this.loader = loader;
        return this;
    }

    /**
     * @description: 添加内容体
     **/
    public MQTTPackBuilder add(IMQTT... imqtts) {
        if (this.imqtts == null) this.imqtts = new ArrayList<>();
        this.imqtts.addAll(Arrays.asList(imqtts));
        return this;
    }

    /**
     * @description: 直接添加内容体
     **/
    public MQTTPackBuilder add(String cacheKey, String sunContent) {
        this.cacheKey = cacheKey;
        super.setSubContent(sunContent);
        return this;
    }

    /**
     * @description: 指令内容
     **/
    public MQTTPackBuilder spack() throws MeterException {
        if (Objects.isNull(this.mqttClient))
            throw new MeterException(MeterStateEnum.MQTT客户端信息不存在);

        Mark mark = new Mark(this.encrypt, this.ack, this.confirm, this.cd);

        // 发送内容体
        String nidContent = ByteUtils.convert(MQTTUtils.int2Hex2(this.mqttClient.getNid()));
        JLog.debug("Command build NidContent:{}--{}", this.mqttClient.getNid(), nidContent);

        if ((StringUtils.isBlank(this.mqttClient.getKey()) || Objects.isNull(this.mqttClient.getCharset())) && Objects.nonNull(this.loader)) {
            ModuleInfo moduleInfo = null;
            if (StringUtils.isNotBlank(this.mqttClient.getUid()))
                moduleInfo = ((ServerMQTTLoader) this.loader).keyAndEncoding(this.mqttClient.getUid());

            if (Objects.isNull(moduleInfo))
                throw new MeterException(MeterStateEnum.模块信息加载失败);

            if (StringUtils.isBlank(this.mqttClient.getKey()))
                this.mqttClient.appendKey(moduleInfo.getKey());

            if (Objects.isNull(this.mqttClient.getCharset()))
                this.mqttClient.appendEncoding(moduleInfo.getCoding());
        }

        // 构建内容
        this.addInner();

        // 加密内容
        this.encrypt(mark);

        int length = buffer.length();
        String lenContent = ByteUtils.convert(MQTTUtils.int2Hex((length) / 2));
        JLog.debug("Command assemble LenContent:{}--{}", length, lenContent);

        mark.setLen(lenContent.length());
        String markContent = mark.assembleMark();
        JLog.debug("Command build Mark:{}--{}", mark.toJSON(), markContent);

        buffer.insert(0, lenContent)// LEN
                .insert(0, nidContent)// NID
                .insert(0, markContent);// Mark
        String crcContent = CrcUtils.crc(buffer.toString());// CRC
        JLog.debug("Command build CRCContent:{}--{}", buffer.toString(), crcContent);

        buffer.insert(0, MQTTConstant.TOP)
                .append(crcContent)
                .append(MQTTConstant.END);
        return this;
    }

    /**
     * @description: 指令内容
     **/
    public MQTTPackBuilder lpack() throws MeterException {
        if (Objects.isNull(this.mqttClient))
            throw new MeterException(MeterStateEnum.MQTT客户端信息不存在);

        Mark mark = new Mark(this.encrypt, this.ack, this.confirm, this.cd);

        // 发送内容体
        String nidContent = ByteUtils.convert(MQTTUtils.int2Hex2(this.mqttClient.getNid()));
        JLog.debug("Command build NidContent:{}--{}", this.mqttClient.getNid(), nidContent);

        String uidContent = "";// UID
        if (!CD.MODULE.equals(mark.getCd()) && StringUtils.isNotBlank(this.mqttClient.getUid())) {
            uidContent = ByteUtils.toHexString(this.mqttClient.getUid().getBytes()) + "00";
            JLog.debug("Command build UidContent:{}--{}", this.mqttClient.getUid(), uidContent);
        }

        if ((StringUtils.isBlank(this.mqttClient.getKey()) || Objects.isNull(this.mqttClient.getCharset())) && Objects.nonNull(this.loader)) {
            ModuleInfo moduleInfo = null;
            if (StringUtils.isNotBlank(this.mqttClient.getUid()))
                moduleInfo = ((LocalMQTTLoader) this.loader).keyAndEncoding();

            if (Objects.isNull(moduleInfo) && !CD.MODULE.equals(mark.getCd()))
                throw new MeterException(MeterStateEnum.模块信息加载失败);

            if (Objects.nonNull(moduleInfo)) {
                if (StringUtils.isBlank(this.mqttClient.getKey()))
                    this.mqttClient.appendKey(moduleInfo.getKey());

                if (Objects.isNull(this.mqttClient.getCharset()))
                    this.mqttClient.appendEncoding(moduleInfo.getCoding());
            }
        }

        // 添加内容
        this.addInner();

        // 加密
        this.encrypt(mark);

        int length = uidContent.length() + buffer.length();
        String lenContent = ByteUtils.convert(MQTTUtils.int2Hex((length) / 2));
        JLog.debug("Command assemble LenContent:{}--{}", length, lenContent);

        mark.setLen(lenContent.length());
        String markContent = mark.assembleMark();
        JLog.debug("Command build Mark:{}--{}", mark.toJSON(), markContent);

        buffer.insert(0, uidContent)// UID
                .insert(0, lenContent)// LEN
                .insert(0, nidContent)// NID
                .insert(0, markContent);// Mark
        String crcContent = CrcUtils.crc(buffer.toString());// CRC
        JLog.debug("Command build CRCContent:{}--{}", buffer.toString(), crcContent);

        buffer.insert(0, MQTTConstant.TOP)
                .append(crcContent)
                .append(MQTTConstant.END);
        return this;
    }

    /**
     * @description: 添加内容端
     **/
    private void addInner() throws MeterException {
        // 构建内容
        if (StringUtils.isNotBlank(this.getSubContent())) {
            super.setInnerContent(super.getSubContent());
            this.buffer = new StringBuffer(super.getSubContent());
            if (StringUtils.isBlank(this.cacheKey)) {
                this.cacheKey = this.cd.getBit();
            }
        } else {
            this.buildSubIMQTT();
        }
    }

    /**
     * @description: 加密内容
     **/
    private void encrypt(Mark mark) throws MeterException {
        if (Encrypt.AES_Encrypt.equals(mark.getEncrypt())) {
            if (StringUtils.isBlank(this.mqttClient.getKey()))
                throw new MeterException(MeterStateEnum.MQTT_AES密钥不存在);

            // 需要加密
            int innerLength = this.buffer.length() / 2;
            int fillingLength = 16 - innerLength % 16;
            for (int i = 1; i <= fillingLength; i++) {
                buffer.append(MQTTUtils.int2Hex(i));
            }
            JLog.debug("Command build AES_Encrypt:{}", buffer.toString());

            // AES加密
            JLog.debug("Command unpack encrypt before:{}", this.buffer.toString());
            this.buffer = new StringBuffer(AesHexUtils.encrypt(this.buffer.toString(), this.mqttClient.getKey()));
            JLog.debug("Command unpack encrypt after:{}", this.buffer.toString());
        }
    }

    /**
     * @description: 组装子内容
     **/
    public static String packSubContent(List<IMQTT> imqtts, Charset charset, boolean base64) {
        if (charset == null) charset = Encoding.GBK.getCharset();
        if (imqtts == null || imqtts.isEmpty()) return null;
        // 拼装内容
        StringBuffer subContent = new StringBuffer();
        for (IMQTT imqtt : imqtts) {
            if (Objects.isNull(imqtt.getCharset()))
                imqtt.setCharset(charset);
            if (imqtt instanceof IDATA) {
                subContent.append(((IDATA) imqtt).packStream().over());
            } else {
                subContent.append(imqtt.packSubContent().over());
            }
        }

        return base64 ? Base64Utils.gzipString(subContent.toString()) : subContent.toString();
    }

    /**
     * @description: 构建子内容
     **/
    private void buildSubIMQTT() throws MeterException {
        if (Objects.isNull(this.cd))
            throw new MeterException(MeterStateEnum.MQTT指令类型未添加);

        if (imqtts == null || imqtts.size() == 0)
            throw new MeterException(MeterStateEnum.MQTT数据包不存在);

        StringBuffer subContent = new StringBuffer();
        if ((CD.MODULE.equals(this.cd) || CD.CMD.equals(this.cd))) {
            if (imqtts.size() != 1)
                throw new MeterException(MeterStateEnum.MQTT数据包错误);

            if (!(this.imqtts.get(0) instanceof ICMD))
                throw new MeterException(MeterStateEnum.MQTT内容指令与类型不配);
            IMQTT imqtt = this.imqtts.get(0);
            if (Objects.isNull(imqtt.getCharset()))
                imqtt.setCharset(this.mqttClient.getCharset());
            subContent.append(this.imqtts.get(0).packSubContent().over());
            this.cacheKey = ((ICMD) this.imqtts.get(0)).getCmd().getCmd();
        } else {
            // 判断其他类型
            // 拼装内容
            for (IMQTT imqtt : imqtts) {
                if (Objects.isNull(imqtt.getCharset()))
                    imqtt.setCharset(this.mqttClient.getCharset());

                if (imqtt instanceof IDATA) {
                    subContent.append(((IDATA) imqtt).packStream().over());
                    this.cacheKey = ((IDATA) imqtt).getDsort().getSort();
                } else {
                    subContent.append(imqtt.packSubContent().over());
                }
            }
        }
        this.buffer.append(subContent);
        super.setInnerContent(subContent.toString());
        JLog.debug("Command assemble SubContent:{}--{}", JSONUtils.objectToJSON(imqtts), subContent.toString());
    }

    /**
     * @description: 完结
     **/
    public String over() {
        return this.buffer.toString();
    }

    /**
     * @description: 串口数据流
     **/
    public byte[] stream() {
        return ByteUtils.fromHexString(this.over());
    }

    /**
     * @description: 指令缓存Key
     **/
    @Override
    public String cacheKey() {
        return this.cd.getVal() + (StringUtils.isBlank(this.cacheKey) ? "" : (":" + this.cacheKey));
    }

    @Override
    public String command() {
        return StringUtils.isNotBlank(this.cacheKey) ? this.cacheKey : this.cd.getVal().toString();
    }
}
