package com.jhscale.meter.mqtt;

import com.jhscale.common.utils.Base64Utils;
import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.exp.MeterStateEnum;
import com.jhscale.meter.log.JLog;
import com.jhscale.meter.mqtt.data.DataResponse;
import com.jhscale.meter.mqtt.em.ACK;
import com.jhscale.meter.mqtt.em.CD;
import com.jhscale.meter.mqtt.em.Encoding;
import com.jhscale.meter.mqtt.em.Encrypt;
import com.jhscale.meter.mqtt.entity.MQTTClient;
import com.jhscale.meter.mqtt.entity.Mark;
import com.jhscale.meter.utils.AesHexUtils;
import com.jhscale.meter.utils.ByteUtils;
import com.jhscale.meter.utils.CrcUtils;
import com.jhscale.meter.utils.MQTTUtils;
import org.apache.commons.lang3.StringUtils;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * @author lie_w
 * @title: MQTTPackBuilder
 * @projectName meter-jar
 * @description: TODO
 * @date 2021/12/313:02
 */
public class MQTTUnpackBuilder extends MQTTBuilder {

    // 指令缓冲区
    private StringBuffer buffer;

    // 加载器
    private MQTTLoader loader;

    // 加密状态
    private Mark mark;

    // 客户端
    private MQTTClient mqttClient;

    // 内容信息
    private List<IMQTT> imqtts;

    // 缓存Key
    private String cacheKey;

    /**
     * @description: 添加解析指令
     **/
    public MQTTUnpackBuilder content(String content) throws MeterException {
        if (StringUtils.isBlank(content) || !content.startsWith(MQTTConstant.TOP) || !content.endsWith(MQTTConstant.END))
            throw new MeterException(MeterStateEnum.MQTT数据包格式不正确);
        buffer = new StringBuffer(content);
        return this;
    }

    /**
     * @description: MQTT加载器
     **/
    public MQTTUnpackBuilder loader(MQTTLoader loader) {
        this.loader = loader;
        return this;
    }

    /**
     * @description: 服务器拆包 包中包含UID
     **/
    public MQTTUnpackBuilder sunpackWithoutSub() throws MeterException {
        // 拆包Mark
        this.unpackMark();

        // NID编号
        this.mqttClient = new MQTTClient();
        this.mqttClient.appendNid(this.unpackNid());

        String lenContent = this.buffer.substring(8, 8 + mark.getLen());
        int length = MQTTUtils.hex2Int(ByteUtils.convertHex(lenContent)) * 2;
        JLog.debug("Command unpack Len:{}--{}", lenContent, length);

        String innerContent = this.buffer.substring(8 + mark.getLen(), 8 + mark.getLen() + length);
        JLog.debug("Command unpack Uid + Content:{}", innerContent);

        // 拆包UID
        innerContent = unpackSUid(innerContent);

        if (this.loader != null && StringUtils.isNotBlank(this.mqttClient.getUid())) {
            this.mqttClient.appendModule(((ServerMQTTLoader) this.loader).keyAndEncoding(this.mqttClient.getUid()));
        }

        JLog.debug("Command unpack MQTTClient:{}", this.mqttClient.toJSON());

        innerContent = this.encrypt(innerContent);
        super.setInnerContent(innerContent);

        String crc = this.buffer.substring(8 + mark.getLen() + length, 10 + mark.getLen() + length);
        String crcContent = CrcUtils.crc(this.buffer.substring(2, this.buffer.length() - 4));
        JLog.debug("Command unpack crc:{}, crcContent:{}", crc, crcContent);
        if (!crc.equalsIgnoreCase(crcContent))
            throw new MeterException(MeterStateEnum.MQTT_CRC检查错误);

        super.setSubContent(innerContent);
        return this;
    }

    /**
     * @description: 服务器拆包 包中包含UID
     **/
    public MQTTUnpackBuilder sunpack() throws MeterException {
        return this.sunpackWithoutSub()
                .unpackSubContent();
    }

    /**
     * @description: 本地拆包 包中不包含UID
     **/
    public MQTTUnpackBuilder lunpack() throws MeterException {
        // 拆包Mark
        this.unpackMark();

        // NID编号
        this.mqttClient = new MQTTClient();
        this.mqttClient.appendNid(this.unpackNid());

        String lenContent = this.buffer.substring(8, 8 + mark.getLen());
        int length = MQTTUtils.hex2Int(ByteUtils.convertHex(lenContent)) * 2;
        JLog.debug("Command unpack Len:{}--{}", lenContent, length);

        String innerContent = this.buffer.substring(8 + mark.getLen(), 8 + mark.getLen() + length);
        JLog.debug("Command unpack Uid + Content:{}", innerContent);

        if (this.loader != null) {
            this.mqttClient.appendModule(((LocalMQTTLoader) this.loader).keyAndEncoding());
        }
        JLog.debug("Command unpack MQTTClient:{}", this.mqttClient.toJSON());

        innerContent = this.encrypt(innerContent);

        super.setInnerContent(innerContent);

        String crc = this.buffer.substring(8 + mark.getLen() + length, 10 + mark.getLen() + length);
        String crcContent = CrcUtils.crc(this.buffer.substring(2, this.buffer.length() - 4));
        JLog.debug("Command unpack crc:{}, crcContent:{}", crc, crcContent);
        if (!crc.equalsIgnoreCase(crcContent))
            throw new MeterException(MeterStateEnum.MQTT_CRC检查错误);

        super.setSubContent(innerContent);
        return this.unpackSubContent();
    }

    /**
     * @description: 拆包Mark
     **/
    public MQTTUnpackBuilder unpackMark() throws MeterException {
        if (Objects.isNull(this.buffer) || this.buffer.length() == 0)
            throw new MeterException(MeterStateEnum.MQTT解析数据包不存在);

        String markContent = this.buffer.substring(2, 4);
        this.mark = new Mark(markContent);
        JLog.debug("Command unpack Mark:{}--{}", markContent, mark.toJSON());
        if (Objects.isNull(mark))
            throw new MeterException(MeterStateEnum.Mark解析失败);
        return this;
    }

    /**
     * @description: 拆包客户端
     **/
    public Integer unpackNid() {
        String nidContent = ByteUtils.convert(this.buffer.substring(4, 8));
        Integer nid = Integer.parseInt(nidContent, 16);
        JLog.debug("Command unpack NID:{}--{}", nidContent, nid);
        return nid;
    }

    /**
     * @description: 服务端拆包UID
     **/
    private String unpackSUid(String innerContent) {
        int index = ByteUtils.indexOf(innerContent, "00");
        if (index != -1) {
            String hexText = innerContent.substring(0, index);
            innerContent = innerContent.substring(index + 2);
            String uid = new String(ByteUtils.fromHexAscii(hexText)).trim();
            this.mqttClient.appendUid(uid);
            JLog.debug("Command unpack Uid:{}--{}", hexText, uid);
        }
        return innerContent;
    }

    /**
     * @description: 解析子内容
     **/
    public static List<IMQTT> unpackSubContent(String subContent, Charset charset, boolean base64) {
        if (base64) {
            subContent = Base64Utils.ungzipString(subContent);
        }

        if (charset == null) charset = Encoding.GBK.getCharset();

        IDATA idata = new IDATA(charset, subContent);

        List<IMQTT> imqtts = new ArrayList<>();
        Mark mark = new Mark();
        mark.setAck(ACK.Request);
        while (idata.hasNext()) {
            IMQTT mqtt = idata.unpackSubContent(mark);
            if (Objects.nonNull(mqtt))
                imqtts.add(mqtt);
        }
        return imqtts;
    }

    /**
     * @description: 解析子内容体
     **/
    private MQTTUnpackBuilder unpackSubContent() throws MeterException {
        IMQTT imqtt = null;
        switch (this.mark.getCd()) {
            case CMD:
                imqtt = new ICMD(this.mqttClient.getCharset(), super.getSubContent());
                break;
            case MODULE:
                imqtt = new ICMD(this.mqttClient.getCharset(), super.getSubContent());
                break;
            case API:
                imqtt = new IDATA(this.mqttClient.getCharset(), super.getSubContent());
                break;
            case DATA:
                imqtt = new IDATA(this.mqttClient.getCharset(), super.getSubContent());
                break;
        }

        if (imqtt == null)
            throw new MeterException(MeterStateEnum.MQTT指令类型不存在);

        this.imqtts = new ArrayList<>();
        while (imqtt.hasNext()) {
            IMQTT mqtt = imqtt.unpackSubContent(mark);
            if (Objects.nonNull(mqtt))
                imqtts.add(mqtt);
        }

        if (((this.mark.getCd().equals(CD.MODULE) || this.mark.getCd().equals(CD.CMD)) && this.imqtts.size() > 0 && this.imqtts.get(0) != null)) {
            this.cacheKey = ((ICMD) this.imqtts.get(0)).getCmd().getCmd();
            // 重置子内容体
            super.setSubContent(null);
        } else if (this.imqtts.size() > 0 && this.imqtts.get(0) != null && !((this.imqtts.get(0)) instanceof DataResponse)) {
            this.cacheKey = ((IDATA) this.imqtts.get(0)).getDsort().getSort();
            // 重置子内容体
            super.setSubContent(null);
        }
        return this;
    }

    /**
     * @description: 解密操作
     **/
    private String encrypt(String innerContent) throws MeterException {
        if (Encrypt.AES_Encrypt.equals(mark.getEncrypt())) {

            if (StringUtils.isBlank(this.mqttClient.getKey()))
                throw new MeterException(MeterStateEnum.MQTT_AES密钥不存在);

            JLog.debug("Command unpack decrypt before:{}", innerContent);
            try {
                innerContent = AesHexUtils.decrypt(innerContent, this.mqttClient.getKey());
            } catch (MeterException e) {
                throw new MeterException(MeterStateEnum.AES解密失败);
            }
            JLog.debug("Command unpack decrypt after:{}", innerContent);

            // 需要获取到密钥完成解密
            int innerLength = innerContent.length();
            int fillingVal = Integer.parseInt(innerContent.substring(innerLength - 2), 16);
            for (int i = 0; i < fillingVal; i++) {
                int fillingPartVal = Integer.parseInt(innerContent.substring(innerLength - (i * 2) - 2, innerLength - (i * 2)), 16);
                if (fillingPartVal != fillingVal - i)
                    throw new MeterException(MeterStateEnum.填充字段错误);
            }
            innerContent = innerContent.substring(0, innerLength - fillingVal * 2);
            JLog.debug("Command unpack AES_Decrypt:{}", innerContent);
        }
        return innerContent;
    }

    public Mark getMark() {
        return mark;
    }

    public MQTTClient getMqttClient() {
        return mqttClient;
    }

    public List<IMQTT> getImqtts() {
        return imqtts;
    }

    /**
     * @description: 指令缓存Key
     **/
    @Override
    public String cacheKey() {
        return this.mark.getCd().getVal() + (StringUtils.isBlank(this.cacheKey) ? "" : (":" + this.cacheKey));
    }

    @Override
    public String command() {
        return StringUtils.isNotBlank(this.cacheKey) ? this.cacheKey : this.mark.getCd().getVal().toString();
    }
}
