package com.jhscale.meter.protocol;

import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.exp.MeterStateEnum;
import com.jhscale.meter.io.PortManager;
import com.jhscale.meter.io.control.DeviceControl;
import com.jhscale.meter.io.listener.DeviceClientEventListener;
import com.jhscale.meter.log.JLog;
import com.jhscale.meter.log.PortLogBack;
import com.jhscale.meter.model.device.Device;
import com.jhscale.meter.protocol.ad.entity.ADPackAssemblyRequest;
import com.jhscale.meter.protocol.entity.IPackRequest;
import com.jhscale.meter.protocol.entity.IPackResponse;
import com.jhscale.meter.protocol.entity.PackAssemblyRequest;
import com.jhscale.meter.protocol.entity.ProtocolEntity;
import com.jhscale.meter.protocol.model.GlobalPara;
import com.jhscale.meter.utils.ByteUtils;

import java.util.Objects;

/**
 * @author lie_w
 * @title: AbstractProtoManager
 * @projectName meter-jar
 * @description: TODO
 * @date 2023/5/2713:36
 */
public abstract class AbstractProtoManager<U extends PackAssemblyRequest> implements IProtocolManager<U> {

    // 通讯器
    protected PortManager portManager;

    // 通讯器打开状态
    protected boolean dopen = false;

    /**
     * @param portManager
     * @description: 初始化通讯器（采用主动方式读取响应信息）
     * 通讯器实现类                           通讯器名称           安卓          WIN10
     * com.jhscale.meter.io.USBPort           USB通讯器            支持          缺少驱动
     * com.jhscale.meter.io.SerialPort        串口通讯器           支持          支持
     * com.jhscale.meter.io.BluetoothPort     蓝牙通讯器           支持          支持，需要支持蓝牙
     * com.jhscale.meter.io.NFCPort           NFC通讯器            不支持        不支持
     * com.jhscale.meter.io.EthernetPort      WIFI通讯器           不支持        不支持
     */
    @Override
    public IProtocolManager initPortmanager(PortManager portManager) {
        this.portManager = portManager;
        return this;
    }

    /**
     * @param deviceControl
     * @param device
     * @description: 初始化通讯器(不初始化回调 升级初始化)
     */
    @Override
    public IProtocolManager initPortmanager(DeviceControl deviceControl, Device device) throws MeterException {
        return this.initPortmanager(new PortManager(deviceControl, device));
    }

    /**
     * @param portManager
     * @param listener
     * @description: 初始化通讯器 指定监听器
     */
    @Override
    public IProtocolManager initPortmanager(PortManager portManager, DeviceClientEventListener listener) {
        if (Objects.nonNull(listener) && Objects.nonNull(portManager)) {
            portManager.setClientEventListener(listener);
        }
        return this.initPortmanager(portManager);
    }

    /**
     * @param request
     * @description: 命令处理（拼装）
     */
    @Override
    public <T extends IPackResponse> T execute(IPackRequest<T> request) throws MeterException {
        return request.execute();
    }

    /**
     * @param request
     * @param protocolResponse
     * @description: 发送命令
     */
    @Override
    public void execute(U request, IProtocolResponse protocolResponse) throws MeterException {
        this.checkManager();
        this.checkPort();
        IPackResponse execute = this.execute(request);
        JLog.debug("Execute: {}", execute.toJSON());
        DeviceClientEventListener clientEventListener = this.portManager.getClientEventListener();
        if (clientEventListener != null) {
            clientEventListener.addProtocolResponse(execute.KEY(), new ProtocolEntity(request, protocolResponse));
        }
        if (GlobalPara.getInstance().isRunLog()) {
            try {
                System.out.printf("%s 发送 %s%n", ((ADPackAssemblyRequest) request).getAdcmd().getContent(), ByteUtils.toHexString(execute.HEX()));
            } catch (Exception e) {
            }
        }
        this.portManager.writeDataImmediately(execute.HEX());
    }

    /**
     * @param request
     * @description: 发送命令(无实际响应 ， 只返回发送成功与否)
     */
    @Override
    public boolean executeSendOnly(U request) throws MeterException {
        IPackResponse execute = request.execute();
        JLog.debug("Execute: {}", execute.toJSON());
        return this.execute(execute.HEX());
    }

    /**
     * @param bytes
     * @description: 使用默认通讯器直接发送数据
     */
    @Override
    public boolean execute(byte[] bytes) throws MeterException {
        checkManager();
        checkPort();
        if (this.portManager.getClientEventListener() == null)
            throw new MeterException(MeterStateEnum.AD_未初始化回调监听);
        this.portManager.writeDataImmediately(bytes);
        return true;
    }

    /**
     * @description: 打开端口
     **/
    @Override
    public void openPort() throws MeterException {
        checkManager();
        this.dopen = this.portManager.openPort();
    }

    /**
     * @description: 关闭端口
     **/
    @Override
    public void closePort() throws MeterException {
        if (this.portManager != null)
            this.dopen = !this.portManager.closePort();
    }

    /**
     * @description: 重启通讯
     */
    @Override
    public boolean restart() throws MeterException {
        if (this.dopen) {
            IBProtocolResponse ibProtocolResponse = this.portManager.getClientEventListener().defaultResponse();
            if (ibProtocolResponse == null) return false;
            this.initPortmanager(this.portManager, ibProtocolResponse);
            this.dopen = this.portManager.openPort();
            return this.dopen;
        } else {
            return false;
        }
    }

    /**
     * @param log
     * @description: 开启日志
     */
    @Override
    public void log(boolean log) throws MeterException {
        checkManager();
        this.portManager.log(log);
    }

    /**
     * @param logBack
     * @param portLogBack
     * @description: 日志输出回调
     */
    @Override
    public void logBack(boolean logBack, PortLogBack portLogBack) throws MeterException {
        checkManager();
        this.portManager.logBack(logBack, portLogBack);
    }


    /**
     * @description: 检查通讯器
     **/
    protected boolean checkManager() throws MeterException {
        if (this.portManager == null)
            throw new MeterException(MeterStateEnum.通讯器未初始化);
        return true;
    }

    /**
     * @description: 检查端口打开
     **/
    protected boolean checkPort() throws MeterException {
        if (!this.dopen)
            throw new MeterException(MeterStateEnum.端口未打开);
        return true;
    }

}
