package com.jhscale.meter.protocol;

import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.io.PortManager;
import com.jhscale.meter.io.control.DeviceControl;
import com.jhscale.meter.io.listener.DeviceClientEventListener;
import com.jhscale.meter.log.PortLogBack;
import com.jhscale.meter.model.device.Device;
import com.jhscale.meter.protocol.entity.IPackRequest;
import com.jhscale.meter.protocol.entity.IPackResponse;
import com.jhscale.meter.protocol.entity.PackAssemblyRequest;

import java.io.Serializable;

/**
 * @author lie_w
 * @title: IProtocolManager
 * @projectName meter-jar
 * @description: 协议通讯器
 * @date 2021/6/914:07
 */
public interface IProtocolManager<U extends PackAssemblyRequest> extends Serializable {

    /**
     * @description: 初始化通讯器（采用主动方式读取响应信息）
     * 通讯器实现类                           通讯器名称           安卓          WIN10
     * com.jhscale.meter.io.USBPort           USB通讯器            支持          缺少驱动
     * com.jhscale.meter.io.SerialPort        串口通讯器           支持          支持
     * com.jhscale.meter.io.BluetoothPort     蓝牙通讯器           支持          支持，需要支持蓝牙
     * com.jhscale.meter.io.NFCPort           NFC通讯器            不支持        不支持
     * com.jhscale.meter.io.EthernetPort      WIFI通讯器           不支持        不支持
     **/
    IProtocolManager initPortmanager(PortManager portManager);

    /**
     * @description: 初始化通讯器(不初始化回调 升级初始化)
     **/
    IProtocolManager initPortmanager(DeviceControl deviceControl, Device device) throws MeterException;

    /**
     * @description: 初始化通讯器 指定监听器
     **/
    IProtocolManager initPortmanager(PortManager portManager, DeviceClientEventListener listener);

    /**
     * @description: 初始化通讯器（采用回调方式读取响应信息）
     * 通讯器实现类                           通讯器名称           安卓          WIN10
     * com.jhscale.meter.io.USBPort           USB通讯器            支持          缺少驱动
     * com.jhscale.meter.io.SerialPort        串口通讯器           支持          支持
     * com.jhscale.meter.io.BluetoothPort     蓝牙通讯器           支持          支持，需要支持蓝牙
     * com.jhscale.meter.io.NFCPort           NFC通讯器            不支持        不支持
     * com.jhscale.meter.io.EthernetPort      WIFI通讯器           不支持        不支持
     **/
    IProtocolManager initPortmanager(PortManager portManager, IBProtocolResponse protocolResponse);

    /**
     * @description: 初始化通讯器（采用回调方式读取响应信息）
     **/
    IProtocolManager initPortmanager(DeviceControl control, Device device, IBProtocolResponse protocolResponse) throws MeterException;

    /**
     * @description: 命令处理（拼装|解析）
     **/
    <T extends IPackResponse> T execute(IPackRequest<T> request) throws MeterException;

    /**
     * @description: 发送命令
     **/
    void execute(U request, IProtocolResponse protocolResponse) throws MeterException;

    /**
     * @description: 发送命令(无实际响应 ， 只返回发送成功与否)
     **/
    boolean executeSendOnly(U request) throws MeterException;

    /**
     * @description: 使用默认通讯器直接发送数据
     **/
    boolean execute(byte[] bytes) throws MeterException;

    /**
     * @description: 添加接受指令内容
     **/
    default void addClientEvent(byte[] bytes) {
    }

    /**
     * @description: 固件升级请求握手
     **/
    void shake(IProtocolResponse protocolResponse) throws MeterException;

    /**
     * @description: 固件升级直接握手
     **/
    void shakeWithoutUpgradeFirmware(IProtocolResponse protocolResponse) throws MeterException;

    /**
     * @description: 固件升级
     **/
    void upgrade(byte[] bytes, IProtocolResponse protocolResponse) throws MeterException;

    /**
     * @description: 打开端口
     **/
    void openPort() throws MeterException;

    /**
     * @description: 关闭端口
     **/
    void closePort() throws MeterException;

    /**
     * @description: 重启通讯
     **/
    boolean restart() throws MeterException;

    /**
     * @description: 开启日志
     **/
    void log(boolean log) throws MeterException;

    /**
     * @description: 日志输出回调
     **/
    void logBack(boolean logBack, PortLogBack portLogBack) throws MeterException;
}
