package com.jhscale.meter.protocol.ad;

import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.exp.MeterStateEnum;
import com.jhscale.meter.io.PortManager;
import com.jhscale.meter.io.control.DeviceControl;
import com.jhscale.meter.log.JLog;
import com.jhscale.meter.model.device.Device;
import com.jhscale.meter.protocol.AbstractProtoManager;
import com.jhscale.meter.protocol.IBProtocolResponse;
import com.jhscale.meter.protocol.IProtocolManager;
import com.jhscale.meter.protocol.IProtocolResponse;
import com.jhscale.meter.protocol.ad.em.CommunicationState;
import com.jhscale.meter.protocol.ad.entity.ADPackAssemblyRequest;
import com.jhscale.meter.protocol.ad.entity.assembly.UpgradeFirmwareADPARequest;
import com.jhscale.meter.protocol.ad.entity.disassembly.UpgradeFirmwareADPDResponse;
import com.jhscale.meter.protocol.ad.listener.ADScaleClientEventListener;
import com.jhscale.meter.protocol.ad.listener.ADUpgradeEventListener;
import com.jhscale.meter.protocol.entity.ProtocolEntity;
import com.jhscale.meter.protocol.model.WeightCal;
import com.jhscale.meter.utils.FileUtils;

/**
 * @author lie_w
 * @title: ScaleProtocolManager
 * @projectName meter-jar
 * @description: 仪表操作
 * POS-2|蓝牙吊秤|POS-1
 * @date 2021/6/1019:27
 */
public class ADScaleProtocolManager extends AbstractProtoManager<ADPackAssemblyRequest> {

    /**
     * @param portManager
     * @param protocolResponse
     * @description: 初始化通讯器（采用回调方式读取响应信息）
     * 通讯器实现类                           通讯器名称           安卓          WIN10
     * com.jhscale.meter.io.USBPort           USB通讯器            支持          缺少驱动
     * com.jhscale.meter.io.SerialPort        串口通讯器           支持          支持
     * com.jhscale.meter.io.BluetoothPort     蓝牙通讯器           支持          支持，需要支持蓝牙
     * com.jhscale.meter.io.NFCPort           NFC通讯器            不支持        不支持
     * com.jhscale.meter.io.EthernetPort      WIFI通讯器           不支持        不支持
     */
    @Override
    public IProtocolManager initPortmanager(PortManager portManager, IBProtocolResponse protocolResponse) {
        return this.initPortmanager(portManager, new ADScaleClientEventListener(protocolResponse, this));
    }

    /**
     * @param deviceControl
     * @param device
     * @param protocolResponse
     * @description: 初始化通讯器（采用回调方式读取响应信息）
     */
    @Override
    public IProtocolManager initPortmanager(DeviceControl deviceControl, Device device, IBProtocolResponse protocolResponse) throws MeterException {
        return this.initPortmanager(new PortManager(deviceControl, device, new ADScaleClientEventListener(protocolResponse, this)));
    }

    /**
     * @param protocolResponse
     * @description: 固件升级握手
     */
    @Override
    public void shake(IProtocolResponse protocolResponse) throws MeterException {
        WeightCal.getInstance().setUpgrade(true);
        checkManager();
        if (this.dopen) this.portManager.closePort();

        ADUpgradeEventListener adUpgradeEventListener = new ADUpgradeEventListener(protocolResponse, this);

        this.portManager.setClientEventListener(adUpgradeEventListener);

        // 清空缓冲区
        adUpgradeEventListener.clear();
        // ADCache.getInstance().clear();

        this.dopen = this.portManager.openPort();

        this.execute(new UpgradeFirmwareADPARequest(), new IProtocolResponse<UpgradeFirmwareADPDResponse>() {
            @Override
            public void target(UpgradeFirmwareADPDResponse target) {
                JLog.debug("UpgradeFirmwareADPDResponse:{}", target.toJSON());
                if (target.getResult().isResult()) { // 重置缓冲模式
                    adUpgradeEventListener.setCommunication(CommunicationState.SHAKE);
                }
            }

            @Override
            public void exp(MeterException e) {
                if (protocolResponse != null)
                    protocolResponse.exp(e);
            }
        });
    }

    /**
     * @param protocolResponse
     * @description: 固件升级握手
     */
    @Override
    public void shakeWithoutUpgradeFirmware(IProtocolResponse protocolResponse) throws MeterException {
        WeightCal.getInstance().setUpgrade(true);
        checkManager();
        if (this.dopen) this.portManager.closePort();
        ADUpgradeEventListener adUpgradeEventListener = new ADUpgradeEventListener(protocolResponse, this);
        this.portManager.setClientEventListener(adUpgradeEventListener);
        // 清空缓冲区
        adUpgradeEventListener.clear();
        this.dopen = this.portManager.openPort();
        adUpgradeEventListener.setCommunication(CommunicationState.SHAKE);
    }

    /**
     * @param bytes
     * @param protocolResponse
     * @description: 固件升级
     */
    @Override
    public void upgrade(byte[] bytes, IProtocolResponse protocolResponse) throws MeterException {
        WeightCal.getInstance().setUpgrade(true);
        checkManager();
        checkPort();
        ADUpgradeEventListener clientEventListener = (ADUpgradeEventListener) this.portManager.getClientEventListener();
        if (clientEventListener.getCommunication() != CommunicationState.SHAKE)
            throw new MeterException(MeterStateEnum.AD_固件升级未握手);
        // 设置发送握手指令
        clientEventListener.setShake_state(false);
        // 设置升级状态
        clientEventListener.setCommunication(CommunicationState.UPGRATE);
        // 添加回调器
        clientEventListener.addProtocolResponse(CommunicationState.UPGRATE.getKey(), new ProtocolEntity(protocolResponse));
        // 添加缓冲区
        clientEventListener.upgradeWithCtrl(FileUtils.upgradeBytes(bytes), this.portManager);
    }
}
