package com.jhscale.meter.protocol.ad.entity.assembly;

import com.jhscale.common.model.inter.JSONModel;
import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.protocol.ad.em.ADCMD;
import com.jhscale.meter.protocol.ad.em.DigitalTube2;
import com.jhscale.meter.protocol.ad.entity.ADPackAssemblyRequest;
import com.jhscale.meter.protocol.ad.entity.ADPackAssemblyResponse;
import com.jhscale.meter.utils.ByteUtils;

import java.util.List;

/**
 * @author Loveven
 * @title: OpenCloseStrandardADPARequest
 * @projectName meter-jar
 * @description: 后显示设定命令(组装请求)
 * @date 2021-06-1312:02
 */
public class SetBackDisplay2ADPARequest extends ADPackAssemblyRequest<ADPackAssemblyResponse> {

    /**
     * @description: 数值显示
     **/
    private DisplayVal displayVal;

    private DisplaySignLeft left;

    private DisplaySignRight right;

    private DisplaySignTail tail;

    /**
     * @description: 不为空 长度为3 才使用此字段
     * 三段内容 第一段6长度 第二段7长度 第三段8长度
     **/
    private List<String> parts;

    /**
     * @description: 三HEX
     **/
    private List<String> signs;

    public SetBackDisplay2ADPARequest() {
        super(ADCMD.Set_Back_Display);
    }

    public SetBackDisplay2ADPARequest(DisplayVal displayVal, DisplaySignLeft left, DisplaySignRight right, DisplaySignTail tail) {
        this();
        this.displayVal = displayVal;
        this.left = left;
        this.right = right;
        this.tail = tail;
    }

    public SetBackDisplay2ADPARequest(List<String> parts, List<String> signs) {
        this();
        this.parts = parts;
        this.signs = signs;
    }

    @Override
    public void assembly() throws MeterException {
        if (this.parts != null && this.parts.size() == 3) {
            this.inner.append(DigitalTube2.digitalTube(12, this.parts.get(0)))
                    .append(DigitalTube2.digitalTube(14, this.parts.get(1)))
                    .append(DigitalTube2.digitalTube(16, this.parts.get(2)));
        } else if (this.displayVal != null) {
            this.inner.append(DigitalTube2.digitalTube(12, this.displayVal.weight))
                    .append(DigitalTube2.digitalTube(14, this.displayVal.price))
                    .append(DigitalTube2.digitalTube(16, this.displayVal.item));
        }
        if (this.signs != null && this.signs.size() == 3) {
            this.signs.stream().forEach(sign -> {
                this.inner.append(sign);
            });
        } else {
            if (this.left == null) this.left = new DisplaySignLeft();
            this.inner.append(this.left.hex());
            if (this.right == null) this.right = new DisplaySignRight();
            this.inner.append(this.right.hex());
            if (this.tail == null) this.tail = new DisplaySignTail();
            this.inner.append(this.tail.hex());
        }
    }

    public DisplayVal getDisplayVal() {
        return displayVal;
    }

    public void setDisplayVal(DisplayVal displayVal) {
        this.displayVal = displayVal;
    }

    public List<String> getParts() {
        return parts;
    }

    public void setParts(List<String> parts) {
        this.parts = parts;
    }

    public List<String> getSigns() {
        return signs;
    }

    public void setSigns(List<String> signs) {
        this.signs = signs;
    }

    /**
     * @description: 显示屏数值
     **/
    public static class DisplayVal implements JSONModel {
        /**
         * @description: 重量
         **/
        private String weight;

        /**
         * @description: 单价
         **/
        private String price;

        /**
         * @description: 小计
         **/
        private String item;

        public DisplayVal() {
        }

        public DisplayVal(String weight, String price, String item) {
            this.weight = weight;
            this.price = price;
            this.item = item;
        }

        public String getWeight() {
            return weight;
        }

        public DisplayVal setWeight(String weight) {
            this.weight = weight;
            return this;
        }

        public String getPrice() {
            return price;
        }

        public DisplayVal setPrice(String price) {
            this.price = price;
            return this;
        }

        public String getItem() {
            return item;
        }

        public DisplayVal setItem(String item) {
            this.item = item;
            return this;
        }
    }

    /**
     * @description: 显示屏标志
     **/
    public static interface DisplaySign extends JSONModel {

        /**
         * @description: 显示HEX值
         **/
        String hex();
    }

    /**
     * @description: 显示屏左标志
     **/
    public static class DisplaySignLeft implements DisplaySign {

        /**
         * @description: 净重
         **/
        private boolean net;

        /**
         * @description: 稳定
         **/
        private boolean stablize;

        /**
         * @description: 零点
         **/
        private boolean zero;

        /**
         * @description: 件
         **/
        private boolean pcs;

        private boolean g;
        private boolean lb;
        private boolean kg;
        private boolean oz;

        public DisplaySignLeft() {
        }

        public DisplaySignLeft(boolean net, boolean stablize, boolean zero, boolean pcs, boolean g, boolean lb, boolean kg, boolean oz) {
            this.net = net;
            this.stablize = stablize;
            this.zero = zero;
            this.pcs = pcs;
            this.g = g;
            this.lb = lb;
            this.kg = kg;
            this.oz = oz;
        }

        public boolean isNet() {
            return net;
        }

        public DisplaySignLeft setNet(boolean net) {
            this.net = net;
            return this;
        }

        public boolean isStablize() {
            return stablize;
        }

        public DisplaySignLeft setStablize(boolean stablize) {
            this.stablize = stablize;
            return this;
        }

        public boolean isZero() {
            return zero;
        }

        public DisplaySignLeft setZero(boolean zero) {
            this.zero = zero;
            return this;
        }

        public boolean isPcs() {
            return pcs;
        }

        public DisplaySignLeft setPcs(boolean pcs) {
            this.pcs = pcs;
            return this;
        }

        public boolean isG() {
            return g;
        }

        public DisplaySignLeft setG(boolean g) {
            this.g = g;
            return this;
        }

        public boolean isLb() {
            return lb;
        }

        public DisplaySignLeft setLb(boolean lb) {
            this.lb = lb;
            return this;
        }

        public boolean isKg() {
            return kg;
        }

        public DisplaySignLeft setKg(boolean kg) {
            this.kg = kg;
            return this;
        }

        public boolean isOz() {
            return oz;
        }

        public DisplaySignLeft setOz(boolean oz) {
            this.oz = oz;
            return this;
        }

        /**
         * @description: 显示HEX值
         **/
        @Override
        public String hex() {
            return ByteUtils.int2Hex((this.net ? 0x10 : 0x00)
                    + (this.stablize ? 0x20 : 0x00)
                    + (this.zero ? 0x40 : 0x00)
                    + (this.pcs ? 0x80 : 0x00)
                    + (this.lb ? 0x01 : 0x00)
                    + (this.kg ? 0x06 : 0x00)
                    + (this.g ? 0x04 : 0x00)
                    + (this.oz ? 0x08 : 0x00));
        }
    }

    /**
     * @description: 显示屏右标志
     **/
    public static class DisplaySignRight implements DisplaySign {
        private boolean lb;
        private boolean lb1_4;
        private boolean g;
        private boolean kg;
        private boolean g100;
        private boolean g500;
        private boolean qz;

        public DisplaySignRight() {
        }

        public DisplaySignRight(boolean lb, boolean lb1_4, boolean g, boolean kg, boolean g100, boolean g500, boolean qz) {
            this.lb = lb;
            this.lb1_4 = lb1_4;
            this.g = g;
            this.kg = kg;
            this.g100 = g100;
            this.g500 = g500;
            this.qz = qz;
        }

        public boolean isLb() {
            return lb;
        }

        public DisplaySignRight setLb(boolean lb) {
            this.lb = lb;
            return this;
        }

        public boolean isLb1_4() {
            return lb1_4;
        }

        public DisplaySignRight setLb1_4(boolean lb1_4) {
            this.lb1_4 = lb1_4;
            return this;
        }

        public boolean isG() {
            return g;
        }

        public DisplaySignRight setG(boolean g) {
            this.g = g;
            return this;
        }

        public boolean isKg() {
            return kg;
        }

        public DisplaySignRight setKg(boolean kg) {
            this.kg = kg;
            return this;
        }

        public boolean isG100() {
            return g100;
        }

        public DisplaySignRight setG100(boolean g100) {
            this.g100 = g100;
            return this;
        }

        public boolean isG500() {
            return g500;
        }

        public DisplaySignRight setG500(boolean g500) {
            this.g500 = g500;
            return this;
        }

        public boolean isQz() {
            return qz;
        }

        public DisplaySignRight setQz(boolean qz) {
            this.qz = qz;
            return this;
        }

        /**
         * @description: 显示HEX值
         **/
        @Override
        public String hex() {
            return ByteUtils.int2Hex((this.lb ? 0x01 : 0x00)
                    + (this.lb1_4 ? 0x02 : 0x00)
                    + (this.g ? 0x04 : 0x00)
                    + (this.kg ? 0x08 : 0x00)
                    + (this.g100 ? 0x40 : 0x00)
                    + (this.g500 ? 0x20 : 0x00)
                    + (this.qz ? 0x10 : 0x00));
        }
    }

    /**
     * @description: 显示屏底
     **/
    public static class DisplaySignTail implements DisplaySign {
        private boolean sign1;
        private boolean sign2;
        private boolean sign3;
        private boolean sign4;

        public DisplaySignTail() {
        }

        public DisplaySignTail(boolean sign1, boolean sign2, boolean sign3, boolean sign4) {
            this.sign1 = sign1;
            this.sign2 = sign2;
            this.sign3 = sign3;
            this.sign4 = sign4;
        }

        public boolean isSign1() {
            return sign1;
        }

        public DisplaySignTail setSign1(boolean sign1) {
            this.sign1 = sign1;
            return this;
        }

        public boolean isSign2() {
            return sign2;
        }

        public DisplaySignTail setSign2(boolean sign2) {
            this.sign2 = sign2;
            return this;
        }

        public boolean isSign3() {
            return sign3;
        }

        public DisplaySignTail setSign3(boolean sign3) {
            this.sign3 = sign3;
            return this;
        }

        public boolean isSign4() {
            return sign4;
        }

        public DisplaySignTail setSign4(boolean sign4) {
            this.sign4 = sign4;
            return this;
        }

        /**
         * @description: 显示HEX值
         **/
        @Override
        public String hex() {
            return ByteUtils.int2Hex((this.sign1 ? 0x08 : 0x00)
                    + (this.sign2 ? 0x04 : 0x00)
                    + (this.sign3 ? 0x02 : 0x00)
                    + (this.sign4 ? 0x01 : 0x00));
        }
    }
}
