package com.jhscale.meter.protocol.other.cmpt;

import com.jhscale.common.model.device._inner.PublicExecuteModify;
import com.jhscale.common.model.inter.DataJSONModel;
import com.jhscale.common.utils.ByteUtils;
import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.io.PortManager;
import com.jhscale.meter.io.TCPServer;
import com.jhscale.meter.io.UDPLink;
import com.jhscale.meter.io.control.SocketControl;
import com.jhscale.meter.io.listener.SocketClientEventListener;
import com.jhscale.meter.model.device.TCPServerDevice;
import com.jhscale.meter.model.device.UDPDevice;
import com.jhscale.meter.protocol.model.GlobalPara;
import com.jhscale.meter.protocol.model.SocketNotify;
import com.jhscale.meter.protocol.other.CMPT;

import java.net.InetAddress;
import java.util.Objects;
import java.util.Timer;
import java.util.TimerTask;

import static com.jhscale.common.model.device.DConstant.PLU;

/**
 * @author wang lie
 * @title: AbstractCMPTAction
 * @projectName meter-jar
 * @description: 兼容协议执行器(抽象类)
 * @date 2023/5/2116:11
 */
public abstract class ACMPTAction<T extends ACMPTAction> implements CMPTAction<ACMPTAction> {

    // TCP 服务端
    private TCPServer TCPServer;
    // UDP连接器
    private UDPLink UDPLink;

    // 协议信息
    private CMPT cmpt;
    // TCP Server控制器
    private SocketControl tcp_server_control;
    // UDP 控制器
    private SocketControl udp_control;
    // 回调通知
    private SocketNotify notify;

    // 缓冲区信息
    protected byte[] buffer_TCP = new byte[40960];
    protected int buffer_TCP_Pose = 0;
    protected long buffer_TCP_Tick = 0;

    /**
     * @param cmpt
     * @description: 加载协议
     */
    @Override
    public T Init_CMPT_Action(CMPT cmpt, SocketControl tcp_server_control, SocketControl udp_control, SocketNotify notify) {
        this.cmpt = cmpt;
        this.tcp_server_control = tcp_server_control;
        this.udp_control = udp_control;
        this.notify = notify;
        return (T) this;
    }

    /**
     * @description: 初始化加载器
     **/
    @Override
    public boolean Init_Action() throws MeterException {
        this.Init_TCP_Server();
        this.Init_UDP();
        return true;
    }

    /**
     * @description: 停止服务
     **/
    @Override
    public boolean Stop_Action() throws MeterException {
        if (this.TCPServer != null) {
            this.TCPServer.closePort();
            this.TCPServer = null;
        }
        if (this.UDPLink != null) {
            this.UDPLink.closePort();
            this.UDPLink = null;
        }
        return true;
    }

    /**
     * @description: 初始化 TCP_Server
     **/
    @Override
    public boolean Init_TCP_Server() throws MeterException {
        TCPServerDevice tcp_server_device = this.cmpt.getTcp_server_device();
        if (Objects.nonNull(tcp_server_device)) {
            this.TCPServer = new TCPServer(this.tcp_server_control, this.cmpt.getTcp_server_device(),
                    new SocketClientEventListener() {
                        /**
                         * @param address
                         * @param port
                         * @description: TCP Server 连接监听
                         */
                        @Override
                        public void onClientLink(InetAddress address, int port) {
                            if (GlobalPara.getInstance().isRunLog())
                                System.out.println("TCP Server onClientLink......");
                        }

                        /**
                         * @param address
                         * @param port
                         * @description: TCP Server断开监听
                         */
                        @Override
                        public void onClientBreak(InetAddress address, int port) {
                            if (GlobalPara.getInstance().isRunLog())
                                System.out.println("TCP Server onClientBreak......");
                        }

                        /**
                         * @param e
                         * @description: 连接异常监听
                         */
                        @Override
                        public void onClientEventExp(MeterException e) {
                            if (notify != null) notify.onClientEventExp(e);
                        }

                        /**
                         * @param portManager
                         * @param bytes
                         * @description: TCP Client 连接监听（带响应回复）
                         */
                        @Override
                        public byte[] onClientEventResponse(PortManager portManager, byte[] bytes) throws MeterException {
                            if (bytes != null && bytes.length > 0) {

                                if (GlobalPara.getInstance().isRunLog())
                                    System.out.printf("onClientEventResponse Content: %s%n", ByteUtils.bytes2HexString(bytes));

                                // TCP Server 监听到的内容
                                byte[] tcp_server = tcp_server(portManager, bytes);
                                if (tcp_server != null && tcp_server.length > 0)
                                    portManager.writeDataImmediately(tcp_server);
                            }
                            return new byte[0];
                        }
                    });
            return this.TCPServer.openPort();
        }
        return false;
    }

    /**
     * @description: 初始化 UDP
     **/
    @Override
    public boolean Init_UDP() throws MeterException {
        UDPDevice udp_device = this.cmpt.getUdp_device();
        if (Objects.nonNull(udp_device)) {
            this.UDPLink = new UDPLink(this.udp_control, this.cmpt.getUdp_device(),
                    new SocketClientEventListener() {
                        /**
                         * @param address 来源地址
                         * @param port    来源端口
                         * @param bytes   数据包
                         * @description: 连接监听（IP PORT）
                         **/
                        @Override
                        public byte[] onClientEvent(PortManager portManager, InetAddress address, int port, byte[] bytes) throws MeterException {
                            if (GlobalPara.getInstance().isRunLog())
                                System.out.println(String.format("收到来自:%s,对方端口号为:%s,消息内容：%s",
                                        address, port, ByteUtils.bytes2HexString(bytes)));

                            // UDP监听到的内容
                            byte[] udp = udp(portManager, address, port, bytes);
                            if (udp != null && udp.length > 0)
                                UDPLink.writeDataImmediately(udp);
                            return new byte[0];
                        }

                        /**
                         * @param e
                         * @description: 连接异常监听
                         */
                        @Override
                        public void onClientEventExp(MeterException e) {
                            if (notify != null) notify.onClientEventExp(e);
                        }
                    });
            return this.UDPLink.openPort();
        }
        return false;
    }

    /**
     * @description: 协议定义
     **/
    @Override
    public CMPT CMPT() {
        return this.cmpt;
    }

    private PublicExecuteModify execute;
    private final long delay = 3000;
    private Timer timer;

    /**
     * @description: 添加缓冲
     **/
    protected void addCache(DataJSONModel model) {
        if (this.execute == null) {
            this.execute = new PublicExecuteModify(PLU);
        }
        this.execute.add(model);
        if (this.timer != null) this.timer.cancel();
        this.timer = new Timer();
        this.timer.schedule(new TimerTask() {
            @Override
            public synchronized void run() {
                if (execute != null) {
                    execute.setSerial(cmpt.getSerial());
                    notify.acceptEvent(execute);
                    execute = null;
                }
            }
        }, delay);

    }

}
