package com.jhscale.meter.protocol.print;

import com.jhscale.common.utils.SystemtUtils;
import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.exp.MeterStateEnum;
import com.jhscale.meter.protocol.print.entity.PrintLOGParam;
import com.jhscale.meter.protocol.print.entity.PrintRequest;
import com.jhscale.meter.protocol.print.entity.PrintResponse;
import com.jhscale.meter.protocol.print.entity.ScheduleState;
import com.jhscale.meter.protocol.print.entity.cmd.Print0000Request;
import com.jhscale.meter.protocol.print.entity.cmd.Print0000Response;
import com.jhscale.meter.protocol.print.entity.cmd.Print0200Request;
import com.jhscale.meter.protocol.print.entity.cmd.Print0200Response;
import com.jhscale.meter.protocol.print.entity.file.PrintFileRequest;
import com.jhscale.meter.protocol.print.entity.file.bitmap.PrintBitMapRequest;
import com.jhscale.meter.protocol.print.entity.file.font.PrintFontRequest;
import com.jhscale.meter.protocol.print.entity.file.upgrade.PrintUpgradeRequest;
import com.jhscale.meter.protocol.print.entity.para.Para;
import com.jhscale.meter.protocol.print.link.IPrintBack;
import com.jhscale.meter.protocol.print.link.Messenger;
import com.jhscale.meter.protocol.print.produce.entity.PrintBackResponse;
import com.jhscale.meter.utils.PrintUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.List;
import java.util.Objects;

/**
 * @author lie_w
 * @title: PrintInterface
 * @projectName meter-jar
 * @description: TODO
 * @date 2022/10/2518:22
 */
public interface PrintCommunication {

    /**
     * @description: 通讯器序号
     **/

    void setSerial(Integer name);

    Integer getSerial();

    default PrintCommunication communication() {
        return this;
    }

    /**
     * @description: 检查通讯器正常状态
     **/
    boolean checkCommunication() throws MeterException;

    /**
     * @description: 通讯器
     **/
    Messenger messenger();

    /**
     * @description: 添加CMD 暂存异步发送
     **/
    void addSendCmd(String cmd, PrintRequest request) throws MeterException;

    /**
     * @description: 端口直接发送指令
     **/
    void sendCmd(String cmd, PrintRequest request) throws MeterException;

    /**
     * @description: 发送指令到缓冲区（内容组装已经完成的 直接获取内容发送）
     **/
    default void sendCmd(PrintRequest request) throws MeterException {
        this.checkCommunication();
        request.setCommunication(this);
        PrintUtils.debug(this.param(), String.format("PrintType [ %s ] Content[ %s ]", "Direct Send", request.getContent()));
        this.addSendCmd(request.cmd(), request);
    }

    /**
     * @description: 发送指令到缓冲区（完成内容组装）
     **/
    default void sendData(PrintRequest request) throws MeterException {
        this.checkCommunication();
        request.setCommunication(this);
        // 拼装数据
        String content = request.assembler().toString();
        if (StringUtils.isNotBlank(content)) request.setContent(content);
        PrintUtils.debug(this.param(), String.format("PrintType [ %s ] Content[ %s ]", request.printType().getBitVal(), request.getContent()));
        // 初始化发送
        this.addSendCmd(request.printType().init(), request);
    }

    /**
     * @description: 发送指令到缓冲区（内容不需要组装 直接发送初始化 发送指令）
     **/
    default void sendDataWithoutAssemble(PrintRequest request) throws MeterException {
        this.checkCommunication();
        request.setCommunication(this);
        PrintUtils.debug(this.param(), String.format("PrintType [ %s ] Content[ %s ]", request.printType().getBitVal(), request.getContent()));
        // 初始化发送
        this.addSendCmd(request.printType().init(), request);
    }

    /**
     * @description: 发送指令到缓冲区 (完成内容组装 发送全部)
     **/
    default void sendDataWithAllPackage(PrintRequest request) throws MeterException {
        this.checkCommunication();
        request.setCommunication(this);
        // 拼装数据
        String content = request.assembler().toString();
        if (StringUtils.isNotBlank(content)) request.setContent(content);
        PrintUtils.debug(this.param(), String.format("PrintType [ %s ] Content[ %s ]", "Direct Send", request.getContent()));
        request.setAll(true);
        request.setFirst(false);
        this.addSendCmd(null, request);
    }

    /**
     * @description: 添加打印部分内容
     **/
    void addParas(List<Para> paras) throws MeterException;

    /**
     * @description: 清空打印内容
     **/
    void clearPara();

    /**
     * @description: 全部打印内容
     **/
    List<Para> paras();


    /**
     * @description: 原始数据响应回调状态
     **/
    boolean originalResponse();

    /**
     * @description: 打印日志配置参数
     **/
    PrintLOGParam param();

    /**
     * @description: 设置节点日志参数
     **/
    void logSwitch(boolean _switch, Integer _length);

    /**
     * @description: 当前发送状态
     **/
    boolean isSendStatus();

    /**
     * @description: 发送完成
     **/
    void sendSuccess();

    /**
     * @description: 关闭通讯器
     **/
    void closeMessenger();


    /********************************** 升级 ******************************************/

    /**
     * @description: 检查升级文件有效性
     **/
    default List<Integer> getUpgradeFileParam(PrintFileRequest request) throws MeterException {
        // 获取升级文件名 并检查文件名
        request.checkUpgradeFile(request.getUpgradeFileName());
        return request.getNameParams();
    }

    /**
     * @description: 文件升级 (不做命令检查升级结果)
     **/
    default void upgrade(PrintFileRequest request) throws MeterException {
        // 关机保护
        Print0200Request print0200Request = new Print0200Request();
        print0200Request.setSdk(true);
        print0200Request.setUpgrade(true);
        print0200Request.setPrintBack(new IPrintBack() {
            /**
             * @param response
             * @description: 操作结果响应信息
             */
            @Override
            public void responseBack(PrintResponse response) {
                try {
                    // 关闭打印保护结果
                    if (((Print0200Response) response).isResult()) {
                        request.setUpgrade(true);
                        sendData(request);
                    } else {
                        // todo 通知关闭打印保护失败
                        throw new MeterException(MeterStateEnum.打印升级保护失败);
                    }
                } catch (MeterException e) {
                    e.printStackTrace();
                    if (Objects.nonNull(request.getPrintBack()))
                        request.getPrintBack().responseBack(PrintBackResponse.fail(getSerial(), e.getMeterState()));
                }
            }
        });
        this.sendData(print0200Request);
    }

    /**
     * @description: 文件升级 (只提供字库升级使用)
     **/
    default void upgrade_all(PrintFileRequest request) throws MeterException {
        SystemtUtils.sleep(2);
        // 关机保护
        Print0200Request print0200Request = new Print0200Request();
        print0200Request.setSdk(true);
        print0200Request.setUpgrade(true);
        print0200Request.setPrintBack(new IPrintBack() {
            /**
             * @param response
             * @description: 操作结果响应信息
             */
            @Override
            public void responseBack(PrintResponse response) {
                try {
                    // 关闭打印保护结果
                    if (((Print0200Response) response).isResult()) {
                        PrintFileRequest printFileRequest = null;
                        if (request instanceof PrintFontRequest) {
                            printFileRequest = new PrintFontRequest();
                        } else if (request instanceof PrintBitMapRequest) {
                            printFileRequest = new PrintBitMapRequest();
                        } else {
                            printFileRequest = new PrintUpgradeRequest();
                        }

                        printFileRequest.setFile(request.getFile());
                        printFileRequest.setName(request.getName());
                        printFileRequest.setInputStream(request.getInputStream());
                        printFileRequest.setNameParams(request.getNameParams());
                        printFileRequest.setHexData(request.getHexData());
                        printFileRequest.setCommunication(request.getCommunication());
                        printFileRequest.setName(request.getName());
                        printFileRequest.setUpgrade(true);
                        PrintUpgradeThread printUpgradeThread = new PrintUpgradeThread("upgrade_all after", communication(), request.getPrintBack());
                        printUpgradeThread.start();

                        printFileRequest.setPrintBack(new IPrintBack() {

                            /**
                             * @param scheduleState
                             * @description: 过程进度
                             */
                            @Override
                            public void schedule(ScheduleState scheduleState) {
                                if (request.getPrintBack() != null) request.getPrintBack().schedule(scheduleState);
                            }

                            /**
                             * @param response
                             * @description: 操作结果响应业务信息
                             */
                            @Override
                            public void responseBack(PrintBackResponse response) {
                                System.out.println(String.format("升级响应：【%s】", response.toString()));
                                printUpgradeThread.response(response);
                            }
                        });
                        sendDataWithAllPackage(printFileRequest);
                    } else {
                        // todo 通知关闭打印保护失败
                        throw new MeterException(MeterStateEnum.打印升级保护失败);
                    }
                } catch (MeterException e) {
                    e.printStackTrace();
                    if (Objects.nonNull(request.getPrintBack()))
                        request.getPrintBack().responseBack(PrintBackResponse.fail(getSerial(), e.getMeterState()));
                }
            }
        });
        this.sendData(print0200Request);
    }

    /**
     * @description: 根据文件名 + 命令 检查升级结果
     **/
    default void checkUpgrade(List<Integer> nameParas, IPrintBack printBack) throws MeterException {
        if (nameParas == null || nameParas.size() > 2)
            throw new MeterException(MeterStateEnum.无效升级文件检查);
        Print0000Request request = new Print0000Request();
        request.setSdk(true);
        request.setPrintBack(new IPrintBack() {
            /**
             * @param printResponse
             * @description: 操作结果响应信息
             */
            @Override
            public void responseBack(PrintResponse printResponse) {
                if (Objects.nonNull(printBack)) {
                    Print0000Response response = (Print0000Response) printResponse;
                    try {
                        if (nameParas.size() == 1) {
                            // 字库检查
                            if (response.getFontVersion().equals(nameParas.get(0).toString())) {
                                printBack.responseBack(PrintBackResponse.success(getSerial()));
                            } else {
                                throw new MeterException(MeterStateEnum.字库未升级成功);
                            }
                        } else if (nameParas.size() == 2) {
                            // 固件检查
                            if (response.getFirmwareVersion().equals(nameParas.get(0).toString()) && response.getUpgradeVersion().equals(nameParas.get(1).toString())) {
                                printBack.responseBack(PrintBackResponse.success(getSerial()));
                            } else {
                                throw new MeterException(MeterStateEnum.版本未升级成功);
                            }
                        } else {
                            printBack.responseBack(PrintBackResponse.success(getSerial()));
                        }

                    } catch (MeterException e) {
                        e.printStackTrace();
                        printBack.responseBack(PrintBackResponse.fail(getSerial(), e.getMeterState()));
                    }
                }
            }
        });
        this.sendData(request);
    }
}
