package com.jhscale.meter.protocol.print;

import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.exp.MeterStateEnum;
import com.jhscale.meter.protocol.print.entity.PrintCmdInfo;
import com.jhscale.meter.protocol.print.entity.PrintLOGParam;
import com.jhscale.meter.protocol.print.entity.PrintRequest;
import com.jhscale.meter.protocol.print.entity.PrintResponse;
import com.jhscale.meter.protocol.print.entity.para.Para;
import com.jhscale.meter.protocol.print.entity.para.RealBitmapPara;
import com.jhscale.meter.protocol.print.link.Messenger;
import com.jhscale.meter.utils.PrintUtils;

import java.lang.reflect.ParameterizedType;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * @author lie_w
 * @title: PrintFactory
 * @projectName jhscale-agreement
 * @description: 打印工厂节点
 * @date 2020-12-2210:33
 */
public class PrintNode implements PrintCommunication {

    // 名称
    private Integer serial;

    // 通讯器
    private Messenger messenger;
    // 异步发送线程
    private PrintThread thread;

    // 单次通讯最大超时时间 默认5S
    private Long communicationMaxTime = PrintConstant.SINGLE_COMMUNICATION_MAX_TIME;

    // 打印缓冲区
    private List<Para> paraCache;

    // 响应原始解析数据
    private boolean originalResponse = false;
    // 全日志
    private PrintLOGParam param;

    /*********************************打印工厂构建 Start******************************************/

    public PrintNode() {
    }

    public PrintNode(Integer serial, Messenger messenger) {
        this.setMessenger(serial, messenger);
    }

    public PrintNode(Integer serial, Messenger messenger, Long communicationMaxTime) {
        this(serial, messenger);
        this.communicationMaxTime = communicationMaxTime;
    }

    /*********************************打印工厂构建 End******************************************/

    /*********************************打印数据字符集 Start******************************************/

    /**
     * @description: 设置原始数据响应 默认不响应false
     **/
    public PrintNode originalResponse(boolean originalResponse) {
        this.originalResponse = originalResponse;
        return this;
    }

    /**
     * @description: 原始数据响应回调状态
     **/
    @Override
    public boolean originalResponse() {
        return this.originalResponse;
    }

    /**
     * @description: 打印日志配置参数
     **/
    @Override
    public PrintLOGParam param() {
        if (this.param == null) this.param = new PrintLOGParam();
        return this.param;
    }

    /**
     * @description: 设置原始数据响应 默认不响应false
     **/
    public void logSwitch(boolean logSwitch, Integer logLength) {
        if (this.param == null) this.param = new PrintLOGParam();
        this.param.set_switch(logSwitch);
        this.param.set_length(Objects.isNull(logLength) || logLength == 0 ? 2048 : logLength);
    }

    /*********************************发送打印命令通讯 Start******************************************/

    /**
     * @description: 设置单次通讯最大超时时间
     **/
    public PrintNode setCommunicationMaxTime(Long communicationMaxTime) {
        this.communicationMaxTime = communicationMaxTime;
        return this;
    }

    /**
     * @description: 设置通讯器
     **/
    public PrintNode setMessenger(Integer serial, Messenger messenger) {
        this.serial = serial;
        this.messenger = messenger;

        if (this.thread == null) {
            this.thread = new PrintThread().reset();
            this.thread.start();
        } else {
            this.thread.reset();
        }
        return this;
    }

    /**
     * @description: 当前发送状态
     **/
    @Override
    public boolean isSendStatus() {
        return this.thread == null ? false : this.thread.isSend();
    }

    /**
     * @description: 发送完成
     **/
    @Override
    public void sendSuccess() {
        if (this.thread != null) {
            this.thread.once();
        }
    }

    /**
     * @description: 关闭通讯器
     **/
    @Override
    public void closeMessenger() {
        if (this.messenger != null) {
            this.messenger.close();
            this.messenger = null;
        }

        if (this.thread != null) {
            this.thread.cease();
            this.thread = null;
        }
    }

    /**
     * @param serial
     * @description: 通讯器名称
     */
    @Override
    public void setSerial(Integer serial) {
        this.serial = serial;
    }

    @Override
    public Integer getSerial() {
        return this.serial;
    }

    /**
     * @description: 检查通讯器正常状态
     **/
    @Override
    public boolean checkCommunication() throws MeterException {
        if (messenger == null || this.thread == null)
            throw new MeterException(MeterStateEnum.通讯组件未初始化);
        return true;
    }

    /**
     * @description: 通讯器
     **/
    @Override
    public Messenger messenger() {
        return this.messenger;
    }


    /**
     * @description: 添加CMD 暂存异步发送
     **/
    public void addSendCmd(String cmd, PrintRequest request) throws MeterException {
        this.checkCommunication();
        this.thread.add(new PrintCmdInfo(cmd, request));
    }

    /**
     * @description: 命令 发送数据
     **/
    public void sendCmd(String cmd, PrintRequest request) throws MeterException {
        try {
            request.setCommunication(this);
            Class clazz = (Class) ((ParameterizedType) request.getClass().getGenericSuperclass()).getActualTypeArguments()[0];
            this.messenger.sendCmd(cmd, request, (PrintResponse) clazz.newInstance());
        } catch (Exception e) {
            e.printStackTrace();
            throw new MeterException(MeterStateEnum.系统异常_V2);
        }
    }

    /**
     * @param paras
     * @description: 添加打印部分内容
     */
    @Override
    public void addParas(List<Para> paras) throws MeterException {
        if (this.paraCache == null) this.paraCache = new ArrayList<>();
        PrintUtils.debug(this.param(), paras);
        if (paras != null && !paras.isEmpty()) {
            for (Para para : paras) {
                if (para instanceof RealBitmapPara)
                    throw new MeterException(MeterStateEnum.实时位图使用专用接口);
            }
            this.paraCache.addAll(paras);
        }
        PrintUtils.debug(this.param(), this.paraCache);
    }

    /**
     * @description: 清空打印内容
     **/
    @Override
    public void clearPara() {
        if (this.paraCache != null) {
            this.paraCache.clear();
            this.paraCache = null;
        }
    }

    /**
     * @description: 全部打印内容
     **/
    @Override
    public List<Para> paras() {
        return this.paraCache;
    }

    /*********************************发送打印命令通讯 End******************************************/
}
