package com.jhscale.meter.protocol.print;

import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.protocol.print.entity.PrintCmdInfo;
import com.jhscale.meter.protocol.print.produce.entity.PrintBackResponse;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * @author lie_w
 * @title: PrintSendThread
 * @projectName meter-jar
 * @description: 打印异步发送线程
 * @date 2022/10/2522:34
 */
public class PrintThread extends Thread {

    // 指令缓冲区
    private List<PrintCmdInfo> printCmdInfo;
    private PrintCmdInfo upgradeInfo;
    // 当前发送指令
    private PrintCmdInfo now;

    // 线程执行状态 false-停止 true-启动
    private boolean execute = false;
    // 线程发送状态 false-未发送 true-启动中
    private boolean send = false;

    private int count = 0; // 循环计数器 记录超时时间
    private int sleep = 10; // 发送休眠时间
    private int max = 50; // 循环计数器 超时时间
    private int max_length = 510;// 发送最大长度

    /**
     * Allocates a new {@code Thread} object. This constructor has the same
     * effect as {@linkplain #(ThreadGroup, Runnable, String) Thread}
     * {@code (null, null, gname)}, where {@code gname} is a newly generated
     * name. Automatically generated names are of the form
     * {@code "Thread-"+}<i>n</i>, where <i>n</i> is an integer.
     */
    public PrintThread() {
        super("PrintThread");
        this.printCmdInfo = Collections.synchronizedList(new ArrayList<>());
    }

    /**
     * @description: 重置线程参数
     **/
    public synchronized PrintThread reset() {
        this.printCmdInfo.clear();
        this.now = null;
        this.upgradeInfo = null;

        this.execute = true;
        this.send = false;

        this.count = 0;

        return this;
    }

    /**
     * @description: 停止线程
     **/
    public synchronized PrintThread cease() {
        this.execute = false;
        this.send = false;

        this.now = null;
        this.upgradeInfo = null;
        if (this.printCmdInfo != null) {
            this.printCmdInfo.clear();
            this.printCmdInfo = null;
        }
        return this;
    }

    /**
     * @description: 单次发送成功
     **/
    public synchronized PrintThread once() {
        this.send = false;
        this.count = 0;
        this.now = null;
        this.upgradeInfo = null;
        return this;
    }

    /**
     * @description: 设置循环间隔
     **/
    public synchronized PrintThread cycle(int sleep, int max) {
        this.sleep = sleep;
        this.max = max;
        return this;
    }

    /**
     * @description: 发送状态
     **/
    public boolean isSend() {
        return this.send;
    }

    /**
     * @description: 添加发送内容
     **/
    public synchronized void add(PrintCmdInfo printCmdInfo) {
        // this.once();
        if (printCmdInfo.getRequest().isUpgrade()) {
            this.upgradeInfo = printCmdInfo;
        } else {
            this.printCmdInfo.add(printCmdInfo);
        }
    }

    /**
     * If this thread was constructed using a separate
     * <code>Runnable</code> run object, then that
     * <code>Runnable</code> object's <code>run</code> method is called;
     * otherwise, this method does nothing and returns.
     * <p>
     * Subclasses of <code>Thread</code> should override this method.
     *
     * @see #start()
     * @see #stop()
     * @see #(ThreadGroup, Runnable, String)
     */
    @Override
    public void run() {
        while (this.execute && !isInterrupted()) {
            if (!this.send) {

                this.printExecute();

            } else {
                try {
                    Thread.sleep(this.sleep);
                } catch (InterruptedException e) {
                }

                this.count++;

                // 超过300ms 当超时处理，重新发送一次读取命令
                if (this.count > this.max && this.now != null) {
                    this.now.getRequest().timeOut();
                }
            }
        }
    }

    /**
     * @description: 同步代码执行
     **/
    private synchronized void printExecute() {
        if (this.upgradeInfo != null) {
            this.now = this.upgradeInfo;
        } else if (this.printCmdInfo != null && !this.printCmdInfo.isEmpty()) {
            try {
                this.now = this.printCmdInfo.remove(0);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        try {
            if (this.now != null && this.now.getRequest() != null) {
                this.send = true;
                if (this.now.getRequest().isAll()) {
                    while (!this.now.getRequest().isEnd()) {
                        this.now.getRequest().setSubContentLength(this.max_length);
                        this.now.getRequest().getCommunication().sendCmd(
                                this.now.getRequest().cmd(),
                                this.now.getRequest()
                        );
                    }
                } else {
                    this.now.getRequest().getCommunication().sendCmd(this.now.getCmd(), this.now.getRequest());
                }
            }
        } catch (MeterException e) {
            e.printStackTrace();

            if (this.now != null && this.now.getRequest() != null && this.now.getRequest().getPrintBack() != null) {
                this.now.getRequest()
                        .getPrintBack()
                        .responseBack(PrintBackResponse.fail(this.now.getRequest().getCommunication().getSerial(),
                                PrintStateCode.COMMUNICATION_TIME_OUT, "系统异常:" + e.getMessage()));
            }

            // 异常直接响应成功
            this.once();
        }

        try {
            Thread.sleep(3);
        } catch (InterruptedException e1) {
        }
    }
}
