package com.jhscale.meter.protocol.print.data;

import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * @author lie_w
 * @title: PrintDataParse
 * @projectName jhscale-agreement
 * @description: 打印模板解析
 * @date 2020-12-2212:43
 */
public class PrintDataParse {

    private static final List<String> SPECIAL_CHAR = Arrays.asList(new String[]{"z", "Z", "T", "R", "I", "i", "S", "s", "#"});

    // 原始模板
    private String data;

    // 模板片段
    private List<PrintPart> printParts;

    public PrintDataParse() {
    }

    public PrintDataParse(String data) {
        this.data = data;
        this.printParts = new ArrayList<>();

        int count = 0;
        while (StringUtils.isNotBlank(data)) {
            int zindex = data.indexOf("z");
            if (zindex == -1) {
                // 没有循环段落
                this.printParts.add(PrintPart.simple(count++, data));
                data = "";
            } else if (zindex == 0) {
                int Zindex = data.indexOf("Z");
                this.printParts.add(PrintPart.cycle(count++, data.substring(2, Zindex)));
                data = data.substring(Zindex + 2);
            } else {
                this.printParts.add(PrintPart.simple(count++, data.substring(0, zindex)));
                data = data.substring(zindex);
            }
        }

        if (!this.printParts.isEmpty())
            this.printParts.forEach(part -> {
                part.resolveField();
            });
    }

    /**
     * @description: 打印模板片段信息
     **/
    public static class PrintPart implements Comparable {

        // 顺序
        private Integer order;

        // 原始内容
        private String part;

        // 带填充内容
        private String filledPart;

        // 循环模块
        private boolean cycle;

        // 替换字段
        private List<String> fields;

        // 隐射关系  Key->field Val-> 对象属性
        private Map<String, String> mappings;

        public PrintPart() {
        }

        public PrintPart(Integer order, String part, boolean cycle) {
            this.order = order;
            this.part = part;
            this.cycle = cycle;
            fields = new ArrayList<>();
        }

        /**
         * @description: 普通段落
         **/
        public static PrintPart simple(Integer order, String part) {
            return new PrintPart(order, part, false);
        }

        /**
         * @description: 循环段落
         **/
        public static PrintPart cycle(Integer order, String part) {
            return new PrintPart(order, part, true);
        }

        /**
         * @description: 解析字段
         **/
        public void resolveField() {
            if (StringUtils.isNotBlank(part)) {
                String[] partArr = this.part.split("\n");
                if (partArr != null && partArr.length > 0) {
                    StringBuffer sb = new StringBuffer();
                    for (String partStr : partArr) {
                        if (SPECIAL_CHAR.contains(String.valueOf(partStr.charAt(0)))) {
                            fields.add(partStr);
                            sb.append("%s");
                        } else {
                            sb.append(partStr);
                        }
                    }
                    this.filledPart = sb.toString();
                }
            }
        }

        /**
         * @description: 获取映射关系
         **/
        public String mapping(String field) {
            if (StringUtils.isBlank(field)) return "";
            return (mappings != null) ? mappings.get(field) : "";
        }

        /**
         * @description: 添加映射关系
         **/
        public PrintPart putMapping(String field, String mapping) {
            mappings.put(field, mapping);
            return this;
        }

        @Override
        public int compareTo(Object o) {
            return this.order.compareTo(((PrintPart) o).getOrder());
        }

        public Integer getOrder() {
            return order;
        }

        public void setOrder(Integer order) {
            this.order = order;
        }

        public String getPart() {
            return part;
        }

        public void setPart(String part) {
            this.part = part;
        }

        public String getFilledPart() {
            return filledPart;
        }

        public void setFilledPart(String filledPart) {
            this.filledPart = filledPart;
        }

        public boolean isCycle() {
            return cycle;
        }

        public void setCycle(boolean cycle) {
            this.cycle = cycle;
        }

        public List<String> getFields() {
            return fields;
        }

        public void setFields(List<String> fields) {
            this.fields = fields;
        }

        public Map<String, String> getMappings() {
            return mappings;
        }

        public void setMappings(Map<String, String> mappings) {
            this.mappings = mappings;
        }
    }

    public String getData() {
        return data;
    }

    public void setData(String data) {
        this.data = data;
    }

    public List<PrintPart> getPrintParts() {
        return printParts;
    }

    public void setPrintParts(List<PrintPart> printParts) {
        this.printParts = printParts;
    }
}
