package com.jhscale.meter.protocol.print.entity;

import com.jhscale.common.utils.SystemtUtils;
import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.exp.MeterPrintThermalException;
import com.jhscale.meter.exp.MeterStateEnum;
import com.jhscale.meter.exp.MeterWarningException;
import com.jhscale.meter.protocol.print.PrintConstant;
import com.jhscale.meter.protocol.print.PrintFactory;
import com.jhscale.meter.protocol.print.PrintStateCode;
import com.jhscale.meter.protocol.print.entity._interface.IPrintResponse;
import com.jhscale.meter.protocol.print.produce.entity.PrintBackResponse;
import com.jhscale.meter.utils.ByteUtils;
import com.jhscale.meter.utils.PrintUtils;
import org.apache.commons.lang3.StringUtils;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.math.BigInteger;
import java.util.Objects;

/**
 * @author lie_w
 * @title: PrintResponse
 * @projectName jhscale-agreement
 * @description: TODO
 * @date 2020-12-2210:27
 */
public class PrintResponse<T extends PrintBackResponse, U extends PrintRequest> implements IPrintResponse<T, U> {

    // 序号
    private Integer serial;

    // 打印Mark 信息
    private PrintMark mark;

    // 内容长度
    private int contentLength;

    // 内容
    private String content;

    // 准备剪切内容
    private String subContent;

    // 校验CRC
    private String crc;

    // 状态码 00 - 数据保存完整 01 - 超出缓冲极限 02 - CRC 失败 04 - 不争取的起始结束标志
    private String state;

    // 缓冲区长度
    private Integer bufferLength;

    // 收到数据长度
    private Integer receiveLength;

    // 暂时收到的数据
    private String cache = "";

    public Integer getSerial() {
        return serial;
    }

    public void setSerial(Integer serial) {
        this.serial = serial;
    }

    public PrintMark getMark() {
        return mark;
    }

    public void setMark(PrintMark mark) {
        this.mark = mark;
    }

    public int getContentLength() {
        return contentLength;
    }

    public void setContentLength(int contentLength) {
        this.contentLength = contentLength;
    }

    public String getContent() {
        return content;
    }

    public void setContent(String content) {
        this.content = content;
    }

    public String getCrc() {
        return crc;
    }

    public void setCrc(String crc) {
        this.crc = crc;
    }

    public String getState() {
        return state;
    }

    public void setState(String state) {
        this.state = state;
    }

    public Integer getBufferLength() {
        return bufferLength;
    }

    public Integer bufferLength() {
        return bufferLength / 2;
    }

    public void setBufferLength(Integer bufferLength) {
        this.bufferLength = bufferLength;
    }

    public Integer getReceiveLength() {
        return receiveLength;
    }

    public void setReceiveLength(Integer receiveLength) {
        this.receiveLength = receiveLength;
    }

    @Override
    public String toString() {
        return "PrintResponse{" +
                "serial=" + serial + '\'' +
                ", mark=" + mark.toString() +
                ", contentLength=" + contentLength +
                ", content='" + content + '\'' +
                ", subContent='" + subContent + '\'' +
                ", crc='" + crc + '\'' +
                ", state='" + state + '\'' +
                ", bufferLength=" + bufferLength +
                ", receiveLength=" + receiveLength +
                ", cache='" + cache + '\'' +
                '}';
    }

    /**
     * @param responseStr
     * @param request
     * @description: 回调处理
     */
    @Override
    public void callBack(String responseStr, U request) {
        try {
            this.serial = request.getCommunication().getSerial();
            // 检查响应体
            this.checkResponse(responseStr);
            responseStr = responseStr.replace(" ", "");
            // 数据解析
            responseStr = this.parse(responseStr);
            if (StringUtils.isBlank(responseStr)) return;
            // 检查通讯数据信息
            this.parseCheck(request);
            if (StringUtils.isNotBlank(this.content)) {
                try {
                    // 具体内容解析
                    this.analysis(request);
                    // 具体内容有效性检查
                    this.analysisCheck(request);
                } catch (MeterPrintThermalException e) {
                    // 打印机过热休眠 重置打印缓冲区内容
                    this.bufferLength = 0;
                    SystemtUtils.sleep(2);
                    PrintUtils.debug(request.getCommunication().param(), e.getMessage());
                    request.getCommunication().sendSuccess();
                    if (Objects.nonNull(request.getPrintBack()))
                        request.getPrintBack()
                                .responseBack(PrintBackResponse.warning(request.getCommunication().getSerial(),
                                        e.getMeterState().getCode(), e.getMeterState().getMsg()));
                }
            }

            if (this.bufferLength == 0) {
                // 缓冲区能接受的空间为0
                try {
                    Thread.sleep(100);
                } catch (InterruptedException e) {
                }
            }

            if (Objects.nonNull(request.getPrintBack())) {
                try {
                    request.getPrintBack().schedule(new ScheduleState(request.getCommunication().getSerial(), request.getEnd(), request.getContent().length()));
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }

            // 检查是否还有数据需要发送
            if (!request.isAll() || request.isEnd()) {
                if (request.isFirst() || !request.sendEnd()) {
                    request.setFirst(false);
                    request.setSubContentLength(this.getBufferLength());
                    request.getCommunication().sendCmd(request.cmd(), request);
                } else {
                    request.getCommunication().sendSuccess();
                    if (Objects.nonNull(request.getPrintBack())) {
                        if (request.isSdk() || request.getCommunication().originalResponse())
                            request.getPrintBack().responseBack(this);

                        T response = this.reflexResponse(request);
                        request.getPrintBack().responseBack(response);
                    }
                }
            }
        } catch (MeterWarningException e) {
            PrintUtils.debug(request.getCommunication().param(), e.getMessage());
            request.getCommunication().sendSuccess();
            if (Objects.nonNull(request.getPrintBack()))
                request.getPrintBack()
                        .responseBack(PrintBackResponse.warning(request.getCommunication().getSerial(),
                                e.getMeterState()));
        } catch (MeterException e) {
            PrintUtils.debug(request.getCommunication().param(), e.getMessage());
            request.getCommunication().sendSuccess();
            if (Objects.nonNull(request.getPrintBack()))
                request.getPrintBack()
                        .responseBack(PrintBackResponse.fail(request.getCommunication().getSerial(),
                                e.getMeterState().getCode(), e.getMeterState().getMsg()));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * @description:反射结果信息
     **/
    public T reflexResponse(U request) throws MeterException {
        T response = null;
        try {
            Type genericSuperclass = this.getClass().getGenericSuperclass();
            Type[] actualTypeArguments = ((ParameterizedType) genericSuperclass).getActualTypeArguments();
            Class<T> clazz = (Class<T>) ((ParameterizedType) this.getClass().getGenericSuperclass()).getActualTypeArguments()[0];
            response = clazz.newInstance();
        } catch (Exception e) {
            e.printStackTrace();
            throw new MeterException(MeterStateEnum.响应对象反射异常);
        }
        if (response == null) throw new MeterException(MeterStateEnum.响应对象反射异常);
        response.setSerial(request.getCommunication().getSerial());
        response.setCode(PrintStateCode.SUCCESS);
        response.setLinkResult(PrintConstant.SUCCESS.equals(this.state));
        if (response.isLinkResult())
            this.responseBody(response, request);
        return response;
    }

    /**
     * @param responseStr
     * @description: 检查响应字符串
     */
    @Override
    public boolean checkResponse(String responseStr) throws MeterException {
        System.out.println("check_response_string --- >:" + responseStr);
        if (StringUtils.isBlank(responseStr))
            throw new MeterException(MeterStateEnum.响应体不存在);
        responseStr = responseStr.replace(" ", "");

        if (responseStr.length() % 2 != 0)
            throw new MeterException(MeterStateEnum.响应体无效);

        return true;
    }

    /**
     * @param responseStr
     * @description: 解析响应体
     */
    @Override
    public String parse(String responseStr) throws MeterException {
        if (!responseStr.startsWith(PrintConstant.PRINT_START_TOP))
            responseStr = cache + responseStr;

        int index = ByteUtils.indexOf(responseStr, PrintConstant.PRINT_START_TOP);
        if (index == -1) {
            System.err.printf("--- 无效数据包：%s%n", responseStr);
            this.cache = "";
            return "";
        } else {
            responseStr = responseStr.substring(index);
        }

        if (!responseStr.startsWith(PrintConstant.PRINT_START_TOP)) {
            System.err.printf("--- 无效数据包：%s%n", responseStr);
            return "";
            // throw new MeterException(MeterStateEnum.起始结束标志无效);
        }

        try {
            this.mark = new PrintMark(responseStr.substring(2, 4));
            this.contentLength = Integer.parseInt(responseStr.substring(4, 6), 16) * 2;
            this.crc = responseStr.substring(6 + this.contentLength);

            String content = responseStr.substring(6, 6 + this.contentLength);
            this.state = content.substring(0, 2);
            this.bufferLength = Integer.parseInt(content.substring(2, 4), 16) * 2;
            this.receiveLength = Integer.parseInt(content.substring(4, 6), 16) * 2;
            this.content = content.substring(6);
            this.subContent = content.substring(6);
            // this.cache = responseStr.substring(6 + this.contentLength);
            this.cache = "";
            return responseStr;
        } catch (Exception e) {
            System.err.printf("数据包[%s]不完整缓存等待下次%n", responseStr);
            this.cache = responseStr;
            return null;
        }
    }

    /**
     * @param request
     * @description: 响应体校验
     */
    @Override
    public boolean parseCheck(U request) throws MeterException {
        // 检查状态码
        switch (this.getState()) {
            case PrintConstant.SUCCESS:
                break;
            case PrintConstant.BUFFER_LIMIT_EXCEEDED:
                throw new MeterException(MeterStateEnum.超出缓冲极限);
            case PrintConstant.CRC_ERROR:
                throw new MeterException(MeterStateEnum.CRC校验失败);
            case PrintConstant.BUFFER_LIMIT_EXCEEDED_AND_CRC_ERROR:
                throw new MeterException(MeterStateEnum.超出缓冲极限同时CRC校验失败);
            case PrintConstant.START_ERROR:
                throw new MeterException(MeterStateEnum.起始结束标志无效);
            default:
                System.err.printf("未识别的状态码:[%s]%n", this.getState());
                throw new MeterException(MeterStateEnum.未识别的状态码);
        }

//        if (PrintFactory.getInstance().isSendStatus() && !request.subContentLength().equals(this.getReceiveLength()))
//            throw new MeterException(MeterStateEnum.收发长度不相同);
        return true;
    }

    /**
     * @description: 剪切指定长度
     **/
    public String parseBytes(Integer length) {
        String part = this.parseLength(length);
        if (StringUtils.isNotBlank(part)) {
            try {
                return String.valueOf(Integer.parseInt(ByteUtils.convert(part), 16));
            } catch (Exception e) {
                return part;
            }
        } else {
            return "";
        }
    }

    public String parseBytesInt(Integer length) {
        String part = this.parseLength(length);
        if (StringUtils.isNotBlank(part)) {
            BigInteger bigInteger = new BigInteger(ByteUtils.convert(part), 16);
            if (bigInteger.bitLength() > 31) {
                bigInteger = bigInteger.subtract(BigInteger.ONE.shiftLeft(32));
            }
            return bigInteger.toString();
        } else {
            return "";
        }
    }

    /**
     * @description: 剪切二进制值
     **/
    public String toBitval() {
        return ByteUtils.formatMark(parseLength(1));
    }

    /**
     * @description: 剪切指定字节长度
     **/
    public String parseLength(Integer length) {
        if (StringUtils.isBlank(this.subContent)) return "";
        String res = this.subContent.substring(0, 2 * length);
        this.subContent = this.subContent.substring(length * 2);
        return res;
    }

    /**
     * @description: 文本解析
     **/
    public String parseText() {
        if (StringUtils.isBlank(this.subContent)) return "";
        int index = index(this.subContent, "00");
        String res = this.subContent.substring(0, index);
        this.subContent = this.subContent.substring(index + 2);
        return new String(ByteUtils.fromHexString(res), PrintFactory.getInstance().charset());
    }

    /**
     * @description: 文本解析没有字符集
     **/
    public String parseTextWithOutCharset() {
        if (StringUtils.isBlank(this.subContent)) return "";
        int index = index(this.subContent, "00");
        String res = this.subContent.substring(0, index);
        this.subContent = this.subContent.substring(index + 2);
        return new String(ByteUtils.fromHexString(res));
    }

    /**
     * @description: 获取字节数组中第一个指定字节的位置
     **/
    private int index(String content, String str) {
        int length = str.length();
        for (int i = 0; i < content.length() / length; i++) {
            String substring = content.substring(i * 2, (i + 1) * 2);
            if (substring.equals(str)) return i * 2;
        }
        return -1;
    }
}
