package com.jhscale.meter.protocol.print.entity.bitmap;

import com.jhscale.common.model.inter.JSONModel;
import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.protocol.print.PrintFactory;
import com.jhscale.meter.protocol.print.dither.DitherEnum;
import com.jhscale.meter.protocol.print.dither.DitherVal;
import com.jhscale.meter.protocol.print.dither.ImageBuffer;
import com.jhscale.meter.protocol.print.em.Compress;
import com.jhscale.meter.protocol.print.em.Dir;
import com.jhscale.meter.protocol.print.em.DotImage;
import com.jhscale.meter.protocol.print.image.*;
import com.jhscale.meter.utils.ByteUtils;

import java.io.File;
import java.util.Arrays;
import java.util.Objects;

/**
 * @author lie_w
 * @title: BitMap
 * @projectName jhscale-agreement
 * @description: TODO
 * @date 2021-01-2717:19
 */
public abstract class BitMap<T extends BitMap> implements JSONModel {

    // 源文件
    private File source;

    // 最终需要资源
    private DitherVal ditherVal;

    /**
     * @description: 图片旋转
     **/
    // 旋转
    private Dir dir = Dir.ZERO;

    // 保存旋转文件
    private boolean _rotate;

    // 旋转文件
    private File rotate;

    /**
     * @description: 图片压缩
     **/
    // 压缩类型
    private Compress compre_type;

    // 宽
    private int width;

    // 高
    private int height;

    // 缩放比例
    private double rate = 1.0;

    // 保存压缩文件
    private boolean _compress = false;

    // 压缩文件
    private File compress;

    /**
     * @description: 点图处理
     **/
    // 点图处理方式
    private DotImage dotImage;

    // 百分比处理
    private Double colorRatio;

    // dither 处理方式
    private DitherEnum ditherType;
    // dither 临界值
    private Integer ditherThreshold;

    // 保存抖动文件
    private boolean _dither = false;

    // 抖动文件
    private File dither;

    // 单行字节长度
    private int lineLength;

    /**
     * @description: 构建文件内容
     **/
    public abstract T bulid();

    /**
     * @description: 添加源文件
     **/
    public T addSource(File file) {
        this.source = file;
        return (T) this;
    }

    /**
     * @description: 添加解析源文件 (旧版 作废)
     **/
    @Deprecated
    public T addSource(DitherVal ditherVal) {
        this.ditherVal = ditherVal;
        return (T) this;
    }

    /**
     * @description: 抖动处理(旧版 作废)
     **/
    @Deprecated
    public T dither(File dither) {
        this.dither = dither;
        this._dither = true;
        return (T) this;
    }

    /**
     * @description: 旋转文件(不保存)
     **/
    protected T rotate(Dir dir) {
        this.dir = dir;
        return (T) this;
    }

    /**
     * @description: 旋转文件
     **/
    protected T rotate(Dir dir, File rotate) {
        this.dir = dir;
        this._rotate = true;
        this.rotate = rotate;
        return (T) this;
    }

    /**
     * @description: 文件比例压缩(保存压缩文件)
     **/
    public T compress(double rate, File compress) {
        return this.compress(Compress.Proportion, rate, null, null, true, compress);
    }

    /**
     * @description: 文件比例压缩(不保存压缩文件)
     **/
    public T compress(double rate) {
        return this.compress(Compress.Proportion, rate, null, null, false, null);
    }

    /**
     * @description: 文件指定宽高压缩(保存压缩文件)
     **/
    public T compress(int width, int height, File compress) {
        return this.compress(Compress.Width_Height, null, width, height, true, compress);
    }

    /**
     * @description: 文件指定宽高压缩(不保存压缩文件)
     **/
    public T compress(int width, int height) {
        return this.compress(Compress.Width_Height, null, width, height, false, null);
    }

    /**
     * @description: 文件压缩
     **/
    private T compress(Compress compre_type, Double rate, Integer width, Integer height, boolean _compress, File compress) {
        this.compre_type = compre_type;
        if (rate != null)
            this.rate = rate;
        if (width != null)
            this.width = width;
        if (height != null)
            this.height = height;
        this._compress = _compress;
        this.compress = compress;
        return (T) this;
    }

    /**
     * @description: 点图颜色百分比处理
     **/
    public T dither(Double colorRatio, File dither) {
        return this.dither(DotImage.Color, colorRatio, null, null, true, dither);
    }

    public T dither(Double colorRatio) {
        return this.dither(DotImage.Color, colorRatio, null, null, false, null);
    }

    /**
     * @description: 点图dither处理
     **/
    public T dither(DitherEnum ditherType, Integer ditherThreshold, File dither) {
        return this.dither(DotImage.Dither, null, ditherType, ditherThreshold, true, dither);
    }

    public T dither(DitherEnum ditherType, Integer ditherThreshold) {
        return this.dither(DotImage.Dither, null, ditherType, ditherThreshold, false, null);
    }

    /**
     * @description: 点图处理
     **/
    private T dither(DotImage dotImage, Double colorRatio, DitherEnum ditherType, Integer ditherThreshold, boolean _dither, File dither) {
        this.dotImage = dotImage;
        if (Objects.nonNull(colorRatio)) this.colorRatio = colorRatio;
        if (Objects.nonNull(ditherType)) this.ditherType = ditherType;
        if (Objects.nonNull(ditherThreshold)) this.ditherThreshold = ditherThreshold;
        this._dither = _dither;
        this.dither = dither;
        return (T) this;
    }

    /**
     * @description: 源文件预处理
     **/
    public T preterate() throws MeterException {
        if (this.source != null && this.source.exists()) {
            if (dir != null && dir != Dir.ZERO && this._rotate && this.rotate == null) {
                File rotate = new File(this.source.getParentFile(), "rotate");
                if (!rotate.exists()) rotate.mkdirs();
                this.rotate = new File(rotate, source.getName());
            }

            if (this.compre_type != null && this._compress && this.compress == null) {
                File compress = new File(this.source.getParentFile(), "compress");
                if (!compress.exists()) compress.mkdirs();
                this.compress = new File(compress, source.getName());
            }

            if (this._dither && this.dither == null) {
                File dither = new File(this.source.getParentFile(), "dither");
                if (!dither.exists()) dither.mkdirs();
                this.dither = new File(dither, source.getName());
            }

            ImageBuffer src;
            if (this.dir != null) {
                ImageBuilder dirImageBuilder = null;
                switch (this.dir) {
                    case ZERO:
                        dirImageBuilder = new ZeroRotateBuilder();
                        break;
                    case Ninety:
                        dirImageBuilder = new NinetyRotateBuilder();
                        break;
                    case One_Hundred_Eighty:
                        dirImageBuilder = new OnehundredandeightyRatateBuilder();
                        break;
                    case TWO_Hundred_Seventy:
                        dirImageBuilder = new TwohundredandseventyRatateBuilder();
                        break;
                    default:
                        dirImageBuilder = new ZeroRotateBuilder();
                        break;
                }

                src = dirImageBuilder
                        .addSource(this.source)
                        .addImageProcess(PrintFactory.getInstance().getImageProcess())
                        .addTarget(this.rotate)
                        .build()
                        .getTargetBuffer();
            } else {
                src = PrintFactory.getInstance().getImageProcess().readImage(this.source);
            }

            if (this.compre_type != null) {
                ImageBuilder compreImageBuilder = null;
                switch (this.compre_type) {
                    case Proportion:
                        compreImageBuilder = new ProportionCompressBuilder()
                                .addRate(this.rate);
                        break;
                    case Width_Height:
                        compreImageBuilder = new FixedCompressBuilder()
                                .addWithdHeight(this.width, this.height);
                        break;
                    default:
                        compreImageBuilder = new ProportionCompressBuilder()
                                .addRate(this.rate);
                        break;
                }
                src = compreImageBuilder
                        .addSource(src)
                        .addImageProcess(PrintFactory.getInstance().getImageProcess())
                        .addTarget(this.compress)
                        .build()
                        .getTargetBuffer();
            }

            DotplotBuilder dotplotBuilder = null;
            if (Objects.nonNull(this.dotImage)) {
                switch (this.dotImage) {
                    case Color:
                        dotplotBuilder = new ColorRatioBuilder()
                                .addRatio(this.colorRatio);

                        new ColorRatioBuilder()
                                .addRatio(this.colorRatio)
                                .addSource(src)
                                .addImageProcess(PrintFactory.getInstance().getImageProcess())
                                .addTarget(this.dither)
                                .build().getDitherVal();
                        break;
                    case Dither:
                        dotplotBuilder = new DitherBuilder()
                                .addDither(this.ditherType)
                                .addThreshold(this.ditherThreshold);
                        break;
                    default:
                        dotplotBuilder = new DitherBuilder();
                        break;
                }
            }
            if (dotplotBuilder == null) dotplotBuilder = new DitherBuilder();

            this.ditherVal = ((DotplotBuilder) (dotplotBuilder
                    .addSource(src)
                    .addImageProcess(PrintFactory.getInstance().getImageProcess())
                    .addTarget(this.dither)
                    .build())).getDitherVal();
        }
        return (T) this;
    }

    /**
     * @description: 每行补充长度信息
     **/
    protected int[] append() {
        int residue = this.ditherVal.getWidth() % 8;
        this.lineLength = residue == 0 ? this.ditherVal.getWidth() / 8 : (((int) (this.ditherVal.getWidth() / 8)) + 1);
        return residue != 0 ? new int[8 - residue] : new int[0];
    }

    /**
     * @description: 拷贝内容信息
     **/
    protected StringBuffer copyOfRange(DitherVal ditherVal, int[] append) {
        StringBuffer content = new StringBuffer();
        for (int i = 0; i < ditherVal.getHeight(); i++) {
            int[] val = ByteUtils.byteMerger(Arrays.copyOfRange(ditherVal.getData(), i * ditherVal.getWidth(), (i + 1) * ditherVal.getWidth()), append);
            for (int j = 0; j < (val.length / 8); j++) {
                int[] ints = Arrays.copyOfRange(val, j * 8, (j + 1) * 8);
                String part = "";
                for (int anInt : ints) {
                    part += anInt;
                }
                content.append(ByteUtils.int2Hex(Integer.parseInt(part, 2)));
            }
        }
        return content;
    }

    public File getSource() {
        return source;
    }

    public File getCompress() {
        return compress;
    }

    public File getDither() {
        return dither;
    }

    public DitherVal getDitherVal() {
        return ditherVal;
    }

    public int getLineLength() {
        return lineLength;
    }
}
