package com.jhscale.meter.protocol.print.image;

import com.jhscale.meter.protocol.print.dither.ImageBuffer;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.Objects;

/**
 * @author Loveven
 * @title: DefaulImageProcess
 * @projectName jhscale-agreement
 * @description: 图片处理器
 * @date 2021-03-2615:05
 */
public class DefaultImageProcess implements ImageProcess {

    /**
     * @description: 读取RGB
     **/
    @Override
    public RGB readRGB(int pixel) {
        Color color = new Color(pixel, true);
        RGB rgb = new RGB();
        rgb.setPixel(pixel);
        rgb.setAlpha(color.getAlpha());
        rgb.setRI(color.getRed());
        rgb.setRH(Integer.toHexString(color.getRed()));
        rgb.setGI(color.getGreen());
        rgb.setGH(Integer.toHexString(color.getGreen()));
        rgb.setBI(color.getBlue());
        rgb.setBH(Integer.toHexString(color.getBlue()));
        return rgb;
    }

    /**
     * @description: 写RGB
     **/
    @Override
    public RGB writeRGB(int red, int green, int blue, int alpha) {
        int pixel = new Color(red, green, blue, alpha).getRGB();
        return this.readRGB(pixel);
    }

    /**
     * @param rgb
     * @description: 写RGB
     */
    @Override
    public int writeRGB(RGB rgb) {
        return new Color(rgb.getRI(), rgb.getGI(), rgb.getBI(), rgb.getAlpha()).getRGB();
    }

    /**
     * @param file
     * @description: 读取图片文件
     */
    @Override
    public ImageBuffer readImage(File file) {
        ImageBuffer imageBuffer = null;
        try {
            BufferedImage bufferedImage = ImageIO.read(file);

            imageBuffer = new ImageBuffer();
            imageBuffer.setMinX(bufferedImage.getMinX());
            imageBuffer.setMinY(bufferedImage.getMinY());
            imageBuffer.setWidth(bufferedImage.getWidth());
            imageBuffer.setHeight(bufferedImage.getHeight());
            imageBuffer.setImageSize(bufferedImage.getData().getDataBuffer().getSize());
            imageBuffer.setAbsolutePath(file.getAbsolutePath());
            String name = file.getName();
            imageBuffer.setName(name.substring(0, name.indexOf(".")));
            imageBuffer.setSuffix(name.substring(name.indexOf(".") + 1));
            imageBuffer.setType(bufferedImage.getType());

            RGB[][] RGBS = new RGB[imageBuffer.getHeight()][imageBuffer.getWidth()];
            for (int j = 0; j < imageBuffer.getHeight(); j++) {
                for (int i = 0; i < imageBuffer.getWidth(); i++) {
                    RGBS[j][i] = this.readRGB(bufferedImage.getRGB(i, j));
                }
            }
            imageBuffer.setRgbs(RGBS);
        } catch (IOException e) {
            e.printStackTrace();
        }
        return imageBuffer;
    }

    /**
     * @param buffer
     * @param target
     * @description: 写文件
     */
    @Override
    public void writeImage(ImageBuffer buffer, File target) {
        BufferedImage bufferedImage = new BufferedImage(buffer.getWidth(), buffer.getHeight(), buffer.getType());
        for (int j = 0; j < buffer.getRgbs().length; j++) {
            for (int i = 0; i < buffer.getRgbs()[j].length; i++) {
                RGB rgb = buffer.getRgbs()[j][i];
                bufferedImage.setRGB(i, j, this.writeRGB(Objects.isNull(rgb) ? new RGB() : rgb));
            }
        }
        try {
            File parentFile = target.getParentFile();
            if (!parentFile.exists()) parentFile.mkdirs();
            if (!target.exists()) target.createNewFile();
            ImageIO.write(bufferedImage, buffer.getSuffix(), target);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * @param buffer
     * @param width
     * @param height
     * @description: 图片压缩
     */
    @Override
    public ImageBuffer compress(ImageBuffer buffer, int width, int height) {
        BufferedImage source = new BufferedImage(buffer.getWidth(), buffer.getHeight(), buffer.getType());
        for (int j = 0; j < buffer.getRgbs().length; j++) {
            for (int i = 0; i < buffer.getRgbs()[j].length; i++) {
                RGB rgb = buffer.getRgbs()[j][i];
                source.setRGB(i, j, this.writeRGB(Objects.isNull(rgb) ? new RGB() : rgb));
            }
        }
        BufferedImage target = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
        //绘制图像  getScaledInstance表示创建此图像的缩放版本，返回一个新的缩放版本Image,按指定的width,height呈现图像
        //Image.SCALE_SMOOTH,选择图像平滑度比缩放速度具有更高优先级的图像缩放算法。
        target.getGraphics().drawImage(source.getScaledInstance(width, height, Image.SCALE_SMOOTH), 0, 0, null);

        ImageBuffer targetBuffer = new ImageBuffer();
        targetBuffer.setMinX(target.getMinX());
        targetBuffer.setMinY(target.getMinY());
        targetBuffer.setWidth(target.getWidth());
        targetBuffer.setHeight(target.getHeight());
        targetBuffer.setImageSize(target.getData().getDataBuffer().getSize());
        targetBuffer.setName(buffer.getName());
        targetBuffer.setSuffix(buffer.getSuffix());
        targetBuffer.setType(buffer.getType());

        RGB[][] RGBS = new RGB[target.getHeight()][target.getWidth()];
        for (int j = 0; j < target.getHeight(); j++) {
            for (int i = 0; i < target.getWidth(); i++) {
                RGBS[j][i] = this.readRGB(target.getRGB(i, j));
            }
        }
        targetBuffer.setRgbs(RGBS);
        return targetBuffer;
    }
}
