package com.jhscale.meter.protocol.print.image;

import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.exp.MeterStateEnum;
import com.jhscale.meter.protocol.print.dither.DitherEnum;
import com.jhscale.meter.protocol.print.dither.DitherVal;
import com.jhscale.meter.protocol.print.dither.IDithering;
import com.jhscale.meter.protocol.print.dither.ImageBuffer;
import com.jhscale.meter.protocol.print.dither.algorithm.*;
import com.jhscale.meter.protocol.print.em.DotImage;

import java.util.Objects;

/**
 * @author lie_w
 * @title: DitherBulider
 * @projectName jhscale-agreement
 * @description: Dither 构建
 * @date 2021/3/2923:22
 */
public class DitherBuilder extends DotplotBuilder<DitherBuilder> {

    // 抖动算法 Dither 存在
    private DitherEnum dither = DitherEnum.burkes;

    // 抖动算法临界值
    private Integer ditherThreshold = ImageConstant.DEFAIL_THRESHOLD;

    // dither 算法
    private IDithering dithering;

    public DitherBuilder() {
        super.setDotImage(DotImage.Dither);
    }

    /**
     * @description: 添加 dither 算法
     **/
    public DitherBuilder addDither(DitherEnum dither) {
        if (Objects.nonNull(dither))
            this.dither = dither;
        return this;
    }

    /**
     * @description: 添加 dither 算法临界值
     **/
    public DitherBuilder addThreshold(Integer threshold) {
        if (Objects.nonNull(threshold))
            this.ditherThreshold = threshold;
        return this;
    }

    @Override
    protected DitherBuilder check() throws MeterException {
        if (Objects.isNull(this.dither))
            throw new MeterException(MeterStateEnum.Dither算法不存在);

        if (Objects.isNull(this.ditherThreshold))
            throw new MeterException(MeterStateEnum.Dither临界值不存在);

        switch (dither) {
            case ordered2By2Bayer:
                dithering = new Ordered2By2Bayer();
                break;
            case ordered3By3Bayer:
                dithering = new Ordered3By3Bayer();
                break;
            case ordered4By4Bayer:
                dithering = new Ordered4By4Bayer();
                break;
            case ordered8By8Bayer:
                dithering = new Ordered8By8Bayer();
                break;
            case floydSteinberg:
                dithering = new FloydSteinberg();
                break;
            case jarvisJudiceNinke:
                dithering = new JarvisJudiceNinke();
                break;
            case sierra:
                dithering = new Sierra();
                break;
            case twoRowSierra:
                dithering = new TwoRowSierra();
                break;
            case sierraLite:
                dithering = new SierraLite();
                break;
            case atkinson:
                dithering = new Atkinson();
                break;
            case stucki:
                dithering = new Stucki();
                break;
            case burkes:
                dithering = new Burkes();
                break;
            case falseFloydSteinberg:
                dithering = new FalseFloydSteinberg();
                break;
            case simpleLeftToRightErrorDiffusion:
                dithering = new SimpleLeftToRightErrorDiffusion();
                break;
            case randomDithering:
                dithering = new RandomDithering();
                break;
            case simpleThreshold:
                dithering = new SimpleThreshold();
                break;
            default:
                throw new MeterException(MeterStateEnum.Dither算法无效);
        }
        return super.check();
    }

    /**
     * @description: 添加点图值信息
     **/
    @Override
    protected DitherVal ditherVal() {
        DitherVal ditherVal = new DitherVal();
        ImageBuffer targetBuffer = super.getTargetBuffer();
        ditherVal.setX(targetBuffer.getMinX());
        ditherVal.setY(targetBuffer.getMinY());
        ditherVal.setWidth(targetBuffer.getWidth());
        ditherVal.setHeight(targetBuffer.getHeight());
        int[] data = new int[targetBuffer.getHeight() * targetBuffer.getWidth()];
        ditherVal.setData(data);

        int index = 0;
        for (int j = 0; j < targetBuffer.getRgbs().length; j++) {
            for (int i = 0; i < targetBuffer.getRgbs()[j].length; i++) {
                RGB rgb = targetBuffer.getRgbs()[j][i];
                data[index] = Objects.isNull(rgb) || rgb.getPixel() == -1 ? 0 : 1;
                index++;
            }
        }
        return ditherVal;
    }

    /**
     * @description: 实际构建方式
     **/
    @Override
    protected ImageBuffer construct() {
        return dithering.dither(super.getSourceBuffer(), super.getImageProcess(), this.ditherThreshold);
    }
}
